/*
 * khotkeys.cpp
 *
 * Copyright (C) 1999 Lubos Lunak <l.lunak@email.cz>
 */

#include "kcmhotkeys.h"
#include <kwm.h>
#include <ksimpleconfig.h>
#include <qmessagebox.h>
#include <kfiledialog.h>
#include <kkeydialog.h>
#include <ctype.h>
#include <kfm.h>
#include <kprocess.h>
#include <qtimer.h>
#include "macro_dialog.h"

static KCMHotKeysApp* kapp_s;

KCMHotKeysApp::KCMHotKeysApp(int &argc, char **argv, const QString& name )
    : KControlApplication( argc, argv, name ), hotkeys_data( 53 ),
      processing_windows( false )
    {
    kapp_s = this;
    wins = NULL;
    hotkeys_data.setAutoDelete( true );
    if (runGUI())
        {
	if( !pages || pages->contains("userhotkeys"))
        addPage( widget = new KHKWidget( dialog ), i18n("&User hotkeys"),
	    "index.html");
        if ( widget )
	    {
    	    dialog->show();
	    kwm_module = new QWidget;
	    CHECK_PTR( kwm_module );
	    KWM::setKWMModule( kwm_module->winId());
	    }
	else
    	    {
	    fprintf(stderr, i18n("usage: kcmhotkeys [-init | userhotkeys]\n"));
	    justInit = TRUE;
	    }
	}
    }
    
KCMHotKeysApp::~KCMHotKeysApp()
    {
    delete kwm_module;
    }
    

void KCMHotKeysApp::init()
    {
    readSettings();
    writeSettings();
    }


void KCMHotKeysApp::apply()
    {
    writeSettings();
    KWM::sendKWMCommand( "khotkeys:reload" );
    }


void KCMHotKeysApp::defaultValues()
    {
    readSettings();
    }


int main(int argc, char **argv)
    {
    KCMHotKeysApp app(argc, argv, "kcmhotkeys");
    app.setTitle( i18n("User hotkeys settings"));
  
    if (app.runGUI())
	{
	app.prepare();
        return app.exec();
	}
    else
	{
	app.init();
	return 0;
        }
    }

    
KHKWidget::~KHKWidget()
    {
    kapp_s->one_act_timeout = get_exec_timeout();
    kapp_s->name_col_width = get_name_col_width();
    kapp_s->hotkey_col_width = get_hotkey_col_width();
    }
    
void KCMHotKeysApp::readSettings()
    {
    if( widget )
	widget->flush();
    KConfig conf( kde_configdir() + "/kcmhotkeysrc", localconfigdir() + "/kcmhotkeysrc" );
    conf.setGroup( "Main" );
    one_act_timeout = conf.readLongNumEntry( "Timeout", 60 * 1000 ); // 1 minute by default
    reset_keycode = stringToKey( conf.readEntry( "Reset hotkey", "" ));
    name_col_width = conf.readNumEntry( "NameColWidth", NAME_COL_WIDTH );
    hotkey_col_width = conf.readNumEntry( "HotkeyColWidth", HOTKEY_COL_WIDTH );
    hotkeys_data.clear();
    int num_sections = conf.readNumEntry( "Num_Sections", 0 );
    KConfig conf_old; // global config file
    KEntryIterator* keyiter = conf_old.entryIterator( "KHotKeys" );
    if( num_sections > 0 )
	{
	for( int cnt = 1;
	     cnt <= num_sections;
    	     ++cnt )
	    {
	    char tmp[ 100 ];
	    sprintf( tmp, "Section%i", cnt );
	    conf.setGroup( tmp );
	    if( strcmp( conf.group(), tmp ) != 0 ) // no such Section
	        continue;
	    QString name = conf.readEntry( "Name", "" );
	    if( name == "" || hotkeys_data.find( name ))
	        continue;
    	    QString type = conf.readEntry( "Type", "" );
	    run_type_t run_type;
	    if( type == "run" )
		run_type = run;
	    else if( type == "macro" )
		run_type = macro;
	    else if( type == "url" )
		run_type = url;
	    else
		continue;
	    KHotData* data = new KHotData;
	    data->run_command = conf.readEntry( "RunCommand", "" );
	    data->run_type = run_type;
	    data->use_command = str_to_w_par_t( conf.readEntry( "UseCommand" ));
	    if( data->use_command == not_sig && conf.readBoolEntry( "UseCommand" ))
		data->use_command = if_ex; // CHECKME backward compatibility
	    data->command = conf.readEntry( "Command", "" );
	    data->use_title = str_to_w_par_t( conf.readEntry( "UseTitle" ));
	    if( data->use_title == not_sig && conf.readBoolEntry( "UseTitle" ))
		data->use_title = if_ex; // CHECKME backward compatibility
	    data->wtitle = conf.readEntry( "Title", "" );
	    data->use_class = str_to_w_par_t( conf.readEntry( "UseClass" ));
	    if( data->use_class == not_sig && conf.readBoolEntry( "UseClass" ))
		data->use_class = if_ex; // CHECKME backward compatibility
	    data->wclass = conf.readEntry( "Class", "" );
	    bool one_act_win = conf.readBoolEntry( "OnlyOneAppWin" )
		&& ( run_type != macro ); // CHECKME backward compatibility
	    QString w_s = conf.readEntry( "WinSearch", "" );
	    if( w_s == "Last" )
		data->win_search = last;
	    else if( w_s == "Created" )
		data->win_search = created;
	    else if( w_s == "Any" )
		data->win_search = any;
	    else
		data->win_search = one_act_win ? created : any;
	    QString keycode = conf.readEntry( "Hotkey", "" );
	    data->keycode = stringToKey( keycode );
            if( keyiter != NULL && data->keycode == 0 ) // CHECKME backward compatibility
    	        {
	        keyiter->toFirst();
	        while( keyiter->current())
		    {
		    if( name == keyiter->currentKey())
			{
			data->keycode = stringToKey( keyiter->current()->aValue );
		      break;
		        }
		    ++(*keyiter);
		    }
		}
	    hotkeys_data.insert( name, data );
	    }
	}
    if( keyiter != NULL )
	delete keyiter;
    if( widget )
	widget->reload();
    }
    
    
void KHKWidget::reload()
    {
    QDictIterator< dt::KHotData > iter( kapp_s->hotkeys_data );
    iter.toFirst();
    while( iter.current())
	{
	( void )new QListViewItem( actions_list, iter.currentKey(),
	    keyToString( iter.current()->keycode )); // inserts automatically
	++iter;
	}
    if( actions_list->currentItem() == NULL && actions_list->firstChild() != NULL )
	actions_list->setSelected( actions_list->firstChild(), true );
    set_enable_empty( actions_list->currentItem() != NULL ); // CHECKME
    actions_list->setFocus();
    char tmp[ 100 ];
    sprintf( tmp, "%li", kapp_s->one_act_timeout );
    execute_timeout->setText( tmp );
    reset_hotkey_text->setText( keyToString( kapp_s->reset_keycode ));
    actions_list->setColumnWidth( 0, kapp_s->name_col_width );
    actions_list->setColumnWidth( 1, kapp_s->hotkey_col_width );
    }

void KCMHotKeysApp::writeSettings()
    {
    if( widget != NULL )
	{
	one_act_timeout = widget->get_exec_timeout();
        name_col_width = widget->get_name_col_width();
	hotkey_col_width = widget->get_hotkey_col_width();
	widget->prepare_data();
	}
    KConfig conf( kde_configdir() + "/kcmhotkeysrc", localconfigdir() + "/kcmhotkeysrc" );
    conf.setGroup( "Main" );
    conf.writeEntry( "Timeout", one_act_timeout );
    conf.writeEntry( "Num_Sections", hotkeys_data.count() );
    conf.writeEntry( "Reset hotkey", keyToString( reset_keycode ));
    if( name_col_width != NAME_COL_WIDTH )
	conf.writeEntry( "NameColWidth", name_col_width );
    if( hotkey_col_width != HOTKEY_COL_WIDTH )
        conf.writeEntry( "HotkeyColWidth", hotkey_col_width );
    QDictIterator< KHotData > iter( hotkeys_data );
    iter.toFirst();
    int cnt = 1;
    while( iter.current())
	{
	char tmp[ 100 ];
	sprintf( tmp, "Section%i", cnt );
	conf.setGroup( tmp );
	switch ( iter.current()->run_type )
	    {
	    case run :
		conf.writeEntry( "Type", QString("run") );
	      break;
	    case url :
		conf.writeEntry( "Type", QString("url") );
	      break;
	    case macro :
		conf.writeEntry( "Type", QString("macro") );
	      break;
	    default :
		++iter;
		++cnt;
	      continue;
	    }
	conf.writeEntry( "Name", iter.currentKey());
	conf.writeEntry( "RunCommand", iter.current()->run_command );
	conf.writeEntry( "UseCommand", w_par_t_to_str( iter.current()->use_command ));
	conf.writeEntry( "Command", iter.current()->command  );
	conf.writeEntry( "UseTitle", w_par_t_to_str( iter.current()->use_title ));
	conf.writeEntry( "Title", iter.current()->wtitle );
	conf.writeEntry( "UseClass", w_par_t_to_str( iter.current()->use_class ));
	conf.writeEntry( "Class", iter.current()->wclass );
	conf.writeEntry( "Hotkey", keyToString( iter.current()->keycode ));
	if( iter.current()->win_search == last )
	    conf.writeEntry( "WinSearch", "Last" );
	else if( iter.current()->win_search == created )
	    conf.writeEntry( "WinSearch", "Created" );
	else
	    conf.writeEntry( "WinSearch", "Any" );
	++cnt;   // CHECKME && run_type == KHotData::run ??
	++iter;
	}
    if( widget )
	widget->after_data();
    }
    
void KHKWidget::after_data()
    {
    if( actions_list->currentItem() != NULL ) // CHECKME
	load_action_data( actions_list->currentItem()->text( 0 )); // CHECKME
    }

void KHKWidget::got_highlight( QListViewItem* item )
    {
    if( ignore )
	return;
    if( last_index != NULL && !may_change_pos())
	{
	// CHECKME hack
	// ok, this is an ugly hack to force the user to enter unique action name
	// I unfortunately can't find out something better than this :(
	ignore = true;
	QTimer::singleShot( 0, this, SLOT(fix_highlight()));
	return;
	}
    if( last_index != NULL )
	save_action_data();
    load_action_data( QString( item->text( 0 )));    
    set_enable_empty( actions_list->currentItem() != NULL ); // CHECKME
    last_index = actions_list->currentItem();
    }
    
void KHKWidget::fix_highlight()
    {
    actions_list->setSelected( last_index, true );
    ignore = false;
    }
    
void KHKWidget::action_name_changed( const char* name )
    {
    if( actions_list->currentItem() != NULL )
	actions_list->currentItem()->setText( 0, name ); // CHECKME
    }

void KHKWidget::action_usecommand_changed(int status)
    {
    if( status == dt::macro ) // CHECKME macro
	{
	combo_win_search->setEnabled( false );
	button_browse->setText( i18n( "Edit &macro" ));
	combo_action_command->changeItem( i18n("Execute if active window's command is :" ), 1 );
	combo_action_command->changeItem( i18n("Execute if active window's command is (regexp) :" ), 2 );
	combo_action_command->update();
	combo_action_title->changeItem( i18n("Execute if active window's title is :" ), 1 );
	combo_action_title->changeItem( i18n("Execute if active window's title is (regexp) :" ), 2 );
	combo_action_title->update();
	combo_action_class->changeItem( i18n("Execute if active window's class is :" ), 1 );
	combo_action_class->changeItem( i18n("Execute if active window's class is (regexp) :" ), 2 );
	combo_action_class->update();
//	button_autofill->setEnabled( false );
	}
    else
	{
	combo_action_command->changeItem( i18n("Don't execute if window command exists :" ), 1 );
	combo_action_command->changeItem( i18n("Don't execute if window command exists (regexp) :" ), 2 );
	combo_action_command->update();
	combo_action_title->changeItem( i18n("Don't execute if window title exists :" ), 1 );
	combo_action_title->changeItem( i18n("Don't execute if window title exists (regexp) :" ), 2 );
	combo_action_title->update();
	combo_action_class->changeItem( i18n("Don't execute if window class exists :" ), 1 );
	combo_action_class->changeItem( i18n("Don't execute if window class exists (regexp) :" ), 2 );
	combo_action_class->update();
	combo_win_search->setEnabled( combo_action_command->currentItem() != 0 // static_cast<>
	    || combo_action_title->currentItem() != 0
	    || combo_action_class->currentItem() != 0 );
	button_browse->setText( i18n( "Browse..." ));
//	button_autofill->setEnabled( combo_action_command->currentItem());
	}
    }

bool KHKWidget::may_change_pos()
    {
    if( last_index != NULL && ( !qstrcmp( action_name->text(), i18n( "New action" ))
	    || !qstrcmp( action_name->text(), i18n( "Reset status" ))))
	QMessageBox::warning( this, i18n( "KHotKeys" ),
	    QString( i18n( "Action '" )) + action_name->text()
		+ i18n( "' is reserved.\nPlease choose different name." ));
    else if( last_index != NULL && strlen( action_name->text()) == 0 )
	QMessageBox::warning( this, i18n( "KHotKeys" ),
	    i18n( "Empty action name!" ));
    else if( kapp_s->hotkeys_data.find( action_name->text() ))
	QMessageBox::warning( this, i18n( "KHotKeys" ),
	    i18n( "Duplicate action name!" ));
    else if( combo_runcommand->currentItem() == dt::macro // static_cast<>
	&& !macro_check( action_runcommand->text()))
	{
	if( QMessageBox::warning( NULL, i18n( "KHotKeys" ),
	    i18n( "The macro contains some invalid keys.\n"
	    "Do you wish to continue and leave it unchanged ?" ), QMessageBox::Yes | QMessageBox::Default,
	    QMessageBox::No | QMessageBox::Escape ) == QMessageBox::Yes )
	    return true;
	}
    else
	return true;
    return false;
    }
    
bool KHKWidget::macro_check( const QString& macro_P )
    {
    if( macro_P == "" )
	return true;
    int last_index = -1, start = 0;
    while(( last_index = macro_P.find( ':', last_index + 1 )) != -1 ) // find next ';'
	{
	if( stringToKey( macro_P.mid( start, last_index - start )) == 0 )
	    return false;
	start = last_index + 1;
	}
    // and the last one
    if( stringToKey( macro_P.mid( start, macro_P.length())) == 0 )
	return false; // the rest
    return true;
    }
    

void KHKWidget::save_action_data()
    {
    dt::KHotData* ptr = new dt::KHotData;
    // static_cast<>
    ptr->run_type = ( dt::run_type_t ) combo_runcommand->currentItem();
    ptr->run_command = action_runcommand->text();
    // static_cast<>
    ptr->use_command = ( dt::w_par_t ) combo_action_command->currentItem(); // CHECKME
    ptr->command = action_command->text();
    ptr->use_title = ( dt::w_par_t ) combo_action_title->currentItem();
    ptr->wtitle = action_title->text();
    ptr->use_class = ( dt::w_par_t ) combo_action_class->currentItem();
    ptr->wclass = action_class->text();
    ptr->win_search = (( combo_action_command->currentItem() != 0 // static_cast<>
	    || combo_action_title->currentItem() != 0
	    || combo_action_class->currentItem() != 0 ) ?
	    ( dt::search_win_t ) combo_win_search->currentItem() : dt::any );
    ptr->keycode = stringToKey( action_hotkey_text->text());
    kapp_s->hotkeys_data.insert( action_name->text(), ptr );
    }
            
void KHKWidget::load_action_data( const char* action )
    {
    dt::KHotData* ptr = kapp_s->hotkeys_data.find( action );
    if( ptr == NULL )
	return; // CHECKME
    action_name->setText( action );
    combo_runcommand->setCurrentItem( ptr->run_type );
    action_runcommand->setText( ptr->run_command.copy() );
    combo_action_command->setCurrentItem( ptr->use_command );
    action_command->setText( ptr->command.copy() );
    combo_action_title->setCurrentItem( ptr->use_title );
    action_title->setText( ptr->wtitle.copy() );
    combo_action_class->setCurrentItem( ptr->use_class );
    action_class->setText( ptr->wclass.copy() );
    combo_win_search->setCurrentItem( ptr->win_search );
    kapp_s->hotkeys_data.remove( action );
    action_hotkey_text->setText( keyToString( ptr->keycode ));
    action_usecommand_changed( combo_runcommand->currentItem());
    action_command_toggle( combo_action_command->currentItem());
    action_title_toggle( combo_action_title->currentItem());
    action_class_toggle( combo_action_class->currentItem());
    }
            
void KHKWidget::action_command_toggle( int status )
    {
    action_command->setEnabled( status != 0 ); // static_cast<>
//    button_autofill->setEnabled( status != 0 && combo_runcommand->currentItem() != 2 );
    combo_win_search->setEnabled( combo_runcommand->currentItem() != 2
	&& ( status != 0 // static_cast<>
	    || combo_action_title->currentItem() != 0
	    || combo_action_class->currentItem() != 0 ));
    }
    
void KHKWidget::action_title_toggle( int status )
    {
    action_title->setEnabled( status != 0 );
    combo_win_search->setEnabled( combo_runcommand->currentItem() != 2
	&& ( combo_action_command->currentItem() != 0 // static_cast<>
	    || status != 0
	    || combo_action_class->currentItem() != 0 ));
    }
    
void KHKWidget::action_class_toggle( int status )
    {
    action_class->setEnabled( status != 0 );
    combo_win_search->setEnabled( combo_runcommand->currentItem() != 2
	&& ( combo_action_command->currentItem() != 0 // static_cast<>
	    || combo_action_title->currentItem() != 0
	    || status != 0 ));
    }


void KHKWidget::action_hotkey_change()
    {
    change_hotkey( false );
    }
    
void KHKWidget::change_hotkey( bool reset_key_P )
    {
    if( !may_change_pos())
	return;
    if( actions_list->currentItem() == NULL )
	return;
    KGlobalAccel* kga = new KGlobalAccel( true ); // do not grab
    if( reset_key_P )
	kga->insertItem( i18n( "Reset status" ), i18n( "Reset status" ),
	    reset_hotkey_text->text());
    else
        kga->insertItem( action_name->text(), action_name->text(),
	    action_hotkey_text->text());
    QDict< KKeyEntry > keys = kga->keyDict();
    QDictIterator< KKeyEntry > keyiter( keys );
    keyiter.toFirst();
    // hack to make KKeyDialog work as we need
    keyiter.current()->aConfigKeyCode = keyiter.current()->aDefaultKeyCode;
    keyiter.current()->aDefaultKeyCode = 0;
    kga->setKeyDict( keys );
    for(;;)
	{
        KKeyDialog::configureKeys( kga, false );
        QDictIterator< dt::KHotData > iter( kapp_s->hotkeys_data );
	iter.toFirst();
	uint this_keycode = kga->currentKey( action_name->text());
	while( iter.current()) // CHECKME this_keycode is not at the moment in hotkeys_data
	    {
	    if( this_keycode == iter.current()->keycode )
		{
		if( !reset_key_P
		    && combo_runcommand->currentItem() == dt::macro
		    && iter.current()->run_type == dt::macro ) // CHECKME
		    {
		    if( QMessageBox::warning( this, i18n( "KHotKeys" ),
			    QString( i18n( "This hotkey is already assigned to another macro.\n"
			    "Do you wish to use it for both ?" )),
			    QMessageBox::Yes | QMessageBox::Default,
			    QMessageBox::No | QMessageBox::Escape ) == QMessageBox::No )
			break;
		    }
		else
		    {
		    QMessageBox::warning( this, i18n( "KHotKeys" ),
			QString( i18n( "This hotkey is already assigned to command or macro '" ))
			+ iter.currentKey() + i18n( "'.\nOne hotkey can be assigned either to one command or to macro(s).\n"
			"Either change the hotkey or assign no hotkey to this action." ));
		    break;
		    }
		}
	    ++iter;
	    }
	if( iter.current() != NULL )
	    continue;  // again
	
#if 0
// this doesn't work yet, since khotkeys grabs the keys, so it always fails
// some comm needed
	QDict< KKeyEntry > keys = kga->keyDict();
	QDictIterator< KKeyEntry > keyiter( keys );
	keyiter.toFirst();
	while( keyiter.current())
	    {
	    uint keysym = keyToXSym( keyiter.current()->aCurrentKeyCode );
	    uint mod = keyToXMod( keyiter.current()->aCurrentKeyCode );
	    if( !kga->grabKey( keysym, mod )
		&& QMessageBox::warning( this, i18n( "KHotKeys" ),
		QString( i18n( "Couldn't grab hotkey for action " ))
		+ keyiter.currentKey() + ".\n"
		+ i18n( "This hotkey is probably already in use.\n"
		"Do you still wish to use it ?" ), QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape ) == QMessageBox::No )
		break;
	    kga->ungrabKey( keysym, mod );
	    ++keyiter;
	    }
	if( !keyiter.current()) // no error while grabbing or user said ok
#endif
	    break;
	}
    if( reset_key_P )
	{
	kapp_s->reset_keycode = kga->currentKey( i18n( "Reset status" ));
	reset_hotkey_text->setText( keyToString( kapp_s->reset_keycode ));
	}
    else
	{
	QString rt = keyToString( kga->currentKey( action_name->text()));
	action_hotkey_text->setText( rt );
        actions_list->currentItem()->setText( 1, rt ); // CHECKME
	}
// CHECKME testy duplikace
    }

void KHKWidget::action_runcommand_browse()
    {
    if( combo_runcommand->currentItem() == 2 ) // macro ??
	{
	macro_dialog* d = new macro_dialog( action_runcommand->text());
	if( d->exec())
	    action_runcommand->setText( d->get_macro_text());
	delete d;
//	QMessageBox::information( this, i18n( "KHotKeys" ),
//	    i18n( "Sorry, not implemented yet.\nTry the documentation." ));
	// CHECKME record macro
	}
    else
	{
	QString ret = KFileDialog::getOpenFileURL();
	if( !ret.isEmpty() )
	    action_runcommand->setText( ret );
	}
    }
    
void KHKWidget::guess_pressed()
    {
    if( qstrlen( action_runcommand->text()) == 0 && combo_runcommand->currentItem() != dt::macro )
	{
	QMessageBox::information( this, i18n( "KHotKeys" ),
	    i18n( "Nothing to execute!" ));
	return;
	}
    kapp_s->wins = new win_dialog;
    if( combo_runcommand->currentItem() == dt::run ) // CHECKME how about macros ?
        {
        KShellProcess proc;
	proc << action_runcommand->text();
    	proc.start( KShellProcess::DontCare );
	}
    else if( combo_runcommand->currentItem() == dt::url )
        {
        KFM* kf = new KFM; // CHECKME doesn't work ?!?!
        kf->openURL( action_runcommand->text() );
        delete kf;
        }
    if( kapp_s->wins->exec())
        {
        QListViewItem* tmp = kapp_s->wins->get_data();
	if( qstrlen( tmp->text( win_dialog::w_command )) > 0 )
	    {
	    action_command->setText( tmp->text( win_dialog::w_command ));
	    combo_action_command->setCurrentItem( 1 );
	    }
	else
	    combo_action_command->setCurrentItem( 0 );
        action_command_toggle( combo_action_command->currentItem());
	if( qstrlen( tmp->text( win_dialog::w_title )) > 0 )
	    {
	    action_title->setText( tmp->text( win_dialog::w_title ));
	    combo_action_title->setCurrentItem( 1 );
	    }
	else
	    combo_action_title->setCurrentItem( 0 );
        action_title_toggle( combo_action_title->currentItem());
	if( qstrlen( tmp->text( win_dialog::w_class )) > 0 )
	    {
	    action_class->setText( tmp->text( win_dialog::w_class ));
	    combo_action_class->setCurrentItem( 1 );
	    }
	else
	    combo_action_class->setCurrentItem( 0 );
        action_class_toggle( combo_action_class->currentItem());
        }
    delete kapp_s->wins;
    kapp_s->wins = NULL;
    }
    
void KHKWidget::new_action()
    {
    if( !may_change_pos())
	return;
    actions_list->setSelected( new QListViewItem( actions_list,
	i18n( "New action" ), "" ), true ); // inserts automatically
    action_name->setText( i18n( "New action" ));
    combo_runcommand->setCurrentItem( 1 );
    action_runcommand->setText( "" );
    combo_action_command->setCurrentItem( 0 );
    action_command->setText( "" );
    combo_action_title->setCurrentItem( 0 );
    action_title->setText( "" );
    combo_action_class->setCurrentItem( 0 );
    action_class->setText( "" );
    combo_win_search->setCurrentItem( 0 );
    action_hotkey_text->setText( "" ); // CHECKME text "NONE" ??
    set_enable_empty( true );
    action_name->setFocus();
    }

void KHKWidget::copy_action()
    {
    if( !may_change_pos())
	return;
    QString copy_name = QString( i18n("Copy of " )) + action_name->text();
    actions_list->setSelected( new QListViewItem( actions_list,
	copy_name, "" ), true ); // inserts automatically
    action_name->setText( copy_name );
    action_hotkey_text->setText( "" );
    action_name->setFocus();
    }

void KHKWidget::delete_action()
    {
    last_index = NULL; // avoid saving CHECKME this is a hack
    QString nm = actions_list->currentItem()->text( 0 );
    QListViewItem* next = actions_list->currentItem()->itemBelow();
    delete actions_list->currentItem(); // CHECKME
    if( next )
	actions_list->setSelected( next, true );
    set_enable_empty( actions_list->currentItem() != NULL ); // CHECKME
    kapp_s->hotkeys_data.remove( nm );
    if( actions_list->currentItem() == NULL )
	set_enable_empty( false );
    }
    
void KHKWidget::set_enable_empty( bool enable )
    {
    action_name->setEnabled( enable );
    combo_runcommand->setEnabled( enable );
    action_runcommand->setEnabled( enable );
    action_title->setEnabled( enable && combo_action_title->currentItem());
    action_class->setEnabled( enable && combo_action_class->currentItem());
    action_hotkey_text->setEnabled( enable );
    action_command->setEnabled( enable && combo_action_command->currentItem());
    combo_action_command->setEnabled( enable );
    combo_action_title->setEnabled( enable );
    combo_action_class->setEnabled( enable );
    combo_win_search->setEnabled( enable && combo_runcommand->currentItem() != 2
	&& ( combo_action_command->currentItem() != 0 // static_cast<>
	    || combo_action_title->currentItem() != 0
	    || combo_action_class->currentItem() != 0 ));
    button_hotkey->setEnabled( enable );
    button_browse->setEnabled( enable );
//    button_autofill->setEnabled( enable
//	&& combo_action_command->currentItem() != 0 && combo_runcommand->currentItem() != 2 );
    delete_button->setEnabled( enable );
    copy_button->setEnabled( enable );
    }

QValidator::State QHKValidator::validate( QString& str, int & )
    {
    for( int i = str.length() - 1;
	 i >= 0;
	 --i )
	if( !isalnum( str[ i ] ) && str[ i ] != ' ' && str[ i ] != '('
	    && str[ i ] != ')' && str[ i ] != '-' && str[ i ] != '_' )
	    return Invalid;
    return str == "" || str == i18n( "New action" ) || str == i18n( "Reset status" )
      ? Valid : Acceptable; // CHECKME
    }

bool KCMHotKeysApp::x11EventFilter( XEvent* ev )
    {
    // from KWMModuleApplication
    static bool atoms = false;
    static Atom module_win_add;
    static Atom module_win_change;
    static Atom module_win_activate;
    if (KControlApplication::x11EventFilter(ev))
	return true;
    if (ev->xany.window == kwm_module->winId() && ev->type == ClientMessage)
	{
	if( !atoms )
	    {
	    module_win_add = XInternAtom(qt_xdisplay(),
				   "KWM_MODULE_WIN_ADD", False);
	    module_win_change = XInternAtom(qt_xdisplay(),
				   "KWM_MODULE_WIN_CHANGE", False);
	    module_win_activate = XInternAtom(qt_xdisplay(),
				   "KWM_MODULE_WIN_ACTIVATE", False);
	    atoms = true;
	    }
	Atom a = ev->xclient.message_type;
	Window w = (Window) (ev->xclient.data.l[0]);
	if( wins && ( a == module_win_add || a == module_win_change ))
	    wins->add_win( w );
	if( wins && a == module_win_activate )
	    {
	    wins->add_win( w );
	    wins->act_win( w );
	    }
	return true;
	}
    return false;
    }
    

static char* txts[] =
	{ "notsignificant", "ifexists", "regexp" };

const char* KCMHotKeysApp::w_par_t_to_str( w_par_t type )
    {
    return txts[ type ];
    }
    
KCMHotKeysApp::w_par_t KCMHotKeysApp::str_to_w_par_t( const QString& str )
    {
    for( unsigned int i = 0;
	 i < sizeof( txts ) / sizeof( txts[ 0 ] );
	 ++i )
	if( txts[ i ] == str )
	      // static_cast<>
	    return ( w_par_t ) i;
    return not_sig;
    }

void KHKWidget::reset_button_change()
    {
    change_hotkey( true );
    }
