/*****************************************************************************
 * This software is distributed under the terms of the General Public License.
 *
 * Program : kfstab
 * Authors : Andreas Reuter
 * E-Mail  : holtwick@uni-duisburg.de
 *           Andreas.Reuter@munich.netsurf.de
*****************************************************************************/

// QT Header Files

// KDE Header Files
#include <kapp.h>
#include <kbuttonbox.h>
#include <kfiledialog.h> // for filedialogs
#include <kmsgbox.h>

// Application Header Files
#include "devpref.h"

#define LOC i18n

/**
 * Constructor
 * Draws the widget, Each button calls a module that checks if theres a 
 * device that can be mounted
 */
 
DevPreferences::DevPreferences( QWidget *parent, const char *name ) : 
   KDialog ( parent, name, TRUE )
{
   int widgetWidth, widgetHeight;
   int button1Height, button2Height;
   int yOff = 15;
   
   setCaption( name );
   
   // Insert labels
   QLabel *label = new QLabel( this );
   CHECK_PTR( label );
   label->setText( LOC( "Please choose the device you want to mount." ));
   label->setGeometry( 0, yOff, label->sizeHint().width(), 
      label->sizeHint().height() );

   QLabel *subLabel = new QLabel( this );
   CHECK_PTR( subLabel );
   subLabel->setText( LOC( "Remember: fdisk cant detect CDROMs !" ));
   subLabel->setGeometry( 0, yOff + label->sizeHint().height(),
      subLabel->sizeHint().width(),
      subLabel->sizeHint().height() );

   // Insert buttons
   button1Height = 2 * yOff + 2 * label->sizeHint().height();
   QPBFloppy = new QPushButton( LOC( "&Floppy" ), this );
   CHECK_PTR( QPBFloppy );
   connect( QPBFloppy, SIGNAL( clicked()), SLOT( devFloppy()) );
   QPBFloppy->setGeometry( 0,
      button1Height, 
      QPBFloppy->sizeHint().width(), QPBFloppy->sizeHint().height() );
   
   QPBEide = new QPushButton( LOC( "&EIDE" ), this );
   CHECK_PTR( QPBEide );
   connect( QPBEide, SIGNAL( clicked()), SLOT( devEide()) );
   QPBEide->setGeometry( QPBFloppy->sizeHint().width(), 
      button1Height, 
      QPBEide->sizeHint().width(), QPBEide->sizeHint().height() );

   QPBScsi = new QPushButton( LOC( "&SCSI" ), this );
   CHECK_PTR( QPBScsi );
   connect( QPBScsi, SIGNAL( clicked()), SLOT( devScsi()) );
   QPBScsi->setGeometry( QPBFloppy->sizeHint().width() + 
      QPBEide->sizeHint().width(), 
      button1Height, 
      QPBScsi->sizeHint().width(), QPBScsi->sizeHint().height() );

   QPBFile = new QPushButton( LOC( "&File" ), this );
   CHECK_PTR( QPBFile );
   connect( QPBFile, SIGNAL( clicked()), SLOT( devFile()) );
   QPBFile->setGeometry( QPBFloppy->sizeHint().width() + 
      QPBEide->sizeHint().width() + QPBScsi->sizeHint().width(), 
      button1Height, 
      QPBFile->sizeHint().width(), QPBFile->sizeHint().height() );

   QPBDevice = new QPushButton( LOC( "&Device" ), this );
   CHECK_PTR( QPBDevice );
   connect( QPBDevice, SIGNAL( clicked()), SLOT( devDevice()) );
   QPBDevice->setGeometry( QPBFloppy->sizeHint().width() +
      QPBEide->sizeHint().width() + QPBScsi->sizeHint().width() +
      QPBFile->sizeHint().width(),
      button1Height,
      QPBDevice->sizeHint().width(), QPBDevice->sizeHint().height() );

   QPBCancel = new QPushButton( LOC( "&Cancel" ), this );
   CHECK_PTR( QPBCancel );
   QPBCancel->setDefault( TRUE );
   connect( QPBCancel, SIGNAL( clicked()), SLOT( reject()) );
   QPBCancel->setGeometry( QPBFloppy->sizeHint().width() + 
      QPBEide->sizeHint().width() + QPBScsi->sizeHint().width() +
      QPBFile->sizeHint().width() + QPBDevice->sizeHint().width(),
      button1Height, 
      QPBCancel->sizeHint().width(), QPBCancel->sizeHint().height() );

   widgetWidth = QPBFloppy->sizeHint().width() + 
      QPBEide->sizeHint().width() +
      QPBScsi->sizeHint().width() +
      QPBFile->sizeHint().width() +
      QPBDevice->sizeHint().width() +
      QPBCancel->sizeHint().width();
   if (widgetWidth < label->sizeHint().width() )
      widgetWidth = label->sizeHint().width();
   widgetHeight = 4 * yOff + 
      2 * QPBFloppy->sizeHint().height() + 
      3 * label->sizeHint().height();

   QPBAccept = new QPushButton( LOC( "&Accept" ), this );
   CHECK_PTR( QPBAccept );
   button2Height = widgetHeight - ( yOff + QPBAccept->sizeHint().height()) ;
   connect( QPBAccept, SIGNAL( clicked()), SLOT( devSelect()));
   QPBAccept->setGeometry( widgetWidth / 2, button2Height,
      widgetWidth / 2,
      QPBAccept->sizeHint().height());
   QPBAccept->hide();

   QCBDevice = new QComboBox( FALSE, this );
   CHECK_PTR( QCBDevice );
   QCBDevice->setGeometry(0, button2Height, 
      widgetWidth / 2,
      QPBAccept->sizeHint().height());
   QCBDevice->hide();
   
   specDev = new QLabel( this );
   CHECK_PTR( specDev );
   specDev->setText( LOC( "Choose available filesystems" ) );
   specDev->setGeometry( ( widgetWidth - specDev->sizeHint().width() ) / 2, 
      button2Height - specDev->sizeHint().height(),
      specDev->sizeHint().width(),
      specDev->sizeHint().height());
   specDev->hide();
   
   setMinimumSize( widgetWidth, widgetHeight );
   setMaximumSize( widgetWidth, widgetHeight );
}

/**
 * This module handles Floppy devices
 */
 
void DevPreferences::devFloppy()
{
   QCBDevice->clear();
   QCBDevice->insertItem( "/dev/fd0" );
   QCBDevice->insertItem( "/dev/fd1" );
   QCBDevice->show();
   QPBAccept->show();
   specDev->show();
}

/**
 * This module handles EIDE devices
 */
 
void DevPreferences::devEide()
{
   emit devEideScsi( LOC( "Device Preferences - EIDE devices" ), 'h', 'i');
}

/**
 * This module handles SCSI devices
 */
void DevPreferences::devScsi()
{
   emit devEideScsi( LOC( "Device Preferences - SCSI devices" ), 's', 'q');
}

/**
 * This module reads the available partitions from stdout
 * which is send as a QString to this module.
 * The partitions are send to a ComboBox, a text is shown.
 */

void DevPreferences::foundPart( QString *stdOutBuffer )
{

   QTextStream *textStream = new QTextStream( *stdOutBuffer, IO_ReadOnly );

      for ( int i = 1; i < 6; i++ )
         QString string = textStream->readLine();
      while ( !textStream->eof() ) {
         QString string = textStream->readLine();
         // Dont scan Unknown or Extended partitions
         if (( !strstr( string, "Unknown" )) &&
             ( !strstr( string, "Extended" )) &&
             ( !strstr( string, "Partition" )) &&
             ( !strstr( string, "phys=" )) ) {
            char tmpString[ string.size() ];
	    strcpy( tmpString, string );
	    string = strtok( tmpString , "\n\t\r " );
	    // Now "string" contains the filesystem e.g. in /dev/hda1
	    QCBDevice->insertItem( string );
            QCBDevice->show();
	    QPBAccept->show();
	    specDev->show();
         }
      }
}

/**
 * Sends the selected device back if the selection is accepted.
 */
 
void DevPreferences::devSelect()
{
   QString *devStr;
   
   emit accept();  // close dialog
   devStr = ( QString * ) QCBDevice->currentText(); // get selected deviceName
   emit newDevice( ( char * ) devStr ); // send new deviceName back
}

/**
 * This module is used by devEide and devScsi to check which partitions 
 * are found by fdisk.
 */
 
void DevPreferences::devEideScsi( const char *name, char majordev, char endDev )
{
   extern int itisroot;

   // Only root can run /sbin/fdisk
   if ( !itisroot )
      KMsgBox::message( this,
      name,
      LOC( "Only root can run fdisk, Sorry." ), KMsgBox::STOP );
   else {
      QCBDevice->clear();
      QCBDevice->hide();
      QPBAccept->hide();
      specDev->hide();
      // now call fdisk.
      // Check first if /sbin/fdisk is installed on this system
      //
      // Later it maybe makes sense to also turn this system() call
      // into a KProcess call.
      if ( system( "/sbin/fdisk -v 1>/dev/null 2>/dev/null" )) {
         KMsgBox::message( this , LOC( "Error" ),
	          LOC( "I cant run /sbin/fdisk !!!" ), KMsgBox::STOP);
         emit reject();
      }
      else {
         kapp->setOverrideCursor( waitCursor );
	 char devChar = 'a';
         for ( ; devChar < endDev; devChar++ ) {
            QString sysStr;
            // now we build the device for command line parameter
            sysStr.sprintf( "/dev/%cd%c", majordev, devChar );
            if ( fdiskProc[ devChar - 'a' ].isRunning() )
               KMsgBox::message( this, "Error",
                  "fdiskProc has already started.", KMsgBox::STOP );
            // We need clearArguments() for obvious
            fdiskProc[ devChar - 'a' ].clearArguments();
            fdiskProc[ devChar - 'a' ] << "/sbin/fdisk" << "-l" << sysStr;
            connect( &fdiskProc[ devChar - 'a' ], SIGNAL( processExited( KProcess * ) ),
               this, SLOT( slotProcessDead( KProcess * ) ));
            connect( &fdiskProc[ devChar - 'a' ], SIGNAL( receivedStdout( KProcess *, char *, int)),
               this, SLOT(slotCmdStdout(KProcess *, char *, int)));
// NotifyOnExit doesnt work as fdisk childs will stay alive
//             if( !fdiskProc[ devChar - 'a' ].start( KProcess::NotifyOnExit, KProcess::AllOutput )) {
            if( !fdiskProc[ devChar - 'a' ].start( KProcess::Block, KProcess::AllOutput ))
               KMsgBox::message( this, "Error",
                  "Could not start process", KMsgBox::STOP );
         }
         devChar = 'a';
         kapp->restoreOverrideCursor();
      }
   }
}

/**
 * This module handles File devices
 */
 
void DevPreferences::devFile()
{
   if ( KMsgBox::yesNo( this, LOC( "Warning" ),
      LOC( "Are you sure you want to mount this filesystem to a file ?"
         "\nThis may cause to a malfunction of your system !" ),
      KMsgBox::EXCLAMATION | KMsgBox::DB_SECOND, LOC( "&Yes" ), LOC( "&No" ) ) == 1 ){
      QString fileName = KFileDialog::getOpenFileName( "/", "*", this );
      if ( ! fileName.isNull() ) {
         QCBDevice->clear();
         QCBDevice->insertItem( fileName );
         QCBDevice->show();
         QPBAccept->show();
         specDev->show();
      }
   }
}

/**
 * This method handles /dev/<somedev> directly as KFileDialog can do so.
 */
void DevPreferences::devDevice()
{
   QString devName = KFileDialog::getDirectory( "/dev/", this );
   if ( ! devName.isNull() ) {
      QCBDevice->clear();
      QCBDevice->insertItem( devName );
      QCBDevice->show();
      QPBAccept->show();
      specDev->show();
   }
}

void DevPreferences::slotProcessDead( KProcess *child )
{
   // disconnect KProcess
   disconnect( child, SIGNAL( receivedStdout( KProcess *, char *, int )),
      this, SLOT( slotCmdStdout( KProcess *, char *, int )));
   disconnect( child, SIGNAL( processExited( KProcess * ) ),
      this, SLOT( slotProcessDead( KProcess * ) ) );
}

void DevPreferences::slotCmdStdout( KProcess *, char *buffer, int buflen )
{
   QString privBuffer = buffer;
   // As buffer doesnt contain a '\0' at the end we must shorten the string
   privBuffer.resize( buflen );
   // We pass a pointer to foundPart
   foundPart( &privBuffer );
}

DevPreferences::~DevPreferences()
{
  // When the dialog is closed theres maybe a process running
  // So we check for it and kill it
  for( int i = 0; i < 16; i++ )
     if ( fdiskProc[ i ].isRunning() )
        fdiskProc[ i ].kill( 15 );
}

#include "devpref.moc"


