
#ifndef StateCache_h
#define StateCache_h

#include <assert.h>
#include <qlist.h>
#include <qstring.h>

#include "KColorMLE.h"
#include "Colorizer.h"



#define STATE_INVALID -1
#define STATE_NORMAL   0



//////////////////////////////////////////////
/**
	This class represents an element in the state
	cache. it describes its position (in a line) and 
	its length; furthermore the state (font index, see 
	@see SyntaxState) and the following state are saved.
	
	this class is used by @ref StateCacheLineElem.

	@short Internal class for @ref StateCache
	@author Michael Riedel
*/	
class StateCacheElem
{
public:
	/**
		Constructs an element and sets its data members to the
		actual parameter values.
	*/
	StateCacheElem(int state, int afterState, int pos, int len)
	{ 	
		State = state; 
		AfterState = afterState; 
		Pos = pos; 
		Len = len; 
		//warning("StateCache (state=%d, afterState=%d, pos=%d, len=%d) created.", state, afterState, pos, len);
	}
	
	/**
		Retrieves the state of a element to be drawn on the screen.
	*/
	int state() { return State; }
	
	/**
		Retrieves the state following the element; this is *not* the
		state of the following element, since there may be (and surely are)
		portions of text, that are not part of any element. The state of 
		such an element is the afterState() of the preceeding element, or the
		state 0, if there is no preceeding element.
	*/
	int afterState() { return AfterState; }
	
	/**
		Returns the character position of the element. Since elements 
		are saved in a list of instances of @ref StateCacheLineElem, 
		there is no need to	save the row here; merely the column is 
		saved (and returned). 
	*/		
	int pos() { return Pos; }
	
	/**
		Returns the length of the element (in chars).
	*/
	int len() { return Len; }

	
protected:
	int State;
	int AfterState;
	int Pos; 
	int Len;
};
//////////////////////////////////////////////



//////////////////////////////////////////////
/**
	This class describes the elements of colored text, that
	appear on one line; each element is saved in the list 
	UnitStateCache; their type is @ref StateCacheElem. this 
	class is used internally by @ref StateCache.
	
	@short Internal class for @ref StateCache
	@author Michael Riedel
*/
class StateCacheLineElem
{
	friend class StateCache;
public:
	/**
		Constructs a StateCacheLineElem; the data member PrevLineEndState
		is set to the parameter prevLEState.
	*/
	StateCacheLineElem(int prevLEState);
	
	/*
		Starts a reparsing operation for the current line; the text of
		the current line is given as parameter s to this method. Its 
		return value is the state, that appears at the end of the line.
	*/
	int reparse(const QString& s);
	
	QList<StateCacheElem>& getUnitList() { return UnitStateCache; }
	int prevLineEndState() 				 { return PrevLineEndState; }
	
	
protected:
	int PrevLineEndState;
	QList<StateCacheElem> UnitStateCache;
	static Colorizer* colorizer;
};
//////////////////////////////////////////////



//////////////////////////////////////////////
/**
	A cache for saving the states of text elements and their 
	colors for a complete text file.

	@short Caches text coloring states for a complete text file.
	@author Michael Riedel
*/
class StateCache
{
public:
		/**
			constructs a StateCache that uses colorizer to detect and
			mark colored text; edit is the editor widget that contains
			the text.
		*/
	StateCache(Colorizer* colorizer, KSyntaxMultiLineEdit* edit);
	
		/** 
			this method parses the complete text in the edit @ref Edit.
		*/
	void parse();
	
		/**
			reparses from line row until there are no more changes in 
			the parsing results; returns the index of the last parsed line.
			@param row to start parsing from.
		*/
	int reparseFromHere(unsigned int row);
	
		/**
			inserts a new LineCacheElem at row; its PrevLineEndState will
			be the same as for the previous LineCacheElem at row.
			
			@param row, at which a @ref StateCacheLineElem is inserted.
		*/
	void insertLine(unsigned int row, int prevState = STATE_INVALID);
	
		/**
			removes line row from the LineStateCache
			@param row to remove from LineStateCache
			@param prevState this param can be STATE_INVALID, which sets the 
			       PrevLineEndState of 
		*/
	void removeLine(unsigned int row);
	
		/**
			retrieve a pointer to the LineStateElem at row.
			@param row desired line.
		*/
	StateCacheLineElem* getLineElem(int row) { return LineStateCache.at(row); }
	
	
protected:
	QList<StateCacheLineElem> LineStateCache;
	KSyntaxMultiLineEdit* Edit; 
};
//////////////////////////////////////////////

#endif // StateCache_h

