/*
 * pgptreelist.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <iostream.h>

#include <qpopmenu.h>
#include <kfiledialog.h>
#include <kapp.h>
#include <kiconloader.h>

#include "pgptreelist.h"
#include "pgptreelist.moc"

#include "pgpkeyring.h"

// -----------------------------------------------------------------

PgpKeyItem::PgpKeyItem(PgpKey *key) :
  myKey(key)
{
  int i;

  setText(myKey->getUserId()->userId());
  setPix();

  for (i=0; i<myKey->count(); i++)
    appendChild(new PgpUserIdItem(myKey->getUserId(i)));
};

PgpKeyItem::~PgpKeyItem()
{};

void PgpKeyItem::setPix()
{
  if (myKey->isRevoked())
    setPixmap(&kapp->getIconLoader()->loadIcon("revoked_key.gif"));
  else
    if (myKey->isPrivate())
      {
	if (myKey->isDefault())
	  setPixmap(&kapp->getIconLoader()->loadIcon("def_sec_key_blue.gif"));
	else
	  setPixmap(&kapp->getIconLoader()->loadIcon("sec_key_blue.gif"));
      }
    else
      {
	if (myKey->isPaused())
	  setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_paused.gif"));
	else
	  if (myKey->isRevoked())
	    setPixmap(&kapp->getIconLoader()->loadIcon("revoked_key.gif"));
	  else
	    switch (myKey->getTrust())
	      {
	      case 0:  
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_gray.gif"));
		break;
	      case 1:  
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_gray.gif"));
		break;
	      case 2:  
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_red.gif"));
		break;		     
	      case 5:  
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_yellow.gif"));
		break;		     
	      case 6:  
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_green.gif"));
		break;		     		     
	      default: 
		setPixmap(&kapp->getIconLoader()->loadIcon("pub_key_gray.gif"));
	      }
      }
};

void  PgpKeyItem::setRightMenu(KPopupMenu *menu)
{
  while (menu->count() > 3)
    menu->removeItemAt(3);

  if (myKey->isPrivate())
    {
      menu->setTitle("Private Key");
      
      menu->insertItem("Add User-ID", MENU_PRIVKEY_EDIT); 
      menu->insertItem("Certify Key", MENU_PUBKEY_CERT);
      // menu->setItemEnabled(MENU_PRIVKEY_EDIT, FALSE);
      menu->insertItem("Revoke Key", MENU_PRIVKEY_REVOKE); 
      menu->setItemEnabled(MENU_PRIVKEY_REVOKE, FALSE);
      // does not work because pgp-option "+batchmode" doesn't work with pgp -kd
      // but only pgp2.x keys know that
      menu->insertItem("Delete Key", MENU_PRIVKEY_DELETE); 
      menu->insertSeparator();
      menu->insertItem("Extract Key", MENU_KEY_EXTRACT); 
      menu->insertItem("Make Default", MENU_PRIVKEY_DEFAULT);
      if (myKey->isDefault())
	menu->setItemEnabled(MENU_PRIVKEY_DEFAULT, FALSE);
      menu->insertItem("Change Passphrase", MENU_PRIVKEY_PASS); 
      menu->setItemEnabled(MENU_PRIVKEY_PASS, FALSE);
    }
  else
    {
      if (myKey->isRevoked())
	{
	  menu->setTitle("Revoked Key");

	  menu->insertItem("Extract Key", MENU_KEY_EXTRACT); 
	  menu->insertItem("Delete Key", MENU_PUBKEY_DELETE); 
	}
      else
	{ 
	  menu->setTitle("Public Key");
	  
	  if (myKey->isPaused())
	    {
	      menu->insertItem("Enable Key", MENU_PUBKEY_PAUSE);
	    }
	  else
	    {
	      menu->insertItem("Edit Trust", MENU_PUBKEY_EDIT);
	      menu->insertItem("Certify Key", MENU_PUBKEY_CERT);
	      menu->insertItem("Disable Key", MENU_PUBKEY_PAUSE);
	      menu->insertItem("Delete Key", MENU_PUBKEY_DELETE);
	      menu->insertSeparator();
	      menu->insertItem("Extract Key", MENU_KEY_EXTRACT); 
	    }
	} 
    }
};

void  PgpKeyItem::setLeftMenu(KPopupMenu *menu)
{
  QString tmp;

  while (menu->count() > 3)
    menu->removeItemAt(3);

  menu->setTitle("Key Info");

  menu->insertItem("Version:\t" + myKey->format(), -1);
  menu->insertItem("Algorithm:\t" + myKey->algorithm(), -1);
  menu->insertSeparator();
  menu->insertItem("Bitlen:\t" + tmp.sprintf("%d", myKey->bitlen()), -1);
  menu->insertItem("Key ID:\t" + myKey->keyId(), -1);
  menu->insertItem("Created:\t" + myKey->created().toString(), -1);
  menu->insertSeparator();
  menu->insertItem("Fingerprint:\t" + myKey->fingerprint(), -1);
};


void  PgpKeyItem::setTrust(UINT8 trust)
{
  myKey->setTrust(trust);
  PgpKeyring::getPgpKeyring()->updateKeys(*myKey);
  setPix();
};

void  PgpKeyItem::pauseKey()
{
  myKey->setPaused(!(myKey->isPaused()));
  setPix();
};

PgpKey*
PgpKeyItem::getKey()
{
  return myKey;
};


void  
PgpKeyItem::addSignature(PgpSignature *new_sig)
{
  myKey->getUserId()->addSignature(new_sig, TRUE);
  
  // update child item
  ((PgpUserIdItem*)getChild())->addSignature(new_sig);
  
};


PgpUserIdItem*  PgpKeyItem::newUserId(QString uid)
{
  PgpUserId      *userId      = myKey->newUserId(uid);
  PgpUserIdItem  *userIdItem;

  if (userId)
    {
      userIdItem = new PgpUserIdItem(userId);
      insertChild(0, userIdItem);
      setText(userId->userId());
      return userIdItem;
    }
  else
    return 0;
};

void  PgpKeyItem::revokeKey()
{
  debug("FIXME: revokeKey");
};

void  PgpKeyItem::setDefault(bool value)
{
  if (myKey->isDefault() != value)
    {
      myKey->setDefault(value);
      setPix();
    }
};

void  PgpKeyItem::update()
{
  setText(myKey->getUserId()->userId());
  setPix();
};

PgpUserIdItem* PgpKeyItem::getUserId(int nr)
{
  return (PgpUserIdItem*) getChild();
};

void       
PgpKeyItem::removeUserId(PgpUserIdItem *uid)
{
  myKey->removeUserId(uid->getUserId());
  // removeChild(uid);
};

// ------------------------------------------------------------------

PgpUserIdItem::PgpUserIdItem(PgpUserId *uid) :
  myUserId(uid)
{
  int i;

  setText(myUserId->userId());
  setPix();

  for (i=0; i<myUserId->count(); i++)
    appendChild(new PgpSignatureItem(myUserId->signature(i)));
}

PgpUserIdItem::~PgpUserIdItem()
{};

void  
PgpUserIdItem::setRightMenu(KPopupMenu *menu)
{
  while (menu->count() > 3)
    menu->removeItemAt(3);

  menu->setTitle("User-ID"); 
  menu->insertItem("Delete User-ID", MENU_USERID_DELETE);  
};

void  
PgpUserIdItem::setLeftMenu(KPopupMenu *menu)
{};

PgpUserId* PgpUserIdItem::getUserId()
{
  return myUserId;
};

void  
PgpUserIdItem::update()
{
  debug("FIXME: userID update() NOT IMPLEMENTED");
  
  setText(myUserId->userId());
  setPix();
};

void  
PgpUserIdItem::addSignature(PgpSignature *new_sig)
{
  // maybe it is possible to use the myUserID->signature field
  insertChild(0, new PgpSignatureItem(new_sig));
  setPix();
};
  
void
PgpUserIdItem::setPix()
{
  switch (myUserId->getTrust())
    {
    case 0:  
      setPixmap(&kapp->getIconLoader()->loadIcon("head_gray.gif"));
      break;
    case 1:  
      setPixmap(&kapp->getIconLoader()->loadIcon("head_red.gif"));
      break;
    case 2:  
      setPixmap(&kapp->getIconLoader()->loadIcon("head_yellow.gif"));
      break;		     
    case 3:  
      setPixmap(&kapp->getIconLoader()->loadIcon("head_green.gif"));  
      break;		     	     
    default: 
      setPixmap(&kapp->getIconLoader()->loadIcon("head_gray.gif"));
    }
};
 
// ------------------------------------------------------------------

PgpSignatureItem::PgpSignatureItem(PgpSignature *signature) :
  mySignature(signature)
{
  setText(mySignature->userId());
  setPix();
};

PgpSignatureItem::~PgpSignatureItem()
{};

void  
PgpSignatureItem::setRightMenu(KPopupMenu *menu)
{
  while (menu->count() > 3)
    menu->removeItemAt(3);

  menu->setTitle("Certificate");

  menu->insertItem("Delete Certificate", MENU_SIGNATURE_DELETE);
};

void  
PgpSignatureItem::setLeftMenu(KPopupMenu *menu)
{};

PgpSignature* 
PgpSignatureItem::getSignature()
{
  return mySignature;
};


void
PgpSignatureItem::setPix()
{
  switch (mySignature->getTrust())
    {
    case 2:  
      setPixmap(&kapp->getIconLoader()->loadIcon("feather_red.gif"));
      break;
    case 5: 
      setPixmap(&kapp->getIconLoader()->loadIcon("feather_yellow.gif"));
      break;
    case 6: 
      setPixmap(&kapp->getIconLoader()->loadIcon("feather_green.gif"));
      break;		     
    case 7:
      setPixmap(&kapp->getIconLoader()->loadIcon("feather_blue.gif"));
      break;		     	     
    default:
      setPixmap(&kapp->getIconLoader()->loadIcon("feather_white.gif"));
    }
};

// ------------------------------------------------------------------

// Constructor
PgpTreeList::PgpTreeList(QWidget *parent = 0, const char *name = 0, WFlags f = 0)
    : KTreeList(parent, name, f)
{};

PgpTreeList::~PgpTreeList()
{};

void 
PgpTreeList::mousePressEvent(QMouseEvent *e)
{
  QPoint          mouseCoord = e->pos();
  int             itemClicked = findRow(mouseCoord.y());
  int  cellX, cellY;
  colXPos(0, &cellX);
  rowYPos(itemClicked, &cellY);
  QPoint          cellCoord(mouseCoord.x() - cellX, mouseCoord.y() - cellY);
  
  KTreeList::mousePressEvent(e);
  
  if (itemClicked != -1)
    {  
      if ((itemAt(itemClicked)->boundingRect(fontMetrics()).contains(cellCoord)))
	{
	  if (e->button() == RightButton)
	    emit rightMenuSelected(mapToGlobal(mouseCoord), itemAt(itemClicked)->getBranch());
	  else
	    if (e->button() == LeftButton)
	      emit leftMenuSelected(mapToGlobal(mouseCoord), itemAt(itemClicked)->getBranch());
	}
    }  
}

void 
PgpTreeList::mouseReleaseEvent(QMouseEvent *e)
{
  QPoint          mouseCoord = e->pos();
  int             itemClicked = findRow(mouseCoord.y());
  PgpKeyItem *item = (PgpKeyItem *) itemAt(itemClicked);

  if (item)
    ;//item->hideMenus();
  
  KTreeList::mouseReleaseEvent(e);
}

