 /***************************************************************************
                  	kapm.cpp - KAPM Main File
                  	-------------------------
    begin                : ??/??/99
    copyright            : (C) 1999 by Ryan Cumming
    email                : bodnar42@bodnar42.dhs.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qmovie.h>
#include "kapm.h"
#include "klocale.h"
#include "kapp.h"

Kapm::Kapm() {
	// Used to dock the window
	KWM *docker;

	// Make sure that the APM BIOS and kernel driver meet requirements
	checkApm();

	// The config struct is used from here on, so we should fill it in
  configFile = KApplication::getKApplication()->getConfig();
	loadConfig();

	// Self explainitory
	supressLowNotification = false;
	supressFullNotification = false;
	timerEvent(0);

	// Build menu
	menu = new QPopupMenu(0, "rclick");
	menu->insertItem(i18n("S&tandby"), this, SLOT(slotStandby()));
	menu->insertItem(i18n("&Suspend"), this, SLOT(slotSuspend()));
	menu->insertItem(i18n("&Power Off"), this, SLOT(slotPowerOff()));
	menu->insertSeparator();
	menu->insertItem(i18n("APM &Status..."), this, SLOT(slotAPMStatus()));
	menu->insertSeparator();
	menu->insertItem(i18n("&Options..."), this, SLOT(slotOptions()));
	menu->insertItem(i18n("&Help..."), this, SLOT(slotHelp()));
	menu->insertItem(i18n("&About KAPM..."), this, SLOT(slotAbout()));	
	menu->insertSeparator();
	menu->insertItem(i18n("&Quit"), this, SLOT(slotQuit()));	

	// Setup timers
	startTimer(config.updateInterval);

	// Dock
	docker = new KWM();
	docker->setDockWindow(this->winId());

	setAutoResize(true);
	setMargin(0);
	setAlignment(AlignLeft | AlignTop);

	// KAPM is a slop sucker
	nice(10);

	// Clean up
	delete docker;
}

Kapm::~Kapm(){
	saveConfig();
}

void Kapm::slotSuspend() {
	changeState( APM_IOC_SUSPEND );
}

void Kapm::slotStandby() {
	changeState( APM_IOC_STANDBY );
}

void Kapm::slotAPMStatus() {
	QString status;
	QString *status2;

	apmRead();

  status.sprintf(i18n("APM BIOS %d.%d (kernel driver %s)\n"),
	               i.apm_version_major, i.apm_version_minor, i.driver_version );

	if (i.apm_flags & 0x10) {
      status.append( i18n("Power Management is currently disabled\n") );
	}
	else {
  		if (i.apm_flags & 0x20) {
      	status.append( i18n("Power Management is currently disengaged\n") );
			
			}
			else {
				status.append( i18n("Power Management is currently enabled\n") );
			};
		};
  status2 = statusString();
	QMessageBox::information(this, i18n("APM Status"), status + *status2);
	delete status2;
}

void Kapm::slotAbout() {
	QMessageBox::about(this, i18n("About KAPM"), 
i18n("KAPM 0.2.4 - A Docking APM Utility For KDE\n
1999 Ryan Cumming (bodnar42@bodnar42.dhs.org)
Internationalization by Masanori Hanawa (hanawa@kde.gr.jp)
Original icons by J.D. Caudle"));
}

void Kapm::mousePressEvent(QMouseEvent *e) {
	switch (e->button()) {
 		case RightButton: {
    	int x = e->x() + this->x();
    	int y = e->y() + this->y();

    	menu->popup(QPoint(x, y));
    	menu->exec();
			break;
		}
		case MidButton: {
			slotAPMStatus();
      break;
		}
		case LeftButton: {
			timerEvent((QTimerEvent *)1);
			break;
		}
	};
}

void Kapm::changeState( int state ) {
   int    fd;

   if ((fd = apmOpen()) < 0) {
      errorMsg(i18n("Cannot open APM device"));
   } else {
			if (!config.apmdRunning) {
      	sync();
      	sleep( 2 );
			}
      ioctl( fd, state );
      ::close( fd );
   }
}

void Kapm::slotPowerOff() {
	execlp("poweroff", 0);
	QMessageBox::warning(this, i18n("KAPM"), i18n("Power off failed. Please upgrade your SysVinit package."));
}

void Kapm::checkApm() {
   switch ( apmCheck() ) {
   	case 1: errorMsg( i18n("No APM support in kernel") );  delete this;
   	case 2: errorMsg( i18n("Old APM support in kernel") ); delete this;
	}

	 if (!(i.apm_flags & APM_32_BIT_SUPPORT)) {
      errorMsg( i18n("32-bit APM interface not supported") );
      delete this;
   }
}

void Kapm::errorMsg(const char *message) {
	QMessageBox::critical(this, i18n("KAPM"), i18n(message));
}

QString* Kapm::statusString() {
	QString *status;
	char percent[8];
	char time[12];
   switch (i.ac_line_status) {
   	case 0: status = new QString(i18n("AC off-line"));     			break;
   	case 1: status = new QString(i18n("AC on-line"));      			break;
   	case 2: status = new QString(i18n("On backup power")); 			break;
		default: status = new QString(i18n("Unknown power source"));break;
   }
   if (i.battery_flags != 0xff
       && i.battery_flags & 0x80) {
      status->append( i18n(", no system battery") );
   } else {
      switch (i.battery_status) {
      	case 0: status->append( i18n(", battery status high") ); break;
      	case 1: status->append( i18n(", battery status low") ); break;
      	case 2: status->append( i18n(", battery status critical") ); break;
      	case 3: status->append( i18n(", battery charging") ); break;
      }
   }
	 if (i.battery_percentage >= 0) {
	 			sprintf((char *)&percent, i18n(": %d%%"), i.battery_percentage );
				status->append(percent);
	 }
   if (i.battery_time >= 0) {
	 	    sprintf(time, i18n(" (%d %s)"), i.battery_time, i.using_minutes ? "min" : "sec" );
				status->append(time);
	 }
	return status;
}

void Kapm::slotQuit() {
	delete this;
}

void Kapm::timerEvent ( QTimerEvent *e ) {
	QString *status;

	apmRead();

	status = statusString();
	QToolTip::remove( this );
	QToolTip::add( this, status->data() );
	delete status;

	if (config.fullNotification) {
		if (supressFullNotification)
			supressFullNotification = (i.battery_percentage > (int)config.fullThresholdPercent);
		else
			if (i.battery_percentage > (int)config.fullThresholdPercent) {
				supressFullNotification = true;
				QMessageBox::information(this, i18n("KAPM"), i18n("You
r battery is full."));				
			}				
	}

	if (config.lowNotification && (i.battery_percentage > 0) ) {
		if (supressLowNotification)
			supressLowNotification = (i.battery_percentage < (int)config.lowThresholdPercent);
		else
			if (i.battery_percentage < (int)config.lowThresholdPercent) {
				supressLowNotification = true;
				QMessageBox::warning(this, i18n("KAPM"), i18n("Your battery is low!"));				
			}				
	}

	drawIcon(e == NULL);
}	

void Kapm::drawIcon(bool forceUpdate) {
	static char lastPercent;
	static char lastStatus;

	// If the icon data cache is outdated, refresh it
	if (config.icons[i.battery_status + 1].valid == false) {
		scanIcon(i.battery_status + 1);
		forceUpdate = true;
	}

	// Stops animated GIFs from restarting, and saves a few cycles too
	if ( ( ( (abs(i.battery_percentage - lastPercent) + (lastPercent % config.icons[i.battery_status + 1].percentSize ) *  config.icons[i.battery_status + 1].drawBar) <
		 (config.icons[i.battery_status + 1].percentSize ) )
		 && (lastStatus == i.battery_status) && (!forceUpdate) ) )
		return;

	lastPercent = i.battery_percentage;
	lastStatus = i.battery_status;

	if ( config.icons[i.battery_status + 1].isMovie )
		setMovie( QMovie( config.icons[i.battery_status + 1].fullname.data() ) );
	else {
		if (config.icons[i.battery_status + 1].drawBar) {
			QPixmap pixmap = QPixmap( config.icons[i.battery_status + 1].fullname );
			QImage image;

			image = pixmap;
			fillBar(image);
			pixmap = image;

			setPixmap(pixmap);
		}
		else
			setPixmap(QPixmap( config.icons[i.battery_status + 1].fullname ));
	}
}

void Kapm::slotOptions() {
	optionsDialog *dialog = new optionsDialog;
	// Allow the dialog to access our internal config struct.
	dialog->config = &config;
	dialog->show();

	// Make the changes take effect
	killTimers();
	startTimer(config.updateInterval);
	timerEvent((QTimerEvent *)1);

	delete dialog;
}

void Kapm::loadConfig() {
	configFile->setGroup("General");
	config.apmdRunning = configFile->readBoolEntry("apmdRunning", false);
	config.updateInterval = configFile->readUnsignedNumEntry("updateInterval", 60000);

	configFile->setGroup("Notification");
	config.lowNotification = configFile->readBoolEntry("lowNotification", false);
	config.lowThresholdPercent = configFile->readUnsignedNumEntry("lowThresholdPercent", 10);	
	config.fullNotification = configFile->readBoolEntry("fullNotification", false);
	config.fullThresholdPercent = configFile->readUnsignedNumEntry("fullThresholdPercent", 95);	

	configFile->setGroup("Icon");	

	config.icons[0].filename = configFile->readEntry("unknownIcon", "kapm_unknown.xpm");
	config.icons[1].filename = configFile->readEntry("critIcon", "kapm_critical.xpm");
	config.icons[2].filename = configFile->readEntry("lowIcon", "kapm_low.xpm");
	config.icons[3].filename = configFile->readEntry("highIcon", "kapm_high.xpm");
	config.icons[4].filename = configFile->readEntry("chargeIcon", "kapm_charging.gif");

	for(char x = 0; x < 6;x++)
		config.icons[x].valid = false;
}

void Kapm::saveConfig() {
	configFile->setGroup("General");
	configFile->writeEntry("apmdRunning", config.apmdRunning);
	configFile->writeEntry("updateInterval", config.updateInterval);

	configFile->setGroup("Notification");
	configFile->writeEntry("lowNotification", config.lowNotification);
	configFile->writeEntry("lowThresholdPercent", config.lowThresholdPercent);
	configFile->writeEntry("fullNotification", config.fullNotification);
	configFile->writeEntry("fullThresholdPercent", config.fullThresholdPercent);

	configFile->setGroup("Icon");
	configFile->writeEntry("unknownIcon", config.icons[0].filename);
	configFile->writeEntry("critIcon", config.icons[1].filename);
	configFile->writeEntry("lowIcon", config.icons[2].filename);
	configFile->writeEntry("highIcon", config.icons[3].filename);
	configFile->writeEntry("chargeIcon", config.icons[4].filename);
}

void Kapm::slotHelp() {
	KApplication::getKApplication()->invokeHTMLHelp("kapm/index.html", "");
}

bool Kapm::isGIF8x(const char* file_name) {

	/* The first test checks if we can open the file */
	QFile gif8x(file_name);
	if (gif8x.open(IO_ReadOnly) == false)
		return false;

	/**
	 * The GIF89 format specifies that the first five bytes of
	 * the file shall have the characters 'G' 'I' 'F' '8' '9'.
	 * The GIF87a format specifies that the first six bytes
	 * shall read 'G' 'I' 'F' '8' '7' 'a'.  Knowing that, we
	 * shall read in the first six bytes and test away.
	 */
	char header[6];
	int bytes_read = gif8x.readBlock(header, 6);

	/* Close the file just to be nice */
	gif8x.close();

	/* If we read less than 6 bytes, then its definitely not GIF8x */
	if (bytes_read < 6)
		return false;

	/* Now test for the magical GIF8(9|7a) */
	if (header[0] == 'G' &&
		 header[1] == 'I' &&
		 header[2] == 'F' &&
		 header[3] == '8' &&
		 header[4] == '9' ||
		(header[4] == '7' && header[5] == 'a'))
	{
		/* Success! */
		return true;
	}

	/* Apparently not GIF8(9|7a) */
	return false;
}

void Kapm::fillBar(QImage image) {
	unsigned int w = image.width();
	QRgb rgb;
	unsigned int x, y;
  QRgb fillColor = 0xff000000|qRgb( int( (100 - i.battery_percentage) * 2.55 ), int( i.battery_percentage * 2.55), 0);

	for (y = config.icons[i.battery_status + 1].bottom; y >= config.icons[i.battery_status + 1].top; y--) {
		if ( (( ( (config.icons[i.battery_status + 1].bottom - config.icons[i.battery_status + 1].top) - (y - config.icons[i.battery_status + 1].top) ) * 100) / (config.icons[i.battery_status + 1].bottom - config.icons[i.battery_status + 1].top + 1)) >= (unsigned int)i.battery_percentage )
			fillColor =  fillColor - 0xff000000;
	
		for (x = 0; x < w; x++) {
			rgb = image.pixel(x, y);
			if (rgb == 0xffff00ff)
				image.setPixel(x, y, fillColor);
		}
	}
}

void Kapm::scanIcon(char index) {
	// Discovers and caches the specs for the icon referenced by "index"

 	// Get a list of all the pixmap paths.  This is needed since the
	// icon loader only returns the name of the pixmap -- NOT it's path!
	QStrList *dir_list = kapp->getIconLoader()->getDirList();
	QFileInfo file;

	for (unsigned int z = 0; z < dir_list->count(); z++) {
		QString here = dir_list->at(z);

		// check if this file exists.  If so, we have our path
		file.setFile(here + '/' + config.icons[index].filename);

		if (file.exists())
	    break;
	}

  config.icons[index].fullname = file.absFilePath();
	config.icons[index].isMovie = isGIF8x(config.icons[index].fullname);

	if (config.icons[index].isMovie) {
			config.icons[index].drawBar = false;
			config.icons[index].percentSize = 255;
			config.icons[index].valid = true;
			return;
	}

	QPixmap pixmap = QPixmap( config.icons[index].fullname );
	QImage image;

	image = pixmap;

	unsigned int w = image.width();
	unsigned int h = image.height();
	QRgb rgb;
	unsigned int x, y;

	config.icons[index].top = 65535;
	config.icons[index].bottom = 65535;

	for (y = 0; y < h; y++) {
		if (config.icons[index].top != 65535) break;
		for (x = 0; x < w; x++) {
			rgb = image.pixel(x, y);
			if (rgb == 0xffff00ff) {
      		config.icons[index].top = y;
					break;
			}
		}
	}

	if (config.icons[index].top == 65535) {
	  config.icons[index].drawBar = false;
		config.icons[index].percentSize = 255;
		config.icons[index].valid = true;
		return;
	}

	for (y = h - 1; y >= config.icons[index].top; y--) {
		if (config.icons[index].bottom != 65535) break;
		for (x = 0; x < w; x++) {
			rgb = image.pixel(x, y);
				if (rgb == 0xffff00ff) {
      		config.icons[index].bottom = y;
					break;
			}
		}
	}

	config.icons[index].drawBar = true;
	config.icons[index].percentSize = ( 100 / (config.icons[index].bottom - config.icons[index].top + 1) );
	config.icons[index].valid = true;
}






































































