// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_KROTDATA_INLINE_H
#define KROT_KROTDATA_INLINE_H


#include "calculationParameter.h"
#include "simulation.h"
#include "spectrum.h"
#include "transition.h"
#include "iar/iar.h"

#include <qobject.h>

#include <kmsgbox.h> 

#include <map>



inline DataRange::DataRange( int64_t strt, int64_t stp, int64_t mn, int64_t mx )
    : max( mx ), min( mn ), start( strt ), stop( stp )
{
}



inline bool DataRange::operator<( const DataRange& two ) const
{
    return ( start != two.start ) ? ( start < two.start ) : ( stop < two.stop );
}



inline bool DataRange::operator==( const DataRange& range ) const
{
    return start == range.start && stop == range.stop && max == range.max && min == range.min;
}



inline KRotData::KRotData( QObject *parent, const char *name )
    : QObject( parent, name ),
      avg( 1 ), config( kapp->getConfig() )
{
    pars.readConfig( config );
}



inline KRotData::~KRotData()
{
    pars.writeConfig( config );
}



inline void KRotData::add( const Simulation& sim )
{
    KROT_LAUNCH( "Launching KRotData::add" );
    bool first = empty();
    sims.insert( make_pair( sim.shortName(), sim ) );
    checkLimits();
    if( first )
	emit position( ( sim.stop() - sim.start() ) / 2 );
    return;
}



inline void KRotData::add( const Spectrum& s )
{
    KROT_LAUNCH( "Launching KRotData::add" );
    bool first = empty();
    spec.insert( make_pair( s.shortName(), s ) );
    checkLimits();
    if( first )
	emit position( ( s.stop() - s.start() ) / 2 );
    return;
}



inline const mapAssignment& KRotData::assignments() const
{
    return ass;
}



inline unsigned int KRotData::averaging() const
{
    return avg;
}



inline bool KRotData::empty( Kind val ) const
{
    bool ret = true;
    switch( val ) {
    case ASSIGNMENTS:
	ret = ass.empty();
	break;
    case DATASETS:
	ret = ( spec.empty() && sims.empty() );
	break;
    case SIMULATIONS:
	ret = sims.empty();
	break;
    case SPECTRA:
	ret = spec.empty();
	break;
    }
    return ret;
}



inline void KRotData::erase( const QNum& qnum )
{
    ass.erase( qnum );
    return;
}



inline void KRotData::erase( const QString& name )
{
    sims.erase( name );
    spec.erase( name );
    checkLimits();
    return;
}



inline void KRotData::iar( CalculationParameter& par )
{
    KROT_LAUNCH( "IAR not implemented yet" );
    return;
}



inline const CalculationParameter& KRotData::parameter() const
{
    return pars;
}



inline const DataRanges& KRotData::dataRanges() const
{
    return ranges;
}



inline const mapSimulation& KRotData::simulations() const
{
    return sims;
}



inline mapSimulation& KRotData::simulations()
{
    return sims;
}



inline const mapSpectrum& KRotData::spectra() const
{
    return spec;
}



inline mapSpectrum& KRotData::spectra()
{
    return spec;
}



inline void KRotData::setAveraging( const unsigned int avp )
{
    avg = ( ( 0 == avp ) ? 1 : avp );
    return;
}



inline int64_t KRotData::start() const
{
    return ranges.begin()->start;
}



inline int64_t KRotData::stop() const
{
    return ranges.end()->stop;
}



#endif
