c     -*- mode: FORTRAN -*-
c
c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"


      subroutine mrqmin( y, sig, ndata, a, ia, covar, alpha, np, chisq, funcs, mp, alamda )

c     Levenberg-Marquardt method, attempting to reduce the value CHISQ (chi-square)
c     of a fit between a set of data points I(1:NDATA), Y(1:NDATA) with common
c     standard deviation SIG and a nonlinear function dependent on NP coefficients
c     A(1:NP). The input array IA(1:NP) indicates by nonzero entries those
c     components of A that should be fitted for, and by zero entries those
c     components that should be held fixed at their input values. The program
c     returns current best-fit values for the parameters A(1:NP) and CHISQ.
c     The arrays COVAR(1:NP,1:NP), ALPHA(1:NP,1:NP) with physical
c     dimension NP (the total number of parameters) are used as working
c     space during most iterations. [Supply a subroutine FUNCS( X, A, YFIT,
c     DYDA, NP ) that evaluates the fitting function YFIT, and its
c     derivatives DYDA with respect to the fitting parameters A at X.]
c     On the first call provide an initial guess for the parameters A, and
c     set ALAMDA < 0 for initialization (which then sets ALAMDA = 0.001). If
c     a step succeeds CHISQ becomes smaller and ALAMDA decreases by a factor
c     of 10. If a step fails ALAMDA grows by a factor of 10. You must call
c     this routine repeatedly until convergence is achieved. Then, make one
c     final call with ALAMDA = 0, so that COVAR(1:NP,1:NP) returns the
c     covariance matrix, and ALPHA the curvature matrix. Parameters held
c     fixed will return zero covariances.

      implicit none

      integer        mmax
      parameter      ( mmax = 2 * ARNIROT_NPAR + 2 )
      integer        ndata, mp, np
      integer        ia( np )
      integer        j, k, l, mfit

      real*8         alamda
      real*8         chisq
      real*8         funcs( mp, np )
      real*8         a( np )
      real*8         alpha( np, np )
      real*8         covar( np, np )
      real*8         sig
      real*8         y( ndata )
      real*8         ochisq, atry( mmax ), beta( mmax ), da( mmax )

cU    USES covsrt, gaussj, mrqcof

      SAVE           ochisq, atry, beta, da, mfit

      ARNIROT_LAUNCH ( "Launching mrqfit." )

c     initialization
      if ( alamda .lt. 0. ) then
         mfit = 0
         do j = 1, np
            if ( ia( j ) .ne. 0 ) mfit = mfit + 1
         end do
#ifdef DEBUG_VERBOSE
         write(*,*) 'mfit = ', mfit
#endif
         alamda = 1.d-3
         call mrqcof( y, sig, ndata, a, ia, alpha, beta, np, chisq, funcs, mp )
         ochisq = chisq
         do j = 1, np
            atry( j ) = a( j )
         end do
      end if

c     alter linearized fitting matrix, by augmenting diagonal elements
      do j = 1, mfit
         do k = 1, mfit
            covar( j, k ) = alpha( j, k )
         end do
         covar( j, j ) = alpha( j, j ) * ( 1.d0 + alamda )
         da( j ) = beta( j )
      end do

c     matrix solution
      call gaussj( covar, da, mfit, np )

c     once converged, evaluate covariance matrix
c     if ( alamda .eq. 0. ) then
c        call covsrt( covar, ia, np, mfit )
c        return
c     end if
c     do NOT expand covariance matrix - this isn't done in svdfit as well
      if ( alamda .eq. 0. ) return

      j = 0
c     did the trial succeed?
      do l = 1, np
         if ( ia( l ) .ne. 0 ) then
            j = j + 1
            atry( l ) = a( l ) + da( j )
         end if
      end do
      call mrqcof( y, sig, ndata, atry, ia, covar, da, np, chisq, funcs, mp )

      if ( chisq .lt. ochisq ) then
c        success, accept the new solution
         alamda = 1.d-1 * alamda
         ochisq = chisq
         do j = 1, mfit
            do k = 1, mfit
               alpha( j, k ) = covar( j, k )
            end do
            beta( j ) = da( j )
         end do
         do l = 1, np
            a( l ) = atry( l )
         end do
      else
c        failure, increase alamda and return
         alamda = 1.d1 * alamda
         chisq = ochisq
      end if

      return
      end


c------------------------------------------------------------------------------
      subroutine mrqcof( y, sig, ndata, a, ia, alpha, beta, np, chisq, funcs, mp )

c     Used by MRQMIN to evaluate the linearized fitting matrix ALPHA, and
c     vector BETA as in (15.5.8), and calculate CHISQ.

      implicit none

      integer        mmax
      parameter      ( mmax = 2 * ARNIROT_NPAR + 2 )
      integer        ndata, mp, np
      integer        ia( np )
      integer        mfit, i, j, k, l, m

      real*8         a( np )
      real*8         alpha( np, np )
      real*8         beta( np )
      real*8         chisq
      real*8         funcs( mp, np )
      real*8         sig
      real*8         y( ndata )
      real*8         dy, sig2i, wt, ymod, dyda( mmax )

      ARNIROT_LAUNCH ( "Launching mrqcof." )

      mfit = 0
      do j = 1, np
         if ( ia( j ) .ne. 0 ) mfit = mfit + 1
      end do

c     initialize (symmetric) alpha, beta
      do j = 1, mfit
         do k = 1, j
            alpha( j, k ) = 0.d0
         end do
         beta( j ) = 0.d0
      end do
      chisq = 0.d0

c     summation loop over all data
      sig2i = 1.d0 / ( sig*sig )
      do i = 1, ndata
c        call funcs( x( i ), a, ymod, dyda, np )
c        x( i )   : abscissa values are not concerned
c        a( j )   : vector of parameters, a( np ) = fsrcor
c        ymod     : value of the model function Delta E_{theor.} / fsrcor
c                   Delta E_{theor.} = nu_0 + Delta E_{rot}
c        dyda( j ): derivatives with respect to the parameters

c        summation over nuzero and any rotational parameters, including fsrcor correction!
         ymod = 0.d0
         do j = 1, np-1
            ymod = ymod + funcs( i, j ) * a( j )
            dyda( j ) = funcs( i, j )
         end do
c        special treatment for fsrcor (last parameter)
         dyda( np ) = -a( np ) * ymod

         dy = y( i ) - ymod
         j = 0
         do l = 1, np
            if ( ia( l ) .ne. 0 ) then
               j = j + 1
               wt = dyda( l ) * sig2i
               k = 0
               do m = 1, l
                  if ( ia( m ) .ne. 0 ) then
                     k = k + 1
                     alpha( j, k ) = alpha( j, k ) + wt * dyda( m )
                  end if
               end do
               beta( j ) = beta( j ) + dy * wt
            end if
         end do
c        and find chi-square
         chisq = chisq + dy * dy * sig2i
      end do
c     final chi-square correction by fsrcor**2
c     chisq = chisq * a( np ) * a( np )

c     fill in the symmetric side
      do j = 2, mfit
         do k = 1, j-1
            alpha( k, j ) = alpha( j, k )
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine gaussj( a, b, n, np )

c     Linear equation solution by Gauss-Jordan elimination, equation (2.1.1).
c     A(1:N,1:N) is an input matrix stored in an array of physical dimensions
c     NP by NP. B(1:N) is an input vector containing the right-hand side
c     vector, stored in an array of physical dimension NP. On output,
c     A(1:N,1:N) is replaced by its matrix inverse, and B(1:N) is replaced
c     by the corresponding solution vector.
c     Parameter NMAX is the largest anticipated value of N.

      implicit none

      integer        nmax
      parameter      ( nmax = 2 * ARNIROT_NPAR + 2 )
      integer        n, np
      integer        i, icol, irow, j, k, l, ll
c     the following integer arrays are used for bookkeeping on the pivoting
      integer        indxc( nmax ), indxr( nmax ), ipiv( nmax )

      real*8         a( np, np ), b( np )
      real*8         big, dum, pivinv

      ARNIROT_LAUNCH ( "Launching gaussj." )

      irow = 0
      icol = 0
      do j = 1, n
         ipiv( j ) = 0
      end do

c     main loop over the columns to be reduced
      do i = 1, n
         big = 0.d0
c        outer loop of the search for a pivot element
         do j = 1, n
            if ( ipiv( j ) .ne. 1 ) then
               do k = 1, n
                  if ( ipiv( k ) .eq. 0 ) then
                     if ( dabs( a( j, k ) ) .ge. big ) then
                        big = dabs( a( j, k ) )
                        irow = j
                        icol = k
                     end if
                  else if ( ipiv( k ) .gt. 1 ) then
                     pause 'singular matrix in gaussj'
                  end if
               end do
            end if
         end do
         ipiv( icol ) = ipiv( icol ) + 1

c        We now have the pivot element, so we interchange rows, if needed,
c        to put the pivot element on the diagonal. The columns are not
c        physically interchanged, only relabeled: indxc(i), the column of
c        the ith pivot element, is the ith column that is reduced, while
c        indxr(i) is the row in which that pivot element was originally
c        located. If indxr(i) != indxc(i) there is an implied column
c        interchange. With this form of bookkeeping, the solution b's will
c        end up in the correct order, and the inverse matrix will be
c        scrambled by columns.

         if ( irow .ne. icol ) then
            do l = 1, n
               dum = a( irow, l )
               a( irow, l ) = a( icol, l )
               a( icol, l ) = dum
            end do
            dum = b( irow )
            b( irow ) = b( icol )
            b( icol ) = dum
         end if

c        We are now ready to divide the pivot row by the pivot element,
c        located at irow and icol.
         indxr( i ) = irow
         indxc( i ) = icol
         if ( a( icol, icol ) .eq. 0. ) then
            pause 'singular matrix in gaussj'
         end if
         pivinv = 1.d0 / a( icol, icol )
         a( icol, icol ) = 1.d0
         do l = 1, n
            a( icol, l ) = a( icol, l ) * pivinv
         end do
         b( icol ) = b( icol ) * pivinv

c        Next, we reduce the rows ...
         do ll = 1, n
c           ... except for the pivot one, of course.
            if ( ll .ne. icol ) then
               dum = a( ll, icol )
               a( ll, icol ) = 0.d0
               do l = 1, n
                  a( ll, l ) = a( ll, l ) - a( icol, l ) * dum
               end do
               b( ll ) = b( ll ) - b( icol ) * dum
            end if
         end do
      end do

c     This is the end of the main loop over columns of the reduction. It
c     only remains to unscramble the solution in view of the column
c     interchanges. We do this by interchanging pairs of columns in the
c     reverse order that the permutation was built up.

      do l = n, 1, -1
         if ( indxr( l ) .ne. indxc( l ) ) then
            do k = 1, n
               dum = a( k, indxr( l ) )
               a( k, indxr( l ) ) = a( k, indxc( l ) )
               a( k, indxc( l ) ) = dum
            end do
         end if
      end do

c     And we are done.
      return
      end


c------------------------------------------------------------------------------
      subroutine covsrt( covar, ia, np, mfit )

c     Expand in storage the covariance matrix COVAR, so as to take into
c     account parameters that are being held fixed. For the latter, return
c     zero covariances.

      implicit none

      integer        np, mfit
      integer        ia( np )
      integer        i, j, k

      real*8         covar( np, np )
      real*8         swap

      ARNIROT_LAUNCH ( "Launching covsrt." )

      do i = mfit + 1, np
         do j = 1, i
            covar( i, j ) = 0.d0
            covar( j, i ) = 0.d0
         end do
      end do
      k = mfit
      do j = np, 1, -1
         if ( ia( j ) .ne. 0 ) then
            do i = 1, np
               swap          = covar( i, k )
               covar( i, k ) = covar( i, j )
               covar( i, j ) = swap
            end do
            do i = 1, np
               swap          = covar( k, i )
               covar( k, i ) = covar( j, i )
               covar( j, i ) = swap
            end do
            k = k - 1
         end if
      end do

      return
      end
