#include <qwt_math.h>
#include <qdrawutl.h>
#include <qwt_slider.h>
#include "qwt_slider.moc"

//-------------------------------------------------------------
//.H QwtSlider | 3 | 27/04/96 | Qwt Widget Library | Qwt Programmer's Manual
//.S  NAME
//	QwtSlider - The Slider Widget
//
//.S  SYNOPSIS
//	#include <qwt_slider.h>
//
//.S DESCRIPTION
//	QwtSlider is a slider widget which deals with double values
//	and which can have a scale.
//
//.S INHERITED CLASSES
//	\l{qwt_sldbase.html}{QwtSliderBase}, \l{qwt_sclif.html}{QwtScaleIf}
//
//.S PUBLIC MEMBERS
//.t
//    void setThumbLength(int l) --	set the lenght of the slider's thumb
//	\R{QwtSlider::QwtSlider}(QWidget *parent, const char *name = 0,	Orientation orient = Horizontal, ScalePos scalePos = None) -- Constructor
//	\R{QwtSlider::QwtSlider}() -- Destructor
//	void \R{QwtSlider::setThumbLength}(int l) --
//					Set the length of the slider's thumb
//	void \R{QwtSlider::setThumbWidth}(int w) --
//					Set the length of the slider's thumb
//	void \R{QwtSlider::setBorderWidth}(int bw) --
//					Set the border width
//
//
//------------------------------------------------------------

const int minHWidth = 40;
const int minVWidth = 10;
const int minHHeight = 10;
const int minVHeight = 40;


//------------------------------------------------------------
//
//.S	QwtSlider::QwtSlider
//
//		Constructor
//
//.s	Parameters
//.p
//	QWidget *parent --	parent widget
//	const char *name --	The Widget's name. Default = 0.
//	Orientation Orient --	Orientation of the slider. Can be QwtSlider::Horizontal
//				or QwtSlider::Vertical.
//		                Defaults to Horizontal.
//	ScalePos scalePos --	Position of the scale.  Can be QwtSlider::None,
//				QwtSlider::Left, QwtSlider::Right, QwtSlider::Top,
//				or QwtSlider::Bottom. Defaults to QwtSlider::None.
//                              
//
//.s	Syntax:
//.f	QwtSlider::QwtSlider(QWidget *parent, const char *name, Orientation orient = Horizontal, ScalePos scalePos = None)
//
//------------------------------------------------------------
QwtSlider::QwtSlider(QWidget *parent, const char *name,
		     Orientation orient, ScalePos scalePos)
: QwtSliderBase(parent,name)
{
	d_thumbLength = 30;
	d_thumbHalf = 15;
	d_thumbWidth = 16;
	d_borderWidth = 2;
	d_scaleDist = 4;
	d_orient = orient;
	d_scaleStep = 0.0;
	d_scalePos = scalePos;
	d_xMargin = 0;
	d_yMargin = 0;
	d_sliderRect.setRect(0,0,8,8);
	adjustMinSize();
}


//------------------------------------------------------------
//
//.S 	QwtSlider::~QwtSlider
// 		Destructor
//.s	Syntax
//.f	QwtSlider::~QwtSlider()
//------------------------------------------------------------
QwtSlider::~QwtSlider()
{

}

//------------------------------------------------------------
//
//.S	QwtSlider::setBorderWidth
//	Change the slider's border width
//
//.s	Parameters
//.p	int bd
//
//.s	Syntax
//.f	void QwtSlider::setBorderWidth(int bd)
//
//------------------------------------------------------------
void QwtSlider::setBorderWidth(int bd)
{
    d_borderWidth = qwtMin(qwtMax(bd,0),10);
}

//----------------------------------------------------
//
//.S 	QwtSlider::setThumbLength
//
//  	Set the slider's thumb length
//
//.s 	Parameters
//.p	int l 	--		new length
//
//.s 	Syntax
// 	void QwtSlider::setThumbLength(int l)
//-----------------------------------------------------
void QwtSlider::setThumbLength(int l)
{
	d_thumbLength = qwtMax(l,8);
	d_thumbHalf = d_thumbLength / 2;
	adjustMinSize();
	resize(size());
}

//------------------------------------------------------------
//
//.S	QwtSlider::setThumbWidth
//	Change the width of the thumb
//
//.s	Parameters
//.p	int w
//
//.s	Syntax
//.p	void QwtSlider::setThumbWidth(int w)
//
//------------------------------------------------------------
void QwtSlider::setThumbWidth(int w)
{
	d_thumbWidth = qwtMax(w,4);
	adjustMinSize();
	resize(size());
}

//------------------------------------------------------------
//.-
//.S	QwtSlider::adjustMinSize
//
//.s	Syntax
//.f	void QwtSlider::adjustMinSize()
//
//------------------------------------------------------------
void QwtSlider::adjustMinSize()
{
    QPainter p;
    int msWidth = 0, msHeight = 0;
    int w = minHWidth,h=minVHeight;
	
    if (d_scalePos != None)
    {
	if(p.begin(this))
	{
	    msWidth = d_scale.maxWidth(&p);
	    msHeight = d_scale.maxHeight(&p);
	}
	p.end();

	switch(d_orient)
	{
	case Horizontal:
	    w = 2 * d_xMargin + qwtMax(minHWidth, d_thumbLength + 2*(d_borderWidth));
	    h = 2*d_yMargin + d_thumbWidth + 2*d_borderWidth + msHeight + d_scaleDist;
	    break;
	default:
	    w = 2*d_xMargin + d_thumbWidth + 2*d_borderWidth + msWidth + d_scaleDist;
	    h = 2*d_yMargin + qwtMax(minVHeight, d_thumbLength + 2*d_borderWidth);
	    break;
	}

    }
    else				// no scale
    {
	switch(d_orient)
	{
	case Horizontal:
	    w = minHWidth;
	    h = minHHeight; 
	    break;
	    
	default:
	    w = minVWidth;
	    h = minVHeight; 
	    break;
	}
    }
    setMinimumSize(w,h);
    
}

//------------------------------------------------------------
//.-	
//.S	QwtSlider::scaleChange
//	Notify changed scale
//
//.s	Description
//	Called by QwtScaledWidget
//
//.s	Syntax
//.f	void QwtSlider::scaleChange()
//
//------------------------------------------------------------
void QwtSlider::scaleChange()
{
    if (!hasUserScale())
       d_scale.setScale(minValue(), maxValue(), d_maxMajor, d_maxMinor);
    update();
}


//------------------------------------------------------------
//.-
//.S	QwtSlider::fontChange
//	Notify changed font
//.s	Description
//	Called by QwtScaledWidget
//.s	Syntax
//.f	 QwtSlider::scaleSizeChange()
//
//------------------------------------------------------------
void QwtSlider::fontChange()
{
    adjustMinSize();
    resize(size());
}

//------------------------------------------------------------
//.-
//.S	QwtSlider::drawSlider
//		Draw the slider into the specified rectangle.	
//
//.s	Parameters:
//.p	QPainter *p, const QRect &r
//
//.s	Return Type:
//		void
//
//.s	Description:
//
//.s	Syntax:
//.f	void QwtSlider::drawSlider(QPaiter *p, const QRect &r)
//
//------------------------------------------------------------
void QwtSlider::drawSlider (QPainter *p, const QRect &r) 
{
    
    QColorGroup g = colorGroup();
    QBrush brBack(g.background());
    QBrush brMid(g.mid());
    QRect cr;
    
    int ipos,dist1;
    double rpos;
    int lineDist;
    
    qDrawShadePanel(p, r.x(), r.y(),
		    r.width(), r.height(),
		    g, TRUE, d_borderWidth,0);
    
    cr.setRect(r.x() + d_borderWidth,
	       r.y() + d_borderWidth,
	       r.width() - 2*d_borderWidth,
	       r.height() - 2*d_borderWidth);
    
    rpos = (value()  - minValue()) / (maxValue() - minValue());
    
    lineDist = d_borderWidth - 1;
    if (lineDist < 1) lineDist = 1;
    
    if (d_orient == Horizontal) 
    {
	
	dist1 = int(double(cr.width() - d_thumbLength) * rpos);
	ipos =  cr.x() + dist1; 
	markerPos = ipos + d_thumbHalf;
	
	p->fillRect(cr.x(),cr.y(),dist1,cr.height(),brMid);
	p->fillRect(ipos + d_thumbLength, cr.y(), 
		    cr.width() - d_thumbLength - dist1, cr.height(),brMid);
	
	qDrawShadePanel(p,ipos, cr.y(), d_thumbLength, cr.height(),
			colorGroup(), FALSE, d_borderWidth, &brBack);
	
	if (lineDist > 1)
	   qDrawShadeLine(p,markerPos, cr.y() + lineDist , markerPos,
			  cr.y() + cr.height() - lineDist, 
			  colorGroup(), TRUE, 1);
	else
	{
	    p->setPen(g.dark());
	    p->drawLine(markerPos -1 , cr.y() + lineDist, markerPos -1,
			cr.y() + cr.height() - lineDist - 1);
	    p->setPen(g.light());
	    p->drawLine(markerPos, cr.y() + lineDist, markerPos,
			cr.y() + cr.height() - lineDist - 1);
	}
    }
    else 
    {
	dist1 = int(double(cr.height() - d_thumbLength) * (1.0 - rpos));
	ipos = cr.y() + dist1;
	markerPos = ipos + d_thumbHalf;
	
	p->fillRect(cr.x(),cr.y(),cr.width(),ipos,brMid);
	p->fillRect(cr.x(), ipos + d_thumbLength, cr.width(),
		    cr.height() - d_thumbLength - dist1, brMid);
	
	qDrawShadePanel(p,cr.x(),ipos , cr.width(), d_thumbLength,
			colorGroup(),FALSE,d_borderWidth, &brBack);
	if (lineDist > 1)
	   qDrawShadeLine(p, cr.x() + lineDist , markerPos,
			  cr.x() + cr.width() - lineDist, markerPos,
			  colorGroup(), TRUE, 1);
	else {
		
		p->setPen(g.dark());
		p->drawLine(cr.x() + lineDist, markerPos - 1 ,
			    cr.x() + cr.width() -  lineDist - 1, markerPos - 1);
		p->setPen(g.light());
		p->drawLine(cr.x() + lineDist, markerPos,
			    cr.x() + cr.width() -  lineDist - 1 , markerPos);
	    }
    }
    
}

//------------------------------------------------------------
//.-
//.S	QwtSlider::getValue
//	Determine the value corresponding to a specified 
//	mouse location.
//
//.s	Parameters
//.p	const QPoint &p -- 
//
//.s	Description
//	Called by QwtSliderBase
//.s	Syntax
//.f     double QwtSlider::getValue(const QPoint &p)
//
//------------------------------------------------------------
double QwtSlider::getValue( const QPoint &p) 
{
    double rv;
    int pos;
    const QRect &r = this->rect();
    
    if (d_orient == Horizontal) 
    {
	
	pos = p.x() - r.x() - d_thumbHalf;
	rv  =  minValue() +
	   rint( (maxValue() - minValue()) * double(pos)
		/ double(r.width() - d_thumbLength)
		/ step() ) * step();
    }
    else 
    {
	pos = p.y() - r.y() - d_thumbHalf;
	rv =  minValue() +
	   rint( (maxValue() - minValue()) *
		(1.0 - double(pos)
		 / double(r.height() - d_thumbLength))
		/ step() ) * step();
    }
    
    return(rv);
}


//------------------------------------------------------------
//.-
//.S	QwtSlider::getScrollMode
//	Determine scrolling mode and direction 
//
//.s	Parameters
//.p	const QPoint &p
//
//.s	Description
//	Called by QwtSliderBase
//
//.s	Syntax
//.f	 void QwtSlider::getScrollMode( const QPoint &p, int &scrollMode, int &direction )
//
//------------------------------------------------------------
void QwtSlider::getScrollMode( const QPoint &p, int &scrollMode, int &direction ) 
{
    int currentPos;
    
    if (d_orient == Horizontal)
       currentPos = p.x();
    else
       currentPos = p.y();
    
    if (d_sliderRect.contains(p))
    {
	if ((currentPos > markerPos - d_thumbHalf)	
	    && (currentPos < markerPos + d_thumbHalf))
	{
	    scrollMode = ScrMouse;
	    direction = 0;
	}
	else 
	{
	    scrollMode = ScrTimer;
	    if (((currentPos > markerPos) && (d_orient == Horizontal))
		|| ((currentPos <= markerPos) && (d_orient != Horizontal)))
	       direction = 1;
	    else
	       direction = -1;
	}
    }
    else
    {
	scrollMode = ScrNone;
	direction = 0;
    }
    
}

//------------------------------------------------------------
//.-
//.S	QwtSlider::paintEvent 
//	Qt paint event
//
//.s	Parameters
//.p	QPaintEvent *e
//
//.s	Syntax
//.f	void QwtSlider::paintEvent(QPaintEvent *e)
//
//------------------------------------------------------------
void QwtSlider::paintEvent(QPaintEvent *e)
{
    QPainter p;
    
    if(p.begin(this))
    {
	if (d_scalePos != None)
	{
	    p.fillRect(this->rect(), this->backgroundColor());
	    d_scale.draw(&p); 
	}
	
	drawSlider(&p, d_sliderRect);
    }
    p.end();
    
}

//------------------------------------------------------------
//
//.S	QwtSlider::resizeEvent
//	Qt resize event
//
//.s	Parameters
//.p	QResizeEvent *e
//
//.s	Syntax
//.f	void QwtSlider::resizeEvent(QResizeEvent *e)
//
//------------------------------------------------------------
void QwtSlider::resizeEvent(QResizeEvent *e)
{
    
    d_resized = true;
    QSize s = e->size();
    int sliderWidth = d_thumbWidth + 2*d_borderWidth;
    
    // reposition slider
    if(d_orient == Horizontal)
    {
	switch(d_scalePos)
	{
	case Top:
	    
	    d_sliderRect.setRect(this->rect().x() + d_xMargin,
				 this->rect().y() + s.height() - 1
				 - d_yMargin - sliderWidth,
				 s.width() - 2 * d_xMargin,
				 sliderWidth);
	    d_scale.setGeometry(d_sliderRect.x() + d_borderWidth + d_thumbHalf,
				d_sliderRect.y() - d_scaleDist,
				d_sliderRect.width() - d_thumbLength - 2*d_borderWidth,
				QwtScaleDraw::Top);
	    
	    break; 
	    
	case Bottom:
	    
	    d_sliderRect.setRect(this->rect().x() + d_xMargin,
				 this->rect().y() + d_yMargin,
				 s.width() - 2*d_xMargin,
				 sliderWidth);
	    d_scale.setGeometry(d_sliderRect.x() + d_borderWidth + d_thumbHalf,
				d_sliderRect.y() + d_sliderRect.height() +  d_scaleDist,
				d_sliderRect.width() - d_thumbLength - 2*d_borderWidth,
				QwtScaleDraw::Bottom);
	    
	    break; 
	    
	default:
	    d_sliderRect.setRect(this->rect().x(), this->rect().x(),
				 s.width(), s.height());
	    break; 
	}
    }
    else
    {
	switch(d_scalePos)
	{
	case Left:
	    d_sliderRect.setRect(this->rect().x() + s.width()
				 - sliderWidth - 1 - d_xMargin,
				 this->rect().y() + d_yMargin,
				 sliderWidth,
				 s.height() - 2 * d_yMargin);
	    d_scale.setGeometry(d_sliderRect.x() - d_scaleDist,
				d_sliderRect.y() + d_thumbHalf + d_borderWidth,
				s.height() - d_thumbLength - 2*d_borderWidth,
				QwtScaleDraw::Left);
	    
	    break; 
	case Right:
	    d_sliderRect.setRect(this->rect().x() + d_xMargin,
				 this->rect().y() + d_yMargin,
				 sliderWidth,
				 s.height() - 2* d_yMargin);
	    d_scale.setGeometry(this->rect().x() + d_sliderRect.width()
				+ d_scaleDist,
				d_sliderRect.y() + d_thumbHalf + d_borderWidth,
				s.height() - d_thumbLength - 2*d_borderWidth,
				QwtScaleDraw::Right);
	    break; 
	default:
	    d_sliderRect.setRect(this->rect().x(), this->rect().x(),
				 s.width(), s.height());
	    break; 
	}
    }
    
}

//------------------------------------------------------------
//.-
//.S	QwtSlider::valueChange
//	Notify changed value
//
//.s	Syntax
//.f	void QwtSlider::valueChange()
//
//------------------------------------------------------------
void QwtSlider::valueChange()
{
    QPainter p;
    
    p.begin(this);
    drawSlider(&p,d_sliderRect);
    p.end();
    
    QwtSliderBase::valueChange();
    
}


//------------------------------------------------------------
//.-	Notify changed range
//.S	QwtSlider::rangeChange
//
//
//.s	Description
//
//.s	Syntax
//.f	void QwtSlider::rangeChange()
//
//------------------------------------------------------------
void QwtSlider::rangeChange()
{
    if (!hasUserScale())
       d_scale.setScale(minValue(), maxValue(), d_maxMajor, d_maxMinor);
    repaint(FALSE);
}


























