/**************************************************************************
 * $Id: SamplinScaleBar.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3 beta $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include <SamplinScaleBar.h>
#include <qpainter.h>
#include <qfontmet.h>
#include <qrect.h>
#include <qwt_math.h>

static char *styleNames[]={"None","Scientific","General","Fixed",NULL
};

int const SamplinScaleBar::minLen = 10;

const double step_eps = 1.0e-6;
const double WorstCase = -8.8888888888888888888888e-88;

SamplinScaleBar::SamplinScaleBar()
{
   d_hpad = 3;
   d_vpad = 3;
   d_majLen = 8;
   d_minLen = 4;
   d_medLen = 6;
   
   d_minAngle = -135 * 16;
   d_maxAngle = 135 * 16;
   d_fmt = General;
   d_prec = 4;
   // initialize scale and geometry
   setGeometry(0,0,100,Bottom);
//   setScale(0,100,0,0,10);

   d_pen=QPen(black,0,SolidLine);
   d_font=QFont("Helvetica",8);
}

SamplinScaleBar::~SamplinScaleBar()
{
   
}

/*
void SamplinScaleBar::setScale(double x1, double x2, int maxMajIntv,
			    int maxMinIntv, double step, int logscale)
{
    d_scldiv.rebuild(x1,x2,maxMajIntv, maxMinIntv,logscale,step, false);
    setDblRange(d_scldiv.lBound(),d_scldiv.hBound(),d_scldiv.logScale());
}

void SamplinScaleBar::setScale(const QwtScaleDiv &s)
{
    d_scldiv = s;
    setDblRange(d_scldiv.lBound(),d_scldiv.hBound(),d_scldiv.logScale());
}
 */

void SamplinScaleBar::draw(QPainter *p) const
{

   double val,hval,majTick;
   
   int i,k,kmax,lwidth;
   
   lwidth=maxLabelWidth(p,FALSE);
   p->setFont(d_font);
   p->setPen(d_pen);
   
   if(d_scldiv.majCnt()==0 && d_scldiv.minCnt()==0)return;
   
   for (i=0; i< d_scldiv.majCnt(); i++)
    {
       val = d_scldiv.majMark(i);
       drawTick(p, val, d_majLen);
       drawLabel(p, val, lwidth);
    }

    if (d_scldiv.logScale())
    {
	for (i=0; i< d_scldiv.minCnt(); i++)
	{
	    drawTick(p,d_scldiv.minMark(i),d_minLen);
	}
    }
    else
    {
	k = 0;
	kmax = d_scldiv.majCnt() - 1;
	if (kmax > 0) 
	{
	   majTick = d_scldiv.majMark(0);
	   hval = majTick - 0.5 * d_scldiv.majStep();

	   for (i=0; i< d_scldiv.minCnt(); i++)
	   {
	       val = d_scldiv.minMark(i);
	       if  (val > majTick)
	       {
		   if (k < kmax)
		   {
		       k++;
		       majTick = d_scldiv.majMark(k);
		   }
		   else
		   {
		       majTick += d_scldiv.majMark(kmax) + d_scldiv.majStep();
		   }
		   hval = majTick - 0.5 * d_scldiv.majStep();
		   
	       }
	       if (fabs(val-hval) < step_eps * d_scldiv.majStep())
		  drawTick(p, val, d_medLen);
	       else
		  drawTick(p, val, d_minLen);
	   }
	}
    }

    //
    // draw backbone
    //
    //if (d_baseEnabled)
       drawBackbone(p);

}

void SamplinScaleBar::drawTick(QPainter *p, double val, int len) const
{
    
  int tval = transform(val);
  float arc;
  int bw2;

  bw2 = p->pen().width() / 2;
    
  switch(d_orient)
    {
    case Right:

	  p->drawLine(d_xorg, tval, d_xorg + len, tval);
	  break;

    case Bottom: 

	  p->drawLine(tval, d_yorg, tval, d_yorg + len);
	  break;

    case Left:

	  p->drawLine(d_xorg, tval, d_xorg - len, tval);
	  break;

    case Round:

	  arc = float(tval) / 16.0 - 90;
	  p->translate(d_xorg, d_yorg);
	  p->rotate(arc);
	  p->drawLine(d_len + bw2,0,d_len + len + bw2,0);
	  p->resetXForm();
	  break;
	
    case Top:
    default:
	
	  p->drawLine(tval, d_yorg, tval, d_yorg - len);
	  break;
	

    }
    
}

void SamplinScaleBar::drawLabel(QPainter *p, double val, int lwidth) const
{

    static QString label;
    static double pi_4 = M_PI * 0.25;
    static double pi_75 = M_PI * 0.75;
    
    double arc;
    int xpos, ypos;
    int tval;

//   p->setFont(d_font);
   QFontMetrics fm = p->fontMetrics();
    
    tval = transform(val);

    // correct rounding errors if val = 0
    if ((!d_scldiv.logScale()) && (fabs(val) < fabs(step_eps * d_scldiv.majStep())))
       val = 0.0;
    
   switch(d_fmt){
    case None:
      label="";
      break;
    case Scientific:
      label.sprintf("%.*e",d_prec,val);
      break;
    case General:
      label.sprintf("%.*g",d_prec,val);
      break;
    case Fixed:
      label.sprintf("%.*f",d_prec,val);
      break;
   }

    switch(d_orient)
    {
    case Right:
//       p->drawLine(d_xorg+d_majLen+d_hpad,0,d_xorg+d_majLen+d_hpad,100);
       p->drawText(d_xorg + d_majLen + d_hpad +2 + lwidth-fm.width(label),
		    tval + (fm.ascent()) / 2,
		    label);
	break;
    case Left:
//       p->drawText(0,tval-(fm.ascent())/2,d_xorg - d_majLen - d_hpad,
//		   tval+fm.ascent()/2+fm.descent(),AlignRight|AlignTop,
//		   label);
//       p->drawLine(d_xorg - d_majLen - d_hpad,0,d_xorg - d_majLen - d_hpad,100);
       p->drawText(d_xorg - d_majLen - d_hpad - fm.width(label),
		    tval + (fm.ascent()) / 2,
		    label);
	break;
    case Bottom:
	p->drawText(tval - (fm.width(label)-1) / 2, d_yorg + d_majLen + d_vpad + fm.ascent(), label);
	break;
    case Round:
	arc = double(tval) / 16.0 * M_PI / 180.0;

	// Map arc into the interval -pi <= arc <= pi
	if ((arc < -M_PI) || (arc > M_PI))
	   arc -= floor((arc + M_PI) / M_PI * 0.5) * 2.0 * M_PI;
	
	xpos = d_xorg + int(rint(double(d_len + d_majLen + d_vpad) * sin(arc)));
	ypos = d_yorg - int(rint(double(d_len + d_majLen + d_vpad) * cos(arc)));
	
	if (arc < -pi_75)
	{
	    p->drawText(xpos - int(rint(double(fm.width(label))
					* (1.0 + (arc + pi_75) * M_2_PI) )),
			ypos + fm.ascent() - 1,
			label);
	}
	else if (arc < -M_PI_4)
	{
	    p->drawText(xpos - fm.width(label),
			ypos - int(rint(double(fm.ascent() - 1)
					* (arc + M_PI_4) * M_2_PI)),
			label);
	}
	else if (arc < pi_4)
	{
	    p->drawText(xpos + int(rint(double(fm.width(label))
					* ( arc - M_PI_4 ) * M_2_PI )),
			ypos,
			label);
	}
	else if (arc < pi_75)
	{
	    p->drawText(xpos,
			ypos + int(rint(double(fm.ascent() - 1)
					* (arc - M_PI_4) * M_2_PI)),
			label);
	}
	else
	{
	    p->drawText(xpos - int(rint(double(fm.width(label))
					* ( arc - pi_75) * M_2_PI )),
			ypos + fm.ascent() - 1,
			label);
	}
	break;
    case Top:
    default:
	p->drawText(tval - (fm.width(label)-1) / 2, d_yorg - d_majLen - d_vpad-2, label);
	break;
    }
    


}

void SamplinScaleBar::drawBackbone(QPainter *p) const
{
  int bw2;
  int a1, a2;
  bw2 = p->pen().width() / 2;
    
    
  switch(d_orient)
    {
    case Left:
	  p->drawLine(d_xorg - bw2, d_yorg, d_xorg - bw2, d_yorg + d_len - 1);
	  break;
    case Right:
	  p->drawLine(d_xorg + bw2, d_yorg, d_xorg + bw2, d_yorg + d_len - 1);
	  break;
    case Round:

	  a1 = qwtMin(i1(), i2()) - 90 * 16; 
	  a2 = qwtMax(i1(), i2()) - 90 * 16; 
	
	  p->drawArc(d_xorg - d_len - bw2, d_yorg - d_len - bw2,
				 2*(d_len + bw2) + 1,
				 2*(d_len + bw2) + 1,
				 -a2, a2 - a1);	// counterclockwise

	  break;
	
    case Top:
	  p->drawLine(d_xorg, d_yorg - bw2, d_xorg + d_len - 1, d_yorg-bw2);
	  break;
    case Bottom:
	  p->drawLine(d_xorg, d_yorg+bw2, d_xorg + d_len - 1, d_yorg+bw2);
	  break;
    default:
	  p->drawLine(d_xorg, d_yorg, d_xorg + d_len - 1, d_yorg);
	  break;
    }
    
}

void SamplinScaleBar::setGeometry(int xorigin, int yorigin, int length, Orientation o)
{

    d_xorg = xorigin;
    d_yorg = yorigin;

    if (length > minLen)
       d_len = length;
    else
       d_len = minLen;

    d_orient = o;
    
    switch(d_orient)
    {
    case Left:
    case Right:
	setIntRange(d_yorg + d_len - 1, d_yorg);
	break;
    case Round:
	setIntRange(d_minAngle, d_maxAngle);
	break;
    case Top:
    case Bottom:
    default:
	setIntRange(d_xorg, d_xorg + d_len - 1);
	break;
    }
}

int SamplinScaleBar::maxWidth(QPainter *p, bool worst) const
{
   int rv = 0;
   int bw = p->pen().width();
   
   QString s;
    
   rv = maxLabelWidth(p,worst);

   switch (d_orient)
    {
    case Left:
    case Right:
	rv += bw + d_hpad + bw + d_majLen;
	break;
    case Round:
	rv += bw + d_vpad + bw + d_majLen;
	break;
    case Top:
    case Bottom:
    default:
	rv += d_len;
    }

   return rv;
    
}

int SamplinScaleBar::maxHeight(QPainter *p) const 
{

    int rv = 0;
    int bw = p->pen().width();
    
   p->setFont(d_font);
   QFontMetrics fm = p->fontMetrics();

    switch (d_orient)
    {
    case Top:
    case Bottom:
    case Round:
	rv = bw + d_vpad + bw + d_majLen + fm.ascent();
	break;
    case Left:
    case Right:
    default:
	rv = d_len + (fm.ascent() + fm.descent() + 1) / 2;
    }
    
    return rv;

}

QRect SamplinScaleBar::maxBoundingRect(QPainter *p) const
{
    int wl,h,wmax,hmax;
    QRect r;

    QFontMetrics fm = p->fontMetrics();

    wl = maxLabelWidth(p, TRUE);
    h = fm.height();

    switch(d_orient)
    {
    case Left:

	r = QRect( d_xorg - d_hpad - d_majLen - wl,
		  d_yorg - fm.ascent(),
		  d_majLen + d_hpad + wl,
		  d_len + fm.height());
	break;
	
    case Right:

	r = QRect( d_xorg,
		  d_yorg - fm.ascent(),
		  d_majLen + d_hpad + wl,
		  d_len + fm.height());
	break;
	
    case Top:

	r = QRect ( d_xorg - wl / 2,
		   d_yorg - d_majLen - fm.ascent(),
		   d_len + wl,
		   d_majLen + d_vpad + fm.ascent());
	break;
	
    case Bottom:

	r = QRect ( d_xorg - wl / 2,
		   d_yorg,
		   d_len + wl,
		   d_majLen + d_vpad + fm.height());
	break;

    case Round:

	wmax = d_len + d_majLen + d_hpad + wl;
	hmax = d_len + d_majLen + d_hpad + fm.ascent();

	r = QRect ( d_xorg - wmax,
		   d_yorg - hmax,
		   2*wmax,
		   2*hmax);
	break;
    }

    return r;
	
}

void SamplinScaleBar::setAngleRange(double angle1, double angle2)
{
    int amin, amax;
    
    amin = int(rint(qwtMin(angle1, angle2) * 16.0));
    amax = int(rint(qwtMax(angle1, angle2) * 16.0));

    if (amin == amax)
    {
	amin -= 1;
	amax += 1;
    }

    d_minAngle = amin;
    d_maxAngle = amax;
    setIntRange(d_minAngle, d_maxAngle);

}

void SamplinScaleBar::setLabelFormat(Format f, int prec)
{
   d_fmt = f;
   d_prec = prec;
   if(d_prec<0)d_prec=0;
   if(d_fmt<None || d_fmt>Fixed)d_fmt=General;
}

int SamplinScaleBar::maxLabelWidth(QPainter *p, int worst) const
{
   
   int i,rv = 0;
   double val;
   QString s;
   
   p->setFont(d_font);
   QFontMetrics fm = p->fontMetrics();
   
    if (worst) // worst case
     {
	switch(d_fmt){
	 case None:
	   s="";
	   break;
	 case Scientific:
	   s.sprintf("%.*e",d_prec,WorstCase);
	   break;
	 case General:
	   s.sprintf("%.*g",d_prec,WorstCase);
	   break;
	 case Fixed:
	   s.sprintf("%.*f",d_prec,WorstCase);
	   break;
	}
//	s.setNum(WorstCase, d_fmt, d_prec);
	rv = fm.width(s);
     }
   else // actual width
     {
	for (i=0;i<d_scldiv.majCnt(); i++)
	 {
	    val = d_scldiv.majMark(i);
	    // correct rounding errors if val = 0
	    if ((!d_scldiv.logScale()) && (fabs(val) < step_eps * fabs(d_scldiv.majStep())))
	      val = 0.0;

	    switch(d_fmt){
	     case None:
	       s="";
	       break;
	     case Scientific:
	       s.sprintf("%.*e",d_prec,val);
	       break;
	     case General:
	       s.sprintf("%.*g",d_prec,val);
	       break;
	     case Fixed:
	       s.sprintf("%.*f",d_prec,val);
	       break;
	    }
//	    s.setNum(val, d_fmt, d_prec);
	    rv = qwtMax(rv,fm.width(s));
	 }
    }
   
   return rv;
   
}

void SamplinScaleBar::update(void)
{
   setDblRange(d_scldiv.lBound(),d_scldiv.hBound(),d_scldiv.logScale());  
}

int SamplinScaleBar::style(const char *str)
{
   char **s=styleNames;
   int i=0;
   
   while(*s!=0){
      if(!strcasecmp(*s,str)&&strlen(*s)==strlen(str))return i;
      ++s;++i;
   }
   
   return -1;
}



