/* -*- mode: C++; c-file-style: "GNU" -*-
 *
 * $Id: TableOfContentsDialog.C,v 1.16 1999/01/15 11:46:22 kuepper Exp $
 */

#include "TableOfContentsDialog.h"
#include <qslider.h>

#include <kapp.h>
#include <klocale.h>
#include <ktreelist.h>
#include <kiconloader.h>

#include <qbttngrp.h>
#include <qpushbt.h>
#include <qradiobt.h>
#include <qlcdnum.h>

#include <ctype.h>

#include "LyXView.h"
#include "lyx_gui_misc.h"
#include <qlayout.h>

struct TocList {
	char counter[6];
	bool appendix;
        KTreeListItem* item;
	TocList *next;
};

static TocList* toclist = NULL;

TableOfContentsDialog::TableOfContentsDialog( LyXView* view, QWidget * parent=0, 
					      const char * name=0, WFlags f=0 ) :
  QDialog( parent, name, false, f ),
  _view( view )
{
  // the treelist + buttons <=> picture buttons
  QHBoxLayout *hlayout1 = new QHBoxLayout(this, 10);
  // the treelist <=> buttons
  QVBoxLayout *vlayout1 = new QVBoxLayout();
  // the picture buttons
  QVBoxLayout *vlayout2 = new QVBoxLayout(5);
  // the buttons
  QHBoxLayout *hlayout2 = new QHBoxLayout();
  // the depth slider
  QHBoxLayout *hlayout3 = new QHBoxLayout();

  hlayout1->addLayout(vlayout1, 1);
  hlayout1->addLayout(vlayout2, 0);

  treeview = new KTreeList( this );
  treeview->setExpandButtonDrawing( true );
  treeview->adjustSize();
  QObject::connect( treeview, SIGNAL( selected( int ) ), 
					this, SLOT( select( int ) ) );
  vlayout1->addWidget(treeview, 10);
  vlayout1->addLayout(hlayout3, 0);
  vlayout1->addLayout(hlayout2, 0);
  
  KIconLoader* loader = kapp->getIconLoader();
  QPushButton* leftPB = new QPushButton( this );
  leftPB->setPixmap( loader->loadIcon( "left.xpm" ) );
  leftPB->adjustSize();
  leftPB->setMaximumSize(leftPB->size());
  leftPB->setMinimumSize(leftPB->size());
  vlayout2->addWidget(leftPB, 0);
  QObject::connect( leftPB, SIGNAL( clicked() ), this, SLOT( leftClicked() ) );

  QPushButton* rightPB = new QPushButton( this );
  rightPB->setPixmap( loader->loadIcon( "right.xpm" ) );
  rightPB->adjustSize();
  rightPB->setMaximumSize(rightPB->size());
  rightPB->setMinimumSize(rightPB->size());
  vlayout2->addWidget(rightPB);
  QObject::connect( rightPB, SIGNAL( clicked() ),
					this, SLOT( rightClicked() ) );

  QPushButton* upPB = new QPushButton( this );
  upPB->setPixmap( loader->loadIcon( "up.xpm" ) );
  upPB->adjustSize();
  upPB->setMaximumSize(upPB->size());
  upPB->setMinimumSize(upPB->size());
  vlayout2->addWidget(upPB);
  QObject::connect( upPB, SIGNAL( clicked() ),
					this, SLOT( upClicked() ) );
  
  QPushButton* downPB = new QPushButton( this );
  downPB->setPixmap( loader->loadIcon( "down.xpm" ) );
  downPB->adjustSize();
  downPB->setMaximumSize(downPB->size());
  downPB->setMinimumSize(downPB->size());
  vlayout2->addWidget(downPB);
  QObject::connect( downPB, SIGNAL( clicked() ),
					this, SLOT( downClicked() ) );
  vlayout2->addStretch(1);

  QLabel* tmplabel = new QLabel( i18n( "Depth" ), 
				 this );
  tmplabel->adjustSize();
  tmplabel->setMinimumSize(tmplabel->size());
  hlayout3->addWidget(tmplabel);
  
  tocdepthSL = new QSlider( -1, 5, 1, 3, QSlider::Horizontal, this );
  connect( tocdepthSL, SIGNAL( valueChanged( int ) ),
	   SLOT( depthChanged( int ) ) );
  hlayout3->addWidget(tocdepthSL, 2);
  tocdepthLC = new QLCDNumber( 2, this );
  tocdepthLC->adjustSize();
  tocdepthLC->setMinimumSize(tocdepthLC->size());
  hlayout3->addWidget(tocdepthLC, 0);

  hlayout2->addStretch( 1 );
  QPushButton* gotoPB = new QPushButton( i18n( "&Goto" ), this );
  gotoPB->adjustSize();
  gotoPB->setFixedHeight( gotoPB->height() );
  hlayout2->addWidget( gotoPB, 2 );
  QObject::connect( gotoPB, SIGNAL( clicked() ),
					this, SLOT( gotoSection() ) );

  hlayout2->addStretch( 1 );
  QPushButton* updatePB = new QPushButton( i18n( "&Update" ), this );
  updatePB->adjustSize();
  updatePB->setMaximumHeight(updatePB->height());
  updatePB->setMinimumHeight(updatePB->height());
  hlayout2->addWidget(updatePB, 2);
  QObject::connect( updatePB, SIGNAL( clicked() ),
					this, SLOT( updateToc() ) );

  QPushButton* closePB = new QPushButton( i18n( "&Close" ), this );
  closePB->adjustSize();
  closePB->setMaximumHeight(closePB->height());
  
  hlayout2->addWidget(closePB, 2);
  QObject::connect( closePB, SIGNAL( clicked() ),
					this, SLOT( hide() ) );
  hlayout2->addStretch(1);

  hlayout1->activate();

  setCaption( i18n( "Document Structure" ) );

  tocdepth = _view->currentBuffer()->params.tocdepth;
  treeview->setExpandLevel( 6);

  tocdepthSL->setValue( tocdepth );
  tocdepthLC->display( tocdepth );

  levelindices = new int[10];
  clear();
  if (height()<QApplication::desktop()->height()*3/4)
    resize(width(), QApplication::desktop()->height()*3/4);
}


TableOfContentsDialog::~TableOfContentsDialog()
{
  delete [] levelindices;
}

void TableOfContentsDialog::updateToc()
{
  static LyXParagraph* stapar = NULL;
  TocList *tmptoclist = NULL;
  
  /* deleted the toclist */ 
  if (toclist){
	while (toclist){
	  tmptoclist = toclist->next;
	  delete toclist;
	  toclist = tmptoclist;
	}
  }
  toclist = NULL;
  tmptoclist = toclist;

  clear();
  if( !_view->currentView()->available() ) {
	insertEntry( i18n("*** No Document ***"), 0 );
	return;
  }
	
  treeview->hide();

  /* get the table of contents */ 
  LyXParagraph *par = _view->currentView()->currentBuffer()->paragraph;
  LyXParagraph *cursorpar = 
    _view->currentView()->currentBuffer()->text->cursor.par->FirstPhysicalPar();
  char labeltype;
  char* line = new char[200];
  int i = 0;
  int pos = 0;
  unsigned char c;
  
  stapar = par;
  bool cursor_already_set = false;
  bool run_over_cursor = false;
   
  while (par) {
	labeltype = lyxstyle.Style(_view->currentView()->currentBuffer()->params.textclass, 
				   par->GetLayout())->labeltype;
	
	if (labeltype >= LABEL_COUNTER_CHAPTER
		&& labeltype <= LABEL_COUNTER_CHAPTER +
		tocdepth) {
	  /* insert this into the table of contents */ 
	  pos = 0;
	  int level = labeltype - 
		lyxstyle.TextClass(_view->currentView()->currentBuffer()->params.textclass)->maxcounter; 

	  /* then the labelstring */ 
	  i = 0;
	  if (!par->labelstring.empty()) {
		while (pos < 199 && i < par->labelstring.length()) {
		  line[pos] = par->labelstring[i];
		  i++;
		  pos++;
		}
	  }
	  
	  line[ pos++ ] = ':';
	  line[ pos++ ] = ' ';
	  
	  /* now the contents */ 
	  i = 0;
	  while (pos < 199 && i < par->last) {
		c = par->GetChar(i);
		if (isprint(c) || c >= 128) {
		  line[pos] = c;
		  pos++;
		}
		i++;
	  }
	  line[pos] = '\0';
	  insertEntry( line, level );

	  /* make a toclist entry */
	  if (!tmptoclist){
		tmptoclist = new TocList;
		toclist = tmptoclist;
	  } else {
		tmptoclist->next = new TocList;
		tmptoclist = tmptoclist->next;
	  }
	  
	  tmptoclist->next = NULL;
	  int a = 0;
	  for (a=0; a<6; a++){
		tmptoclist->counter[a] = par->GetCounter(a);
	  }
	  tmptoclist->appendix = par->appendix;
	  tmptoclist->item = treeview->itemAt(current_index-1);
	}
	run_over_cursor |= (par == cursorpar);
	if (run_over_cursor && !cursor_already_set && current_index > 0){
	  cursor_already_set = true;
	  treeview->setCurrentItem(current_index-1);
	}
	  
	par = par->LastPhysicalPar()->Next();
	
  }
  delete[] line;

  treeview->show();
}



void TableOfContentsDialog::clear()
{
  treeview->clear();
  for( int i = 0; i < 10; i++ )
	levelindices[i] = -1;
  current_index = 0;
}


void TableOfContentsDialog::insertEntry( const char* line, int level )
{
  // all indices at lower (numerically higher) levels are worthless from now
  for( int i = level+1; i < 10; i++ )
	levelindices[i] = -1;


  printf("insert entry in level %d: %s\n", level, line);

  if( level == 0 ) // top-level entry
	treeview->insertItem( line, 0, -1 );
  else
	{
	  /* Find a parent.
		 If there is not yet an entry at the parent level, generate one. */
	  generateAncestorsIfNecessary( level );
	  int parent = levelindices[level-1];
	  treeview->addChildItem( line, 0, parent );
	}
  
  levelindices[level] = current_index;
  current_index++;
}


void TableOfContentsDialog::gotoSection()
{
  select( treeview->currentItem() );
}


void TableOfContentsDialog::select( int item )
{
  if (!_view->currentView()->available())
	return;
  
  TocList* tmptoclist = toclist;
  while (tmptoclist && tmptoclist->item != treeview->itemAt(item))
    tmptoclist = tmptoclist->next;
  
  if (!tmptoclist)
	return;
  
  
  LyXParagraph *par = _view->currentView()->currentBuffer()->paragraph;
  while (par && 
		 (
		  par->GetCounter(0) != tmptoclist->counter[0] ||
		  par->GetCounter(1) != tmptoclist->counter[1] ||
		  par->GetCounter(2) != tmptoclist->counter[2] ||
		  par->GetCounter(3) != tmptoclist->counter[3] ||
		  par->GetCounter(4) != tmptoclist->counter[4] ||
		  par->GetCounter(5) != tmptoclist->counter[5] ||
		  par->appendix != tmptoclist->appendix
		  )
		 ) {
	par = par->LastPhysicalPar()->Next();
  }
  
  if (par) {
	_view->currentView()->getScreen()->HideCursor(true);
	_view->currentView()->currentBuffer()->text->SetCursor(par, 0);
	_view->currentView()->currentBuffer()->text->sel_cursor =
	  _view->currentView()->currentBuffer()->text->cursor;
	_view->currentView()->currentBuffer()->updateFull(false);
  }
  else {
	WriteAlert(i18n("Error"), 
			   i18n("Couldn't find this label\n"
				"in current document."));
  }
}

void TableOfContentsDialog::generateAncestorsIfNecessary( int level )
{
  // If there already is an entry at the parent level, we are done.
  if( levelindices[level-1] != -1 )
	return;

  // If we are at level 1 and there is no parent, simply generate one 
  // and return.
  if( level == 1 )
	{
	  treeview->insertItem( i18n( "(*** No entry ***)" ), 0, -1 );
	  levelindices[0] = current_index;
	  current_index++;
	  return;
	}

  // Else, generate the ancestors of this level and afterwards this one
  generateAncestorsIfNecessary( level-1 );
  treeview->addChildItem( i18n( "(*** No entry ***)" ), 0, 
			  levelindices[level-2] );
  levelindices[level-1] = current_index;
  current_index++;
}


void TableOfContentsDialog::leftClicked() 
{
  int index = treeview->currentItem();
  KTreeListItem* item = treeview->itemAt(index);

  if (!_view->currentView()->available())
	return;
  select(index);
  TocList* tmptoclist = toclist;
  while (tmptoclist && tmptoclist->item != item)
    tmptoclist = tmptoclist->next;
  if (!tmptoclist)
	return;
  LyXParagraph *par = 
    _view->currentView()->currentBuffer()->text->cursor.par;
  if (par && 
      !(
	par->GetCounter(0) != tmptoclist->counter[0] ||
	par->GetCounter(1) != tmptoclist->counter[1] ||
	par->GetCounter(2) != tmptoclist->counter[2] ||
	par->GetCounter(3) != tmptoclist->counter[3] ||
	par->GetCounter(4) != tmptoclist->counter[4] ||
	par->GetCounter(5) != tmptoclist->counter[5] ||
	par->appendix != tmptoclist->appendix
	)
      ) {
    _view->currentView()->getScreen()->HideCursor();
    _view->currentBuffer()->updateCursorMove(false);
    _view->currentBuffer()->text
      ->SetLayout(lyxstyle.LeftOfLayout(_view->currentView()->currentBuffer()->params.textclass, 
					par->GetLayout()));
    _view->currentBuffer()->updateFull();

  }

  updateToc();
}


void TableOfContentsDialog::rightClicked()
{
  int index = treeview->currentItem();
  KTreeListItem* item = treeview->itemAt(index);

  if (!_view->currentView()->available())
	return;
  select(index);
  TocList* tmptoclist = toclist;
  while (tmptoclist && tmptoclist->item != item)
    tmptoclist = tmptoclist->next;
  if (!tmptoclist)
	return;
  LyXParagraph *par = 
    _view->currentView()->currentBuffer()->text->cursor.par;
  if (par && 
      !(
       par->GetCounter(0) != tmptoclist->counter[0] ||
       par->GetCounter(1) != tmptoclist->counter[1] ||
       par->GetCounter(2) != tmptoclist->counter[2] ||
       par->GetCounter(3) != tmptoclist->counter[3] ||
       par->GetCounter(4) != tmptoclist->counter[4] ||
       par->GetCounter(5) != tmptoclist->counter[5] ||
       par->appendix != tmptoclist->appendix
       )
      ) {
    _view->currentView()->getScreen()->HideCursor();
    _view->currentBuffer()->updateCursorMove(false);
    _view->currentBuffer()->text
      ->SetLayout(lyxstyle.RightOfLayout(_view->currentView()->currentBuffer()->params.textclass, 
					par->GetLayout()));
    _view->currentBuffer()->updateFull();

  }

  updateToc();
}


void TableOfContentsDialog::upClicked()
{
  int index = treeview->currentItem();
  KTreeListItem* item = treeview->itemAt(index);

  if (!_view->currentView()->available())
	return;
  select(index);
  TocList* tmptoclist = toclist;
  while (tmptoclist && tmptoclist->item != item)
    tmptoclist = tmptoclist->next;
  if (!tmptoclist)
	return;
  LyXText* text = _view->currentView()->currentBuffer()->text;
  LyXParagraph *par = text->cursor.par;
  if (par && 
      !(
       par->GetCounter(0) != tmptoclist->counter[0] ||
       par->GetCounter(1) != tmptoclist->counter[1] ||
       par->GetCounter(2) != tmptoclist->counter[2] ||
       par->GetCounter(3) != tmptoclist->counter[3] ||
       par->GetCounter(4) != tmptoclist->counter[4] ||
       par->GetCounter(5) != tmptoclist->counter[5] ||
       par->appendix != tmptoclist->appendix
       )
      ) {
    _view->currentView()->getScreen()->HideCursor();
    _view->currentBuffer()->updateCursorMove(false);


    /* clear the selection */ 
    _view->currentView()->getScreen()->ToggleSelection();
    text->ClearSelection();
    _view->currentView()->currentBuffer()->updateFull(false);

    /* move the structure */
    text->StructureUp(par);

    /* update the buffer */
    _view->currentBuffer()->updateFull();
  }

  updateToc();
}


void TableOfContentsDialog::downClicked()
{
  int index = treeview->currentItem();
  KTreeListItem* item = treeview->itemAt(index);

  if (!_view->currentView()->available())
	return;
  select(index);
  TocList* tmptoclist = toclist;
  while (tmptoclist && tmptoclist->item != item)
    tmptoclist = tmptoclist->next;
  if (!tmptoclist)
	return;
  LyXText* text = _view->currentView()->currentBuffer()->text;
  LyXParagraph *par = text->cursor.par;
  if (par && 
      !(
       par->GetCounter(0) != tmptoclist->counter[0] ||
       par->GetCounter(1) != tmptoclist->counter[1] ||
       par->GetCounter(2) != tmptoclist->counter[2] ||
       par->GetCounter(3) != tmptoclist->counter[3] ||
       par->GetCounter(4) != tmptoclist->counter[4] ||
       par->GetCounter(5) != tmptoclist->counter[5] ||
       par->appendix != tmptoclist->appendix
       )
      ) {
    _view->currentView()->getScreen()->HideCursor();
    _view->currentBuffer()->updateCursorMove(false);


    /* clear the selection */ 
    _view->currentView()->getScreen()->ToggleSelection();
    text->ClearSelection();
    _view->currentView()->currentBuffer()->updateFull(false);

    /* move the structure */
    text->StructureDown(par);

    /* update the buffer */
    _view->currentBuffer()->updateFull();
  }

  updateToc();
}

void TableOfContentsDialog::depthChanged(int value){
  if (value == tocdepth)
    return;
  tocdepth = value;
  tocdepthLC->display( tocdepth );
  updateToc();
}


/*
 * $Log: TableOfContentsDialog.C,v $
 * Revision 1.16  1999/01/15 11:46:22  kuepper
 * Made TOC inset editable to open the TableOfContents dilaog when pressed.
 * Modified TableOfContents dilaog to use a colon as additional separator of
 * label and entries content.
 *
 * Revision 1.15  1999/01/05 11:16:55  kulow
 * s/klocale->translate/i18n
 * updated po files with the newly i18ned file dialogs
 *
 * Revision 1.14  1998/06/21 12:19:49  ettrich
 * Matthias: bugfixes
 *
 * Revision 1.13  1998/06/20 21:40:41  ettrich
 * Matthias: fixed the problem with goto and appendix sections
 *
 * Revision 1.12  1998/04/12 18:32:34  ettrich
 * Matthias: a few cleanups, bugfixes in the document structure editor,
 *    uses KTMainWindow instead of KTopLevelWidget now
 *
 * Revision 1.11  1998/03/28 22:15:28  kalle
 * - Pagestyles in DocumentLayoutDialog are dynamically updated when changing text classes
 * - InsertCitationDialog starts up with correct bibliographic reference
 * - Added Goto button to DocumentStructureDialog
 *
 * Revision 1.10  1998/03/11 13:31:14  ettrich
 * Matthias:
 *
 *      - Document Structure Editor
 *
 *      - cut and paste via qclipboard (only from outside to lyx)
 *
 * Revision 1.9  1998/03/03 22:36:28  kulow
 * removed all those ->show(), they make me crazy
 *
 * Revision 1.8  1998/03/02 22:48:45  kulow
 * fixed FMM
 *
 * Revision 1.7  1998/03/02 21:13:55  kalle
 * Kalle: Started to make dialogs more Qt-like
 *
 * Revision 1.6  1998/03/02 19:34:56  kulow
 * geometry managment for TableOfContentsDialog. This one looks nice, I think
 *
 * Revision 1.5  1998/03/01 22:56:53  kulow
 * some cosmetic changes on the strings
 *
 * Revision 1.4  1998/02/08 02:08:45  ettrich
 * Matthias: small fixes
 *
 * Revision 1.3  1998/02/08 01:33:11  ettrich
 * Matthias:
 *            - patches for 0.12.0
 *            - few LyX kernel bugfixes
 *            - other bugfixes
 *            - table dialog halfway done (stay tuned!)
 *            - "zoom" now a property of the LyXView. Useless but
 *              great visual effect :-)
 *
 * Revision 1.2  1998/01/24 22:20:57  kalle
 * toc dialog now supports up/down/left/right (currently works in
 * the dialog only, not in reality...)
 *
 * Revision 1.1  1998/01/24 21:41:18  kalle
 * TableOfContents dialog works with updating and selectinTableOfContents dialog works with updating and selectingg
 *
 */
