/* This file is part of KLyX, the High Level Word Processor
 *
 * Copyright (C) 1997-1999 Matthias Ettrich and the KLyX Team
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <qdialog.h>

#include <kmenubar.h>
#include <ktmainwindow.h>

#include "klyx.h"
#include "KLyXMenu.h"
#include "KLyXToolBar.h"
#include "commandtags.h"
#include "LyXView.h"
#include "lyxfunc.h"
#include "lastfiles.h"
#include "bufferlist.h"
#include "lyx_gui_misc.h"
#include "minibuffer.h"
#include "lyxscreen.h"
#include "lyxrc.h"
#include "lyx_main.h"
#include "filetools.h"
#include "intl.h"
#include "FontLoader.h"
#include "ScreenOptionsTabDialog.h"
#include "ScreenFontsDialog.h"
#include "ScreenToolbarsDialog.h"


extern BufferList bufferlist;
extern BufferView* current_view;

// global toolbar settings
extern ScreenToolbarsDialogData global_toolbar_visibilities;

extern void MenuLayoutSave();
extern void ShowCredits();
extern void ShowCopyright();
extern void show_symbols_form(LyXFunc*);
extern void ProhibitInput();
extern void AllowInput();
extern void OpenStuff();
extern void ToggleFloat();
extern void AllFloats(char flag, char figmar);
extern void LaTeXOptions();
extern void SpellCheckerOptions();
extern void ShowCredits();
extern void ScreenApplyCB();

// these are nowhere needed but here, and since they should not be exported anyway, we might just as
// well define them here
#define LFUN_PSEUDO_EDITFLOATS_OPENCLOSE 20000
#define LFUN_PSEUDO_EDITFLOATS_OPENALLFOOTNOTES 20001
#define LFUN_PSEUDO_EDITFLOATS_CLOSEALLFOOTNOTES 20002
#define LFUN_PSEUDO_EDITFLOATS_OPENALLFIGURES 20003
#define LFUN_PSEUDO_EDITFLOATS_CLOSEALLFIGURES 20004
#define LFUN_PSEUDO_EDITTABLE_MULTICOLUMN 20010
#define LFUN_PSEUDO_EDITTABLE_LINETOP 20011
#define LFUN_PSEUDO_EDITTABLE_LINEBOTTOM 20012
#define LFUN_PSEUDO_EDITTABLE_LINELEFT 20013
#define LFUN_PSEUDO_EDITTABLE_LINERIGHT 20014
#define LFUN_PSEUDO_EDITTABLE_ALIGNLEFT 20015
#define LFUN_PSEUDO_EDITTABLE_ALIGNRIGHT 20016
#define LFUN_PSEUDO_EDITTABLE_ALIGNCENTER 20017
#define LFUN_PSEUDO_EDITTABLE_APPENDROW 20018
#define LFUN_PSEUDO_EDITTABLE_APPENDCOLUMN 20019
#define LFUN_PSEUDO_EDITTABLE_DELETEROW 20020
#define LFUN_PSEUDO_EDITTABLE_DELETECOLUMN 20021
#define LFUN_PSEUDO_EDITTABLE_DELETETABLE 20022
#define LFUN_PSEUDO_EDIT_PASTESELECTION_LINE 20030
#define LFUN_PSEUDO_EDIT_PASTESELECTION_PARA 20031
#define LFUN_PSEUDO_LAYOUT_SAVELAYOUT 20040
#define LFUN_PSEUDO_OPTIONS_SCREENOPTIONS 20060
#define LFUN_PSEUDO_OPTIONS_SPELLCHECKER 20061
#define LFUN_PSEUDO_OPTIONS_KEYBOARD 20062
#define LFUN_PSEUDO_OPTIONS_LATEX 20063
#define LFUN_PSEUDO_INSERTASCII_LINE 20070
#define LFUN_PSEUDO_INSERTASCII_PARAGRAPH 20071
#define LFUN_PSEUDO_INSERTFLOAT_FIGURE 20080
#define LFUN_PSEUDO_INSERTFLOAT_TABLE 20081
#define LFUN_PSEUDO_INSERTFLOAT_WIDEFIG 20082
#define LFUN_PSEUDO_INSERTFLOAT_WIDETAB 20083
#define LFUN_PSEUDO_INSERTFLOAT_ALGORITHM 20084
#define LFUN_PSEUDO_VIEW_TOGGLETOOLBAR 20090
#define LFUN_PSEUDO_VIEW_TOGGLEMATHTOOLBAR 20091


static void addToHelpMenu( QPopupMenu* helpmenu, KLyXMenuReceiver* that );


KLyXMenuReceiver::KLyXMenuReceiver( LyXView* view )
{
  bar = view->_k_tlw->menuBar();
  _view = view;
  QPopupMenu* m;
  /* File menu */
  m = new QPopupMenu;
  connect(m, SIGNAL(activated(int)), SLOT(m_fileActivated(int)));
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowFile()));
  // remove this line when menu handling code is changed
    bar->insertItem(i18n("&File"), m, i_file);
  m_file = m;

  m = new QPopupMenu;
  connect(m, SIGNAL(activated(int)), SLOT(m_fileActivated(int)));
  m_preview = m;

  m = new QPopupMenu;
  connect(m, SIGNAL(activated(int)), SLOT(m_exportActivated(int)));
  m_export = m;

  m = new QPopupMenu;
  connect(m, SIGNAL(activated(int)), SLOT(activated(int)));
  m_window = m;

  /* Edit menu */
  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_editActivated( int ) ) );
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowEdit()));
  // remove this line when menu handling code is changed
    bar->insertItem( i18n( "&Edit" ), m, i_edit );
  m_edit = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_editFloatActivated( int ) ) );
  m_editfloat = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_editTableActivated( int ) ) );
  m_edittable = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_vcActivated( int ) ) );
  m_vc = m;

  /* Layout menu */

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_layoutActivated( int ) ) );
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowLayout()));
  // remove this line when menu handling code is changed
    bar->insertItem( i18n( "&Layout" ), m, i_layout );
  m_layout = m;

  /* Insert menu */

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_insertActivated( int ) ) );
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowInsert()));
  // remove this line when menu handling code is changed
    bar->insertItem( i18n( "&Insert" ), m, i_insert );
  m_insert = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_insertASCIIActivated( int ) ) );
  m_insertascii = m;

  m = new QPopupMenu;
  connect( m, SIGNAL(activated( int ) ), SLOT( m_insertTableActivated( int ) ) );
  m_inserttable = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_insertFloatActivated( int ) ) );
  m_insertfloat = m;

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_insertSpecialActivated( int ) ) );
  m_insertspecial = m;

  /* Options menu */

  m = new QPopupMenu;
  connect( m, SIGNAL( activated( int ) ), SLOT( m_optionsActivated( int ) ) );
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowOptions()));
  // remove this line when menu handling code is changed
    bar->insertItem( i18n( "&Options" ), m, i_options );
  m_options = m;

  /* Buffer menu */

  m = new QPopupMenu;
  connect(m, SIGNAL(activated(int)), SLOT(m_bufferActivated(int)));
  connect(m, SIGNAL(aboutToShow()), SLOT(aboutToShowBuffer()));
  // remove this line when menu handling code is changed
    bar->insertItem(i18n("&Documents"), m, i_buffer);
  m_buffer = m;

  /* Help menu */
  // remove this line when menu handling code is changed
  bar->insertSeparator();
  QString aboutKLyX( "KLyX "VERSION"\n"
		     "Copyright (C) 1997-1999 Matthias Ettrich, Kalle Dalheimer" );
  aboutKLyX += i18n( ", and the KLyX team\n\n"
		     "based on LyX-0.10 - 0.12\n" );
  aboutKLyX += "Copyright (C) 1995 Matthias Ettrich\n"
      "Copyright (C) 1995-1997 LyX Team\n\n";
  aboutKLyX += i18n( "This program is free software; you can redistribute it and/or modify it under\n"
		     "the terms of the GNU General Public License as published by the Free Software\n"
		     "Foundation; either version 2 of the License, or (at your option) any later version.\n\n"
		     "KLyX is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;\n"
		     "without even the implied warranty of MERCHANTABILITY or FITNESS FOR A\n"
		     "PARTICULAR PURPOSE. See the GNU General Public License for more details.\n"
		     "You should have received a copy of the GNU General Public License along with this\n"
		     "program; if not, write to the" );
  aboutKLyX += "Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.";
  QPopupMenu *help = kapp->getHelpMenu( true, aboutKLyX );
  addToHelpMenu( help, this );
  bar->insertItem(i18n("&Help"), help);

    hidden = false;
    hideMenus();
}



KLyXMenuReceiver::~KLyXMenuReceiver()
{
}



static void addToHelpMenu( QPopupMenu* helpmenu, KLyXMenuReceiver* that )
{
    helpmenu->insertItem( i18n( "&Credits" ), that, SLOT( helpCredits() ) );
    helpmenu->insertSeparator();
    helpmenu->insertItem( i18n( "&Introduction" ), that, SLOT( helpIntroduction() ) );
    helpmenu->insertItem( i18n( "&Tutorial" ), that, SLOT( helpTutorial() ) );
    helpmenu->insertItem( i18n( "&How Do I" ), that, SLOT( helpHowDoI() ) );
    helpmenu->insertItem( i18n( "&User's Guide" ), that, SLOT( helpUsersGuide() ) );
    helpmenu->insertItem( i18n( "&Advanced Editing" ), that, SLOT( helpAdvancedEditing() ) );
    helpmenu->insertItem( i18n( "Cu&stomization" ), that, SLOT( helpCustomization() ) );
    helpmenu->insertItem( i18n( "&Reference Manual" ), that, SLOT( helpReferenceManual() ) );
    helpmenu->insertItem( i18n( "&Known Bugs" ), that, SLOT( helpKnownBugs() ) );
    helpmenu->insertItem( i18n( "&LaTeX Configuration" ), that, SLOT( helpLaTeXConfiguration() ) );
}



void KLyXMenuReceiver::showMenus()
{
    if( !hidden )
	return;
    hidden = false;

    bar->setItemEnabled(i_edit, true);
    bar->setItemEnabled(i_layout, true);
    bar->setItemEnabled(i_insert, true);
    bar->setItemEnabled(i_buffer, true);
    bar->repaint();

  return;

}



void KLyXMenuReceiver::hideMenus()
{
    if( hidden )
	return;
    hidden = true;

    bar->setItemEnabled(i_edit, false);
    bar->setItemEnabled(i_layout, false);
    bar->setItemEnabled(i_insert, false);
    bar->setItemEnabled(i_buffer, false);
    bar->repaint();

  return;
}

void KLyXMenuReceiver::activated( int f )
{
    LyXFunc *tmpfunc=_view->getLyXFunc();
    tmpfunc->Dispatch(f);
}



void KLyXMenuReceiver::m_fileActivated( int i )
{
    if (i>100000){ // lastfiles
	LastFiles_Iter liter( *lastfiles ); // :-)
	_view->currentView()->setBuffer(bufferlist.loadLyXFile(liter[i-100001])); // ok one more
    } else
	activated(i);
}



void KLyXMenuReceiver::m_exportActivated( int i )
{
    LyXFunc *tmpfunc=_view->getLyXFunc();
    switch (i){
    case 0:
	if ( ! _view->currentBuffer()->isLinuxDoc() )
	    tmpfunc->Dispatch( LFUN_EXPORT, "latex" );
	else
	    tmpfunc->Dispatch( LFUN_EXPORT, "linuxdoc" );
	break;
    case 1:
	tmpfunc->Dispatch( LFUN_EXPORT, "dvi" );
	break;
    case 2:
	tmpfunc->Dispatch( LFUN_EXPORT, "postscript" );
	break;
    case 3:
	tmpfunc->Dispatch( LFUN_EXPORT, "ascii" );
	break;
    case 4:
	tmpfunc->Dispatch( LFUN_EXPORT, "custom" );
	break;
    }
}



void KLyXMenuReceiver::m_editActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    switch( i )
	{
	case LFUN_PSEUDO_EDIT_PASTESELECTION_LINE:
	    tmpfunc->Dispatch( LFUN_PASTESELECTION, "line" );
	    break;
	case LFUN_PSEUDO_EDIT_PASTESELECTION_PARA:
	    tmpfunc->Dispatch( LFUN_PASTESELECTION, "paragraph" );
	    break;
	default:
	    tmpfunc->Dispatch( i );
	}
}

void KLyXMenuReceiver::m_editTableActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    if( i == LFUN_TABLE )
	tmpfunc->Dispatch( LFUN_TABLE );
    else
	{
	    Buffer* tmpbuffer = _view->currentBuffer();
	    if( _view->currentView()->available() ) {
		_view->currentView()->getScreen()->HideCursor();
		tmpbuffer->updateFull(false);
		switch( i )
		    {
		    case LFUN_PSEUDO_EDITTABLE_MULTICOLUMN:
			tmpbuffer->text->TableFeatures( 12 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_LINETOP:
			tmpbuffer->text->TableFeatures( 4 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_LINEBOTTOM:
			tmpbuffer->text->TableFeatures( 5 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_LINELEFT:
			tmpbuffer->text->TableFeatures( 6 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_LINERIGHT:
			tmpbuffer->text->TableFeatures( 7 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_ALIGNLEFT:
			tmpbuffer->text->TableFeatures( 8 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_ALIGNRIGHT:
			tmpbuffer->text->TableFeatures( 9 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_ALIGNCENTER:
			tmpbuffer->text->TableFeatures( 10 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_APPENDROW:
			tmpbuffer->text->TableFeatures( 0 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_APPENDCOLUMN:
			tmpbuffer->text->TableFeatures( 1 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_DELETEROW:
			tmpbuffer->text->TableFeatures( 2 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_DELETECOLUMN:
			tmpbuffer->text->TableFeatures( 3 );
			break;
		    case LFUN_PSEUDO_EDITTABLE_DELETETABLE:
			tmpbuffer->text->TableFeatures( 11 );
			break;
		    }
		tmpbuffer->updateFull();
	    }
	}
}

void KLyXMenuReceiver::m_editFloatActivated( int i ) {
    LyXFunc *tmpfunc=_view->getLyXFunc();

    if( i == LFUN_MELT || i == LFUN_REMOVEERRORS )
	tmpfunc->Dispatch( i );
    else
	switch( i )
	    {
	    case LFUN_PSEUDO_EDITFLOATS_OPENCLOSE:
		ToggleFloat();
		break;
	    case LFUN_PSEUDO_EDITFLOATS_OPENALLFOOTNOTES:
		AllFloats( 1, 0 );
		break;
	    case LFUN_PSEUDO_EDITFLOATS_CLOSEALLFOOTNOTES:
		AllFloats( 0, 0 );
		break;
	    case LFUN_PSEUDO_EDITFLOATS_OPENALLFIGURES:
		AllFloats( 1, 1 );
		break;
	    case LFUN_PSEUDO_EDITFLOATS_CLOSEALLFIGURES:
		AllFloats( 0, 1 );
		break;
	    }
}

void KLyXMenuReceiver::m_vcActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    tmpfunc->Dispatch( i );
}


void KLyXMenuReceiver::m_layoutActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    if( i == LFUN_PSEUDO_LAYOUT_SAVELAYOUT )
	MenuLayoutSave();
    else
	tmpfunc->Dispatch( i );
}

void KLyXMenuReceiver::m_insertActivated( int i ){
    _view->getLyXFunc()->Dispatch( i );
}

void KLyXMenuReceiver::m_insertASCIIActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    switch( i )
	{
	case LFUN_PSEUDO_INSERTASCII_LINE:
	    tmpfunc->Dispatch( LFUN_FILE_INSERT_ASCII, "line" );
	    break;
	case LFUN_PSEUDO_INSERTASCII_PARAGRAPH:
	    tmpfunc->Dispatch( LFUN_FILE_INSERT_ASCII, "paragraph" );
	    break;
	}
}

void KLyXMenuReceiver::m_insertFloatActivated( int i ){
    LyXFunc *tmpfunc=_view->getLyXFunc();

    switch( i )
	{
	case LFUN_PSEUDO_INSERTFLOAT_FIGURE:
	    tmpfunc->Dispatch( LFUN_INSERTFOOTNOTE, "figure" );
	    break;
	case LFUN_PSEUDO_INSERTFLOAT_TABLE:
	    tmpfunc->Dispatch( LFUN_INSERTFOOTNOTE, "table" );
	    break;
	case LFUN_PSEUDO_INSERTFLOAT_WIDEFIG:
	    tmpfunc->Dispatch( LFUN_INSERTFOOTNOTE, "wide-fig" );
	    break;
	case LFUN_PSEUDO_INSERTFLOAT_WIDETAB:
	    tmpfunc->Dispatch( LFUN_INSERTFOOTNOTE, "wide-tab" );
	    break;
	case LFUN_PSEUDO_INSERTFLOAT_ALGORITHM:
	    tmpfunc->Dispatch( LFUN_INSERTFOOTNOTE, "algorithm" );
	    break;
	}
}

void KLyXMenuReceiver::m_insertTableActivated( int i ){
    _view->getLyXFunc()->Dispatch( i );
}

void KLyXMenuReceiver::m_insertSpecialActivated( int i ){
    _view->getLyXFunc()->Dispatch( i );
}

void KLyXMenuReceiver::m_optionsActivated( int i ){
    switch( i )
	{
	case LFUN_PSEUDO_OPTIONS_SCREENOPTIONS:
	    ScreenOptions();
	    break;
	case LFUN_PSEUDO_OPTIONS_SPELLCHECKER:
	    SpellCheckerOptions();
	    break;
	case LFUN_PSEUDO_OPTIONS_KEYBOARD:
	    _view->getIntl()->MenuKeymap();
	    break;
	case LFUN_PSEUDO_OPTIONS_LATEX:
	    LaTeXOptions();
	    break;
	default:
	    _view->getLyXFunc()->Dispatch( i );
	}
}

void KLyXMenuReceiver::m_bufferActivated(int i){
    _view->currentView()->switchBuffer(bufferlist.getBuffer(i+1));
}


void KLyXMenuReceiver::createTableMenu(QPopupMenu* m_edittable){
    Buffer* tmpbuffer = _view->currentBuffer();

    m_edittable->clear();
		
    if( _view->currentView()->available() &&
	tmpbuffer->text->cursor.par->table) {

	m_edittable->insertItem( i18n( "&Multicolumn" ), LFUN_PSEUDO_EDITTABLE_MULTICOLUMN );
	m_edittable->setCheckable( true ); // is it really not possible to set this for singular items?
	if( tmpbuffer->text->cursor.par->table->
	    IsMultiColumn( tmpbuffer->text->
			   NumberOfCell( tmpbuffer->text->cursor.par,
					 tmpbuffer->text->cursor.pos ) ) )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_MULTICOLUMN, true );
	m_edittable->insertSeparator();
		
	m_edittable->insertItem( i18n( "Line &Top" ), LFUN_PSEUDO_EDITTABLE_LINETOP );
	if( tmpbuffer->text->cursor.par->table->
	    TopLine( tmpbuffer->text->
		     NumberOfCell( tmpbuffer->text->cursor.par,
				   tmpbuffer->text->cursor.pos ) ) )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_LINETOP, true );
		
	m_edittable->insertItem( i18n( "Line &Bottom" ), LFUN_PSEUDO_EDITTABLE_LINEBOTTOM );
	if( tmpbuffer->text->cursor.par->table->
	    BottomLine( tmpbuffer->text->
			NumberOfCell( tmpbuffer->text->cursor.par,
				      tmpbuffer->text->cursor.pos ) ) )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_LINEBOTTOM, true );
		
	m_edittable->insertItem( i18n( "Line &Left" ), LFUN_PSEUDO_EDITTABLE_LINELEFT );
	if( tmpbuffer->text->cursor.par->table->
	    LeftLine( tmpbuffer->text->
		      NumberOfCell( tmpbuffer->text->cursor.par,
				    tmpbuffer->text->cursor.pos ) ) )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_LINELEFT, true );
		
	m_edittable->insertItem( i18n( "Line &Right" ), LFUN_PSEUDO_EDITTABLE_LINERIGHT );
	if( tmpbuffer->text->cursor.par->table->
	    RightLine( tmpbuffer->text->
		       NumberOfCell( tmpbuffer->text->cursor.par,
				     tmpbuffer->text->cursor.pos ) ) )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_LINERIGHT, true );
	m_edittable->insertSeparator();
		
	int align = tmpbuffer->text->cursor.par->
	    table->GetAlignment( tmpbuffer->text->
				 NumberOfCell( tmpbuffer->text->cursor.par,
					       tmpbuffer->text->cursor.pos ) );
	m_edittable->insertItem( i18n( "Align L&eft" ),
				 LFUN_PSEUDO_EDITTABLE_ALIGNLEFT );
	if( align == LYX_ALIGN_LEFT )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_ALIGNLEFT, true );
	m_edittable->insertItem( i18n( "Align R&ight" ),
				 LFUN_PSEUDO_EDITTABLE_ALIGNRIGHT );
	if( align == LYX_ALIGN_RIGHT )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_ALIGNRIGHT, true );
	m_edittable->insertItem( i18n( "Align &Center" ),
				 LFUN_PSEUDO_EDITTABLE_ALIGNCENTER );
	if( align == LYX_ALIGN_CENTER )
	    m_edittable->setItemChecked( LFUN_PSEUDO_EDITTABLE_ALIGNCENTER, true );
	m_edittable->insertSeparator();
		
	m_edittable->insertItem( i18n( "Append R&ow" ),
				 LFUN_PSEUDO_EDITTABLE_APPENDROW );
	m_edittable->insertItem( i18n( "Append Col&umn" ),
				 LFUN_PSEUDO_EDITTABLE_APPENDCOLUMN );
	m_edittable->insertSeparator();
	m_edittable->insertItem( i18n( "Delete Ro&w" ),
				 LFUN_PSEUDO_EDITTABLE_DELETEROW );
	m_edittable->insertItem( i18n( "Delete Colum&n" ),
				 LFUN_PSEUDO_EDITTABLE_DELETECOLUMN );
	m_edittable->insertSeparator();
	m_edittable->insertItem( i18n( "&Delete Table" ),
				 LFUN_PSEUDO_EDITTABLE_DELETETABLE );
    }
    else
	m_edittable->insertItem( i18n( "&Insert table" ),
				 LFUN_TABLE );
}


void KLyXMenuReceiver::createEditMenu(QPopupMenu* m_edit){
  Buffer* tmpbuffer = _view->currentBuffer();
  if( _view->currentView()->available()){
    m_edit->clear();
    m_edit->insertItem( i18n( "&Undo" ), LFUN_UNDO );
    if (tmpbuffer->undostack.Top() == 0)
      m_edit->setItemEnabled(LFUN_UNDO, False);
    m_edit->insertItem( i18n( "&Redo" ), LFUN_REDO );
    if (tmpbuffer->redostack.Top() == 0)
      m_edit->setItemEnabled(LFUN_REDO, False);
    m_edit->insertSeparator();
    m_edit->insertItem( i18n( "&Cut" ), LFUN_CUT );
    if (!tmpbuffer->hasSelection())
      m_edit->setItemEnabled(LFUN_CUT, False);
    m_edit->insertItem( i18n( "C&opy" ), LFUN_COPY );
    if (!tmpbuffer->hasSelection())
      m_edit->setItemEnabled(LFUN_COPY, False);
    m_edit->insertItem( i18n( "&Paste" ), LFUN_PASTE );
    if (!tmpbuffer->hasSomethingToPaste())
      m_edit->setItemEnabled(LFUN_PASTE, False);
    m_edit->insertSeparator();
    m_edit->insertItem( i18n( "&Find & Replace..." ), LFUN_MENUSEARCH );
    m_edit->insertItem( i18n( "&Go to Error" ), LFUN_GOTOERROR );
    m_edit->insertItem( i18n( "&Go to Note" ), LFUN_GOTONOTE );
  }
}

void KLyXMenuReceiver::createMenu(int m){
  Buffer* tmpbuffer = _view->currentBuffer();
  bool LinuxDoc = false;
  if( tmpbuffer )
	LinuxDoc = tmpbuffer->isLinuxDoc();

    switch (m){
    case i_file: {

	m_file->clear();
	m_file->insertItem(i18n("&New..."), LFUN_MENUNEW);
	m_file->insertItem(i18n("New from &template..."), LFUN_MENUNEWTMPLT);
	m_file->insertItem(i18n("&Open..."), LFUN_MENUOPEN);

	if (tmpbuffer){


	    m_file->insertSeparator();
	    m_file->insertItem(i18n("&Close"), LFUN_CLOSEBUFFER);
	    m_file->insertItem(i18n("&Save"), LFUN_MENUWRITE);
	    m_file->insertItem(i18n("Save &as..."), LFUN_MENUWRITEAS);
	    m_file->insertItem(i18n("&Revert to saved"), LFUN_MENURELOAD);
	    m_file->insertSeparator();
	    m_file->insertItem(i18n("Pre&view"), m_preview);
	    m_file->insertItem(i18n("&Print..."), LFUN_MENUPRINT);
	    m_file->insertItem(i18n("&Fax..."), LFUN_FAX);

	    m_preview->clear();
	    m_preview->insertItem(i18n("View &DVI"), LFUN_PREVIEW);
	    m_preview->insertItem(i18n("View &PostScript"), LFUN_PREVIEWPS);
	    m_preview->insertItem(i18n("&Update DVI"), LFUN_RUNLATEX);
	    m_preview->insertItem(i18n("Update P&ostscript"), LFUN_RUNDVIPS);

	    // These commands are disabled when the corresponding programs
	    // are not installed. I simply grey them out, since I do not
	    // want to change their number (JMarc)
	    bool hasLaTeX = lyxrc->latex_command != "none";

	    if (!hasLaTeX || lyxrc->view_dvi_command == "none")
		m_preview->setItemEnabled(LFUN_PREVIEW, False);

	    if (!hasLaTeX || lyxrc->view_ps_command == "none")
		m_preview->setItemEnabled(LFUN_PREVIEWPS, False);

	    if (!hasLaTeX) {
		m_preview->setItemEnabled(LFUN_RUNLATEX, False);
		m_preview->setItemEnabled(LFUN_RUNDVIPS, False);
	    }

	    if (!hasLaTeX || lyxrc->print_command == "none")
		m_file->setItemEnabled(LFUN_MENUPRINT, False);

	    if (!hasLaTeX || lyxrc->fax_command == "none")
		m_file->setItemEnabled(LFUN_FAX, False);

	    if (!hasLaTeX)
		m_file->setItemEnabled(LFUN_RUNLATEX, False);

	    m_export->clear();
	    if (!tmpbuffer->isLinuxDoc()){
		m_export->insertItem(i18n("as &LaTeX..."));
		m_export->insertItem(i18n("as &DVI..."));
		m_export->insertItem(i18n("as &PostScript..."));
		m_export->insertItem(i18n("as Ascii &Text..."));
		m_export->insertItem(i18n("Custo&m..."));
	    }else {
		m_export->insertItem(i18n("as &LinuxDoc..."));
		m_export->insertItem(i18n("as &DVI..."));
		m_export->insertItem(i18n("as &PostScript..."));
		m_export->insertItem(i18n("as Ascii &Text..."));
	    }

	    m_file->insertItem(i18n("&Export"), m_export);
	}

	m_file->insertSeparator();
	m_window->clear();

	m_window->insertItem(i18n("&New Window"), LFUN_WINDOW_NEW);
	m_window->insertItem(i18n("&Delete Window"), LFUN_WINDOW_DELETE);
	m_window->insertItem(i18n("&Split horizontal"), LFUN_WINDOW_SPLIT);	// { Rocher 981130
	m_window->insertItem(i18n("Split &vertical"), LFUN_WINDOW_SPLIT_VERTICAL);
	if (_view->isSplitted()) {
	    m_window->setItemEnabled( LFUN_WINDOW_SPLIT, false );
	    m_window->setItemEnabled( LFUN_WINDOW_SPLIT_VERTICAL, false );	// }
	}

	m_window->insertItem(i18n("&Un-Split"), LFUN_WINDOW_UNSPLIT);
	if (!_view->isSplitted())
	    m_window->setItemEnabled( LFUN_WINDOW_UNSPLIT, false );

	m_file->insertItem(i18n("&Window"), m_window);

	m_file->insertSeparator();
	m_file->insertItem(i18n("E&xit"), LFUN_QUIT);
	m_file->insertSeparator();

	// make the lastfiles menu
	LastFiles_Iter liter(*lastfiles); // :-)
	LString filname;
	int ii = 1;
	while( ii < 10 && !(filname = liter()).empty() ) {
	    LString tmp = "&";
	    LString relbuf = MakeDisplayPath(filname,30);
	    tmp += ii;
	    tmp += ". " + relbuf;
	    m_file->insertItem(tmp.c_str(), 100000+ii);
	    ii++;
	}
    }break;

    /* edit menu */
    case i_edit:
	{
	    if( !tmpbuffer )
		return;
	    createEditMenu(m_edit);
	    /* edit menu, submenu "Floats and Insets" */
	    m_editfloat->clear();
	    m_editfloat->insertItem( i18n( "&Open/Close" ), LFUN_PSEUDO_EDITFLOATS_OPENCLOSE );
	    m_editfloat->insertItem( i18n( "&Melt" ), LFUN_MELT );
	    m_editfloat->insertItem( i18n( "Open &All Footnotes/Margin Notes" ), LFUN_PSEUDO_EDITFLOATS_OPENALLFOOTNOTES );
	    m_editfloat->insertItem( i18n( "&Close All Footnotes/Margin Notes" ), LFUN_PSEUDO_EDITFLOATS_CLOSEALLFOOTNOTES );
	    m_editfloat->insertItem( i18n( "O&pen All Figures/Tables" ), LFUN_PSEUDO_EDITFLOATS_OPENALLFIGURES );
	    m_editfloat->insertItem( i18n( "C&lose All Figures/Tables" ), LFUN_PSEUDO_EDITFLOATS_CLOSEALLFIGURES );
	    m_editfloat->insertSeparator();
	    m_editfloat->insertItem( i18n( "&Remove All Error Boxes" ), LFUN_REMOVEERRORS );
	    m_edit->insertItem( i18n( "Flo&ats and Insets" ), m_editfloat );
	
	    /* edit menu, submenu "Table" */
	    createTableMenu(m_edittable);
	
	    m_edit->insertItem( i18n( "Ta&ble" ), m_edittable );
	
	    /* edit menu, continuation */
	    m_edit->insertItem( i18n( "&Spellchecker..." ), LFUN_SPELLCHECK );
	    m_edit->insertItem( i18n( "C&heck TeX" ), LFUN_RUNCHKTEX );
	    m_edit->insertItem( i18n( "&Document Structure..." ), LFUN_DOCUMENT_STRUCTURE );
	    m_edit->insertSeparator();
	
	    /* edit menu, VC submenu */
	    m_vc->clear();
	    if( tmpbuffer->lyxvc.inUse() ) {
		m_vc->insertItem( i18n( "&Register" ), LFUN_VC_REGISTER );
		m_vc->insertItem( i18n( "Check&In Changes" ), LFUN_VC_CHECKIN );
		m_vc->insertItem( i18n( "Check &Out for Edit" ), LFUN_VC_CHECKOUT );
		if( tmpbuffer->isReadonly() ) {
		    m_vc->setItemEnabled( LFUN_VC_CHECKIN, false );
		    m_vc->setItemEnabled( LFUN_VC_CHECKOUT, true );
		} else {
		    m_vc->setItemEnabled( LFUN_VC_CHECKIN, true );
		    m_vc->setItemEnabled( LFUN_VC_CHECKOUT, false );
		}
		m_vc->insertItem( i18n( "Revert to &last version" ), LFUN_VC_REVERT  );
		m_vc->insertItem( i18n( "&Undo last check in" ), LFUN_VC_UNDO  );
		m_vc->insertItem( i18n( "Show &History" ), LFUN_VC_HISTORY ) ;
	    }
	    else
		m_vc->insertItem( i18n( "&Register" ) , LFUN_VC_REGISTER );
	    m_edit->insertItem( i18n( "&Version Control" ), m_vc );
	    m_edit->insertSeparator();

	    /* edit menu, continuation */
	    m_edit->insertItem( i18n( "Vie&w LaTeX log file" ), LFUN_LATEX_LOG );
	    m_edit->insertSeparator();
	    m_edit->insertItem( i18n( "Paste Primary Selection as Lines" ),
				LFUN_PSEUDO_EDIT_PASTESELECTION_LINE );
	    m_edit->insertItem( i18n( "Paste Primary Selection as Paragraph" ),
				LFUN_PSEUDO_EDIT_PASTESELECTION_PARA );

	    // disable unavailable items
	    if( tmpbuffer->undostack.Top() == 0 )
		m_edit->setItemEnabled( LFUN_UNDO, false );
	
	    if( tmpbuffer->redostack.Top() == 0 )
		m_edit->setItemEnabled( LFUN_REDO, false );
	
	    if( lyxrc->chktex_command == "none" )
		m_edit->setItemEnabled( LFUN_RUNCHKTEX, false );

	    if( tmpbuffer->isReadonly() ) {
		m_edit->setItemEnabled( LFUN_UNDO, false );
		m_edit->setItemEnabled( LFUN_REDO, false );
		m_edit->setItemEnabled( LFUN_CUT, false );
		m_edit->setItemEnabled( LFUN_PASTE, false );
		m_edit->setItemEnabled( LFUN_PSEUDO_EDIT_PASTESELECTION_LINE, false );
		m_edit->setItemEnabled( LFUN_PSEUDO_EDIT_PASTESELECTION_PARA, false );
	    }

	    break;
	}

    case i_layout:
	{
	    if( !tmpbuffer )
		return;

	    /* layout menu */
	    m_layout->clear();
	    m_layout->insertItem( i18n( "&Character..." ), LFUN_LAYOUT_CHARACTER );
	    m_layout->insertItem( i18n( "&Paragraph..." ), LFUN_LAYOUT_PARAGRAPH );
	    m_layout->insertItem( i18n( "&Document..." ), LFUN_LAYOUT_DOCUMENT );
	    m_layout->insertItem( i18n( "P&aper..." ), LFUN_LAYOUT_PAPER );
	    m_layout->insertItem( i18n( "&Quotes..." ), LFUN_LAYOUT_QUOTES );
	    m_layout->insertSeparator();
	    m_layout->setCheckable( true );
	    m_layout->insertItem( i18n( "E&mphasize Style" ), LFUN_EMPH );
	    m_layout->insertItem( i18n( "&Noun Style" ), LFUN_NOUN );
	    m_layout->insertItem( i18n( "&Bold Style" ), LFUN_BOLD );
	    m_layout->insertItem( i18n( "&TeX Style" ), LFUN_TEX );
	    m_layout->insertSeparator();
	    m_layout->insertItem( i18n( "Start of Appendi&x" ), LFUN_APPENDIX);
	    m_layout->insertItem( i18n( "Change En&vironment Depth" ), LFUN_DEPTH_PLUS );
	    m_layout->insertItem( i18n( "&LaTeX Preamble..." ), LFUN_LAYOUT_PREAMBLE );
	    m_layout->insertItem( i18n( "&Save layout as default" ),
				  LFUN_PSEUDO_LAYOUT_SAVELAYOUT );

	    // Set values of checkboxes according to font
	    LyXFont font = tmpbuffer->text->real_current_font;
	    if( font.emph() == LyXFont::ON )
		m_layout->setItemChecked( LFUN_EMPH, true );
	    if( font.noun() == LyXFont::ON )
		m_layout->setItemChecked( LFUN_NOUN, true );
	    if( font.series() == LyXFont::BOLD_SERIES )
		m_layout->setItemChecked( LFUN_BOLD, true );
	    if( font.latex() == LyXFont::ON )
		m_layout->setItemChecked( LFUN_TEX, true );

	    // Set the value of the appenix checkbox
	    if (tmpbuffer->text->cursor.par->FirstPhysicalPar()->start_of_appendix)
		m_layout->setItemChecked(LFUN_APPENDIX, true);
	
	    // Grey out unavailable entries
	    if( tmpbuffer->isReadonly() ) {
		m_layout->setItemEnabled( LFUN_LAYOUT_CHARACTER, false );
		m_layout->setItemEnabled( LFUN_LAYOUT_QUOTES, false );
		m_layout->setItemEnabled( LFUN_DEPTH_PLUS, false );
	    }
	    break;
	}

    case i_insert:
	{
	    if( !tmpbuffer )
		return;

	    /* insert menu */
	    m_insert->clear();
	    m_insert->insertItem( i18n( "Fi&gure..." ), LFUN_FIGURE );
	    m_insert->insertItem( i18n( "Ta&ble..." ), LFUN_TABLE );
	    m_insert->insertSeparator();
	    m_insert->insertItem( i18n( "In&clude File..." ),
				  LFUN_CHILDINSERT );
	
	    /* insert menu, submenu Import ASCII File */
	    m_insertascii->clear();
	    m_insertascii->insertItem( i18n( "As &Lines" ),
				       LFUN_PSEUDO_INSERTASCII_LINE );
	    m_insertascii->insertItem( i18n( "As &Paragraphs" ),
				       LFUN_PSEUDO_INSERTASCII_PARAGRAPH );
	    m_insert->insertItem( i18n( "Insert &ASCII File" ),
				  m_insertascii );
						
	    /* insert menu, continuation */
	    m_insert->insertItem( i18n( "Insert Ly&X File..." ),
				  LFUN_FILE_INSERT );
	    m_insert->insertSeparator();
	    m_insert->insertItem( i18n( "&Footnote" ), LFUN_FOOTMELT );
	    m_insert->insertItem( i18n( "&Margin Note" ),
				  LFUN_MARGINMELT );

	    /* insert menu, submenu Floats */
	    m_insertfloat->clear();
	    m_insertfloat->insertItem( i18n( "&Figure Float" ),
				       LFUN_PSEUDO_INSERTFLOAT_FIGURE );
	    m_insertfloat->insertItem( i18n( "&Table Float" ),
				       LFUN_PSEUDO_INSERTFLOAT_TABLE );
	    m_insertfloat->insertItem( i18n( "&Wide Figure Float" ),
				       LFUN_PSEUDO_INSERTFLOAT_WIDEFIG );
	    m_insertfloat->insertItem( i18n( "W&ide Table Float" ),
				       LFUN_PSEUDO_INSERTFLOAT_WIDETAB );
	    m_insertfloat->insertItem( i18n( "&Algorithm Float" ),
				       LFUN_PSEUDO_INSERTFLOAT_ALGORITHM );
	    m_insert->insertItem( i18n( "Fl&oats" ), m_insertfloat );
	    m_insert->insertSeparator();

	    /* insert menu, submenu Lists & TOC */
	    m_inserttable->clear();
	    m_inserttable->insertItem( i18n( "Table of &Contents" ),
				       LFUN_TOC_INSERT );
	    m_inserttable->insertItem( i18n( "List of &Figures" ),
				       LFUN_LOF_INSERT );
	    m_inserttable->insertItem( i18n( "List of &Tables" ),
				       LFUN_LOT_INSERT );
	    m_inserttable->insertItem( i18n( "&Index List" ),
				       LFUN_INDEX_PRINT );
	    m_inserttable->insertItem( i18n( "&BibTeX Reference..." ),
				       LFUN_INSERT_BIBTEX );
	    m_inserttable->insertItem( i18n( "List of &Algorithms" ),
				       LFUN_LOA_INSERT );
	    m_insert->insertItem( i18n( "Lists & TOC" ),
				  m_inserttable );
	    m_insert->insertSeparator();

	    /* insert menu, submenu  Special Character */
	    m_insertspecial->clear();
	    m_insertspecial->insertItem( i18n( "&HFill" ),
					 LFUN_HFILL );
	    m_insertspecial->insertItem( i18n( "Hyphenation &Point" ),
					 LFUN_HYPHENATION );
	    m_insertspecial->insertItem( i18n( "Protected &Blank" ),
					 LFUN_PROTECTEDSPACE );
	    m_insertspecial->insertItem( i18n( "&Linebreak" ),
					 LFUN_BREAKLINE );
	    m_insertspecial->insertItem( i18n( "Ell&ipsis" ),
					 LFUN_LDOTS );
	    m_insertspecial->insertItem( i18n( "&End of sentence period" ),
					 LFUN_END_OF_SENTENCE );
	    m_insertspecial->insertItem( i18n( "Ordinary &Quote (\")" ),
					 LFUN_QUOTE );
	    m_insert->insertItem( i18n( "&Special Character" ),
				  m_insertspecial );
	    m_insert->insertSeparator();

	    /* insert menu, continuation */
	    m_insert->insertItem( i18n( "&Note" ), LFUN_INSERT_NOTE );
	    m_insert->insertItem( i18n( "&Label..." ),
				  LFUN_INSERT_LABEL );
	    m_insert->insertItem( i18n( "Cross-&Reference..." ),
				  LFUN_INSERT_REF );
	    m_insert->insertItem( i18n( "C&itation Reference..." ),
				  LFUN_INSERT_CITATION );
	    m_insert->insertItem( i18n( "In&dex Entry..." ),
				  LFUN_INDEX_INSERT );

	    if( LinuxDoc ) {
		/* for linuxdoc sgml */
		m_insert->insertItem( i18n( "&URL" ), LFUN_URL );
		m_insert->insertItem( i18n( "&HTML URL" ),
				      LFUN_HTMLURL );
	    }

	    if( tmpbuffer->isReadonly() ) {
		if( !tmpbuffer )
		    return;

		m_insert->setItemEnabled( LFUN_FIGURE, false );
		m_insert->setItemEnabled( LFUN_TABLE, false );
		m_insert->setItemEnabled( LFUN_CHILDINSERT, false );
		m_insertascii->setItemEnabled( LFUN_PSEUDO_INSERTASCII_LINE, false );
		m_insertascii->setItemEnabled( LFUN_PSEUDO_INSERTASCII_PARAGRAPH,
					       false );
		m_insert->setItemEnabled( LFUN_FILE_INSERT , false );
		m_insert->setItemEnabled( LFUN_FOOTMELT , false );
		m_insert->setItemEnabled( LFUN_MARGINMELT , false );
		m_insertfloat->setItemEnabled( LFUN_PSEUDO_INSERTFLOAT_FIGURE, false );
		m_insertfloat->setItemEnabled( LFUN_PSEUDO_INSERTFLOAT_TABLE,
					       false );
		m_insertfloat->setItemEnabled( LFUN_PSEUDO_INSERTFLOAT_WIDETAB,
					       false );
		m_insertfloat->setItemEnabled( LFUN_PSEUDO_INSERTFLOAT_WIDEFIG,
					       false );
		m_insertfloat->setItemEnabled( LFUN_PSEUDO_INSERTFLOAT_ALGORITHM,
					       false );
		m_inserttable->setItemEnabled( LFUN_TOC_INSERT, false );
		m_inserttable->setItemEnabled( LFUN_LOF_INSERT, false );
		m_inserttable->setItemEnabled( LFUN_LOA_INSERT, false );
		m_inserttable->setItemEnabled( LFUN_LOT_INSERT, false );
		m_inserttable->setItemEnabled( LFUN_INDEX_PRINT, false );
		m_inserttable->setItemEnabled( LFUN_INSERT_BIBTEX, false );
		m_insertspecial->setItemEnabled( LFUN_HFILL, false );
		m_insertspecial->setItemEnabled( LFUN_HYPHENATION, false );
		m_insertspecial->setItemEnabled( LFUN_PROTECTEDSPACE, false );
		m_insertspecial->setItemEnabled( LFUN_BREAKLINE, false );
		m_insertspecial->setItemEnabled( LFUN_LDOTS, false );
		m_insertspecial->setItemEnabled( LFUN_END_OF_SENTENCE, false );
		m_insertspecial->setItemEnabled( LFUN_QUOTE, false );
		m_insert->setItemEnabled( LFUN_INSERT_NOTE, false );
		m_insert->setItemEnabled( LFUN_INSERT_LABEL, false );
		m_insert->setItemEnabled( LFUN_INSERT_REF, false );
		m_insert->setItemEnabled( LFUN_INSERT_CITATION, false );
		m_insert->setItemEnabled( LFUN_INDEX_INSERT, false );
		if( LinuxDoc ) {
		    m_insert->setItemEnabled( LFUN_URL, false );
		    m_insert->setItemEnabled( LFUN_HTMLURL, false );
		}
	    }
							
	    break;
	}

    case i_options:
	{
	    m_options->clear();
	    m_options->insertItem( i18n( "Screen &Options..." ),
				   LFUN_PSEUDO_OPTIONS_SCREENOPTIONS );
	    m_options->insertItem( i18n( "&Spellchecker Options..." ),
				   LFUN_PSEUDO_OPTIONS_SPELLCHECKER );
	    m_options->insertItem( i18n( "&Keyboard..." ),
				   LFUN_PSEUDO_OPTIONS_KEYBOARD );
	    m_options->insertItem( i18n( "&LaTeX..." ),
				   LFUN_PSEUDO_OPTIONS_LATEX );
	    m_options->insertSeparator();
	    m_options->insertItem( i18n( "&Reconfigure" ),
				   LFUN_RECONFIGURE );
	    break;
	}

    case i_buffer:
	{
	    m_buffer->clear();
	    bufferlist.makePup(m_buffer);
	    break;
	}
    }
}


void KLyXMenuReceiver::aboutToShowFile() {createMenu( i_file);}
void KLyXMenuReceiver::aboutToShowEdit() {createMenu( i_edit);}
void KLyXMenuReceiver::aboutToShowLayout() {createMenu( i_layout);}
void KLyXMenuReceiver::aboutToShowInsert() {createMenu( i_insert);}
void KLyXMenuReceiver::aboutToShowOptions() {createMenu( i_options);}
void KLyXMenuReceiver::aboutToShowBuffer() {createMenu( i_buffer);}

void KLyXMenuReceiver::ScreenOptions()
{
    if( !klyxdialogs->screenoptions )
	{
	    klyxdialogs->screenoptions = new ScreenOptionsTabDialog( k_tlw );
	}
    KConfig* config = kapp->getConfig();
    QString oldgroup = config->group();
    config->setGroup( "ScreenOptions" );

    // this is not very nice....
    ScreenOptionsTabDialogData* dlg_data = klyxdialogs->screenoptions->data();

    dlg_data->fontsdata->roman_font = config->readEntry( "RomanFontName",
							 "-*-utopia" );
    dlg_data->fontsdata->sans_serif_font = config->readEntry( "SansFontName",
							      "-*-helvetica" );
    dlg_data->fontsdata->typewriter_font = config->readEntry( "TypewriterFontName",
							      "-*-courier" );
    dlg_data->fontsdata->font_norm = config->readEntry( "FontNorm",
							DEFCHSET );
    dlg_data->fontsdata->font_zoom.sprintf( "%d",
					    config->readNumEntry( "FontZoom",150 ) );

    // Matthias: zoom is now a property of the single LyXView
    if (_view && _view->getFontloader()){
	dlg_data->fontsdata->font_zoom.sprintf("%d",
					       _view->getFontloader()->getZoom());
    }
    *(dlg_data->toolbarsdata) = global_toolbar_visibilities;

    klyxdialogs->screenoptions->setData( dlg_data );

    config->setGroup( oldgroup );

    ScreenOptionsTabDialog* savedialog = klyxdialogs->screenoptions;
    int ret = klyxdialogs->screenoptions->exec();
    klyxdialogs->screenoptions = savedialog;
    if( ret == QDialog::Accepted )
	ScreenApplyCB();
}


void KLyXMenuReceiver::helpCredits()
{
    ShowCredits();
}


void KLyXMenuReceiver::helpIntroduction()
{
    menuDocu( "Intro" );
}


void KLyXMenuReceiver::helpTutorial()
{
    menuDocu( "Tutorial" );
}


void KLyXMenuReceiver::helpHowDoI()
{
    menuDocu( "HowDoI-" );
}


void KLyXMenuReceiver::helpUsersGuide()
{
    menuDocu( "UserGuide" );
}


void KLyXMenuReceiver::helpAdvancedEditing()
{
    menuDocu( "AdvancedEdit" );
}


void KLyXMenuReceiver::helpCustomization()
{
    menuDocu( "Customization" );
}


void KLyXMenuReceiver::helpReferenceManual()
{
    menuDocu( "Reference" );
}


void KLyXMenuReceiver::helpKnownBugs()
{
    menuDocu( "BUGS" );
}


void KLyXMenuReceiver::helpLaTeXConfiguration()
{
    menuDocu( "LaTeXConfig" );
}


void KLyXMenuReceiver::menuDocu( const char* docname )
{
    LString fname = LibFileSearch( "doc",docname,"lyx" );
    _view->getMiniBuffer()->Set( i18n( "Opening help file" ),
				 MakeDisplayPath( fname ), "..." );
    _view->currentView()->setBuffer( bufferlist.loadLyXFile( fname, false ) );

    if( docname == "Reference" )
	_view->getLyXFunc()->Dispatch( LFUN_TOCVIEW );
}



// Local Variables:
// mode: C++
// c-file-style: "Stroustrup"
// End:
