/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qfile.h>
#include <qtextstream.h>
#include <qpopupmenu.h>
#include <qstringlist.h>
#include <qstring.h>

#include <kapp.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kmenubar.h>

#include "kbiblioshell.h"
#include "kbibliomainview.h"
#include "kbibliodoc.h"
#include "prefdialog.h"
#include "finddialog.h"
#include "savetobibtex.h"


KBiblioShell::KBiblioShell( KBiblioDoc *doc, const QString &fileName, const char *name )
{

    mDocument = doc;
    mFileName = fileName;
    mMainView = new KBiblioMainView( this, this );
    QPopupMenu *file = new QPopupMenu( this );
    file->insertItem( "&New", this, SLOT( slotNewDoc() ), CTRL+Key_N );
    file->insertItem( "&Open",  this, SLOT( load() ), CTRL+Key_O );
    file->insertSeparator();
    file->insertItem( "&Save", this, SLOT( save() ), CTRL+Key_S );
    file->insertItem( "Save &as", this, SLOT( saveAs() ) );
    file->insertItem( "Save to &Bibtex", this, SLOT( saveToBibtex() ) );
    file->insertSeparator();
    file->insertItem( "&Quit",  this, SLOT(quit()), CTRL+Key_Q );

    QPopupMenu *edit = new QPopupMenu( this );
    edit->insertItem( "&New Entry", mMainView, SLOT( newEntry() ) );
    edit->insertItem( "&Edit Entry", mMainView, SLOT( editSelected() ) );
    edit->insertItem( "&Copy Entry", mMainView, SLOT( copySelected() ) );
    edit->insertItem( "&Delete Entry", mMainView, SLOT( deleteSelected() ) );
    edit->insertSeparator();
    edit->insertItem( "&Find", this, SLOT( find() ), CTRL+Key_F );

    QPopupMenu *options = new QPopupMenu( this );
    options->insertItem( "&Preferences", this, SLOT( preferences() ) );


    QPopupMenu *help = new QPopupMenu( this );
    help->insertItem( "About &kbiblio", this, SLOT( about() ) );

    menuBar()->insertItem( "&File", file );
    menuBar()->insertItem( "&Edit", edit );
    menuBar()->insertItem( "&Options", options );
    menuBar()->insertItem( "&Help", help );


    setView( mMainView );
    resize( 450, 600 );
}


KBiblioShell::~KBiblioShell()
{
}


QStringList
KBiblioShell::editableFields( const QString &entryName ) const
{
    BibEntryFieldType *fieldType;

    QStringList fieldTypes = mDocument->fieldTypesNames( entryName );
    QStringList editableFields;
    QStringList::Iterator it = fieldTypes.begin();

    for ( ; it != fieldTypes.end(); ++it ) {
	fieldType = mDocument->lookupEntryFieldType( (*it) );
	if ( fieldType != 0 )
	    if ( fieldType->isEditable() )
		editableFields.append( *it );
    }
    return editableFields;
}


QStringList
KBiblioShell::viewableFields( const QString &entryName ) const
{
    return mDocument->fieldTypesNames( entryName );
}


bool
KBiblioShell::fieldIsEditable( const QString &fieldName )
{
    // later on, this information should be stored in
    // the shell instead of in the  document!

    BibEntryFieldType *fieldType = mDocument->lookupEntryFieldType( fieldName );
    if ( fieldType != 0 )
	return fieldType->isEditable();
    else return false;
}


bool
KBiblioShell::fieldIsViewable( const QString &fieldName )
{
    return true;
}


KBiblioDoc*
KBiblioShell::document() const
{
    return mDocument;
}


KBiblioMainView*
KBiblioShell::mainView() const
{
    return mMainView;
}


int
KBiblioShell::marginHint() const
{
    return 12; //TODO: get rid of absolute pixel values
}


int
KBiblioShell::spacingHint() const
{
    return 8; //TODO: get rid of absolute pixel values
}


QString
KBiblioShell::entryDescription( const QString &entryName ) const
{
    // later on, this information should be stored in
    // the shell instead of in the  document!

    BibEntryType *entryType = mDocument->lookupEntryType( entryName );

    if ( entryType != 0 )
	return entryType->description();
    else return entryName;
}


QStringList
KBiblioShell::entryDescriptionList( const QStringList &entryList ) const
{
    QStringList returnList;

    QStringList::ConstIterator it = entryList.begin();
    for ( ; it != entryList.end(); ++it ) {
	returnList.append( entryDescription( *it ) );
    }
    return returnList;
}


QString
KBiblioShell::fieldDescription( const QString &fieldName ) const
{
    // later on, this information should be stored in
    // the shell instead of in the  document!

    BibEntryFieldType *fieldType = mDocument->lookupEntryFieldType( fieldName );
    if ( fieldType != 0 )
	return fieldType->description();
    else return fieldName;
}


QStringList
KBiblioShell::fieldDescriptionList( const QStringList &fieldList ) const
{
    QStringList returnList;

    QStringList::ConstIterator it = fieldList.begin();
    for ( ; it != fieldList.end(); ++it ) {
	returnList.append( fieldDescription( *it ) );
    }
    return returnList;
}


void
KBiblioShell::setFieldEditable( const QString &fieldName, bool editable )
{
    BibEntryFieldType *fieldType;

    fieldType = mDocument->lookupEntryFieldType( fieldName );
    if ( fieldType != 0 )
	fieldType->setEditable( editable );
}


void
KBiblioShell::setFieldViewable( const QString &fieldName, bool viewable )
{
    /*
    BibEntryFieldType *fieldType;

    fieldType = mDocument->lookupEntryFieldType( fieldName );
    if ( fieldType != 0 )
	fieldType->setViewable( viewable );
    */
}


void
KBiblioShell::load()
{
    QString fn = KFileDialog::getOpenFileName( QString::null, "*.kbi",
                                               this);
    if ( !fn.isEmpty() )
	load( fn );
}


void
KBiblioShell::load( const QString &fileName )
{
    QFile f( fileName );
    if ( !f.open( IO_ReadOnly ) )
	return;


    QTextStream t(&f);
    QString text = t.read();
    f.close();

    if ( setNewDoc( text ) ){
	mFileName = fileName;
	
	setCaption( fileName );
    }
}


void
KBiblioShell::save()
{
    if ( mFileName.isEmpty() ) {
	saveAs();
	return;
    }

    QFile f( mFileName );
    if ( !f.open( IO_WriteOnly ) ) {
	printf ( "Could not write to %s\n", mFileName.latin1() );
	return;
    }

    QTextStream t( &f );
    mDocument->save( t );
    f.close();

    mDocument->setEdited( false );

    setCaption( mFileName );
}


void
KBiblioShell::saveAs()
{
    QString fn = KFileDialog::getSaveFileName( QString::null, "*.kbi",
                                               this );
    if ( !fn.isEmpty() ) {
	if ( fn.findRev ( ".kbi", -1, false ) < 0 )
	    fn += ".kbi";
	mFileName = fn;
	save();
    }
}


void
KBiblioShell::saveToBibtex()
{
    QString fn = KFileDialog::getSaveFileName( QString::null, "*.bib",
                                               this );
    if ( !fn.isEmpty() ) {
	if ( fn.findRev ( ".bib", -1, false ) != fn.length()-4 )
	    fn += ".bib";
	
	QFile f( fn );
	if ( !f.open( IO_WriteOnly ) ) {
	    printf ( "Could not write to %s\n", fn.latin1() );
	    return;
	}

	QTextStream t( &f );
	SaveToBibtex::save( mDocument->firstEntry(), t );
	f.close();

    }
}


void
KBiblioShell::quit()
{
    if ( mDocument->edited() ){
	if ( KMessageBox::Cancel == KMessageBox::warningContinueCancel( 0, QString( i18n( "The document %1 has not been saved.\nAll changes will be lost. Exit anyway?" ) ).arg( mFileName ), "", i18n( "&Exit" ) ) )
	    return;
    }

    qApp->quit();
}

void
KBiblioShell::slotNewDoc()
{
    if ( setNewDoc( "" ) ){
	mFileName = i18n( "untitled.kbi" );
	setCaption( mFileName );
    }
}


bool
KBiblioShell::setNewDoc( const QString &text )
{
    if ( mDocument->edited() ){
	if ( KMessageBox::Cancel == KMessageBox::warningContinueCancel( 0, QString( i18n( "The document %1 has not been saved.\nAll changes will be lost. Proceed anyway?" ) ).arg( mFileName ), "", i18n( "&Proceed" ) ) )
	    return false;
    }

    if ( text.isEmpty() )
	mDocument->setEmptyDoc();
    else mDocument->setContent( text );

    mDocument->setEdited( false );

    emit newDocument();
    return true;
}


void
KBiblioShell::preferences()
{
    PreferencesDialog *dia = new PreferencesDialog( this, 0 );
    dia->show();
}


void
KBiblioShell::find()
{
    FindDialog *dia = new FindDialog( this, 0 );
    dia->show();
}


void
KBiblioShell::about()
{
    KMessageBox::about( 0, "KBiblio 0.0.2\n\nWritten by Claus Wilke, 1999\nclaus.wilke@gmx.net", "About kbiblio" );

}
#include "kbiblioshell.moc"






