/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qdom.h>
#include <qvaluelist.h>
#include <klocale.h>

#include <stdio.h>

#include "kbiblioshell.h"
#include "bibentrylistview.h"
#include "entrymanipulator.h"



BibEntryListViewItem::BibEntryListViewItem( QListView *parent, QDomElement entry )
    : QListViewItem( parent )
{
    mEntry = entry;

    KBiblioShell *shell = ( (BibEntryListView *) listView() )->mShell;
    mManip =
	    new EntryManipulator( entry, shell );
    setText( 0, mManip->label( ) );
    setText( 1, mManip->fieldFlatText( "AUTHOR" ) );
    setText( 2, mManip->fieldFlatText( "TITLE" ) );
    setText( 3, mManip->fieldFlatText( "YEAR" ) );
    setText( 4, shell->entryDescription( mManip->name() ) );
}


BibEntryListViewItem::~BibEntryListViewItem()
{
}


QDomElement
BibEntryListViewItem::entry() const
{
    return mEntry;
}


QString
BibEntryListViewItem::key ( int column, bool ascending ) const
{
    /*
      if ( 0 == column )
      return mEntry->sortKey();
      else
    */
    return mManip->label( );
}



BibEntryListView::BibEntryListView( KBiblioShell *shell, QDomNode baseNode, QWidget *parent, const char *name )
    : QListView( parent, name )
{
    mShell = shell;
    setup();
    newDocument( baseNode );
}


BibEntryListView::BibEntryListView( KBiblioShell *shell, QValueList<QDomElement> *elementList, QWidget *parent, const char *name )
    : QListView( parent, name )
{
    mShell = shell;
    setup();
    newDocument( elementList );
}


BibEntryListView::~BibEntryListView()
{
}


void
BibEntryListView::setup()
{
    addColumn( i18n( "Label" ) );
    addColumn( i18n( "Authors" ) );
    addColumn( i18n( "Title" ) );
    addColumn( i18n( "Year" ) );
    addColumn( i18n( "Type" ) );
    setAllColumnsShowFocus( TRUE );
    setSorting( 0, true );
}


void
BibEntryListView::newDocument( QDomNode baseNode )
{
    mBaseNode = baseNode;
    mUseElementList = false;

    rebuildList();
}


void
BibEntryListView::newDocument( QValueList<QDomElement> *elementList )
{
    mElementList = elementList;
    mUseElementList = true;

    rebuildList();
}


void
BibEntryListView::rebuildList()
{
    clearSelection();
    clear();

    if ( mUseElementList ){
	QValueList<QDomElement>::Iterator it = mElementList->begin();
	for( ; it != mElementList->end(); ++it )	
	    (void) new BibEntryListViewItem( this, *it );
    }
    else{
	QDomElement entry = mBaseNode.firstChild().toElement();

	for ( ; !entry.isNull(); entry = entry.nextSibling().toElement() )
	    (void) new BibEntryListViewItem( this, entry );
    }

    setSelected( firstChild(), true );
}


void
BibEntryListView::add( QDomElement entry )
{
    clearSelection();

    BibEntryListViewItem* be = new BibEntryListViewItem( this, entry );
    sort();
    setSelected( be, true );
}

void
BibEntryListView::remove( BibEntryListViewItem *item )
{
    if ( 0 != item ){
	clearSelection();
	BibEntryListViewItem *newSelectedItem = ( BibEntryListViewItem* ) item->itemBelow();
	if ( 0 == newSelectedItem )
	    newSelectedItem = ( BibEntryListViewItem* ) item->itemAbove();
	delete item;
	if ( 0 != newSelectedItem )
	    setSelected( newSelectedItem, true );
	sort();
    }
}

void
BibEntryListView::remove( QDomElement entry )
{
    BibEntryListViewItem *item;

    item = findItem( entry );

    remove( item );
}

void
BibEntryListView::update( QDomElement entry )
{
    BibEntryListViewItem *item;

    item = findItem( entry );

    if ( 0 != item ){
	clearSelection();
	delete item;
	setSelected( new BibEntryListViewItem( this, entry ), true );
	sort();
    }
    else
	printf("BibEntryListView::update( QDomElement entry ): entry to update does not exist!\n");
}



BibEntryListViewItem*
BibEntryListView::findItem( QDomElement entry)
{
    BibEntryListViewItem *item = 0;

    QListViewItemIterator it( this );
    for ( ; it.current(); ++it ){
	item = ( BibEntryListViewItem* ) it.current();
	if ( item->entry() == entry ){
	    break;
	}
    }
    return item;
}


#include "bibentrylistview.moc"






