//=============================================================================
// File:       nntp.cpp
// Contents:   Definitions for KMTNntpClient
// Maintainer: Doug Sauder <dwsauder@fwb.gulf.net>
// WWW:        http://www.fwb.gulf.net/~dwsauder/mimepp.html
// $Revision: 1.1.1.1 $
// $Date: 1999/07/13 15:54:55 $
//
// Copyright (c) 1996, 1997 Douglas W. Sauder
// All rights reserved.
//
// IN NO EVENT SHALL DOUGLAS W. SAUDER BE LIABLE TO ANY PARTY FOR DIRECT,
// INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
// THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF DOUGLAS W. SAUDER
// HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// DOUGLAS W. SAUDER SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
// NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
// BASIS, AND DOUGLAS W. SAUDER HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
// SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
//
//=============================================================================

#define DW_IMPLEMENTATION

#include <mimelib/config.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include "nntp.h"
#include <qglobal.h>
#include <ksimpleconfig.h>

#define NNTP_PORT 119
#define RECV_BUFFER_SIZE  8192
#define SEND_BUFFER_SIZE  1024

#if defined(DW_DEBUG_NNTP)
#  define DBG_NNTP_STMT(x) x
#else
#  define DBG_NNTP_STMT(x)
#endif

#include <qstring.h>
#include <qstrlist.h>
#include <kapp.h>
#include <pth.h>

#include "util.h"
#include "kfileio.h"
#include "connmgr.h"
#include "connmgrwidget.h"
#include "serversroot.h"
#include "resource.h"

QString speed (long siz,long delay)
{
    float bps=((float)1000000*siz)/delay;
    QString unit=" B/s";
    if (bps > 128)
    {
        bps=bps/1024;
        unit=" KB/s";
    }
    if (bps > 128)
    {
        bps=bps/1024;
        unit=" MB/s";
    }

    QString num;
    num.setNum(bps,'g',2);
    return num+unit;
}



KMTNntpClient::KMTNntpClient()
{
    canFetchByID=false;
    canFetchByIDWithoutGroup=false;
    stopnow=false;
    busy=0;
    mSendBuffer = new char[SEND_BUFFER_SIZE];
    mRecvBuffer = new char[RECV_BUFFER_SIZE];
    mLastChar = -1;
    mLastLastChar = -1;
    mNumRecvBufferChars = 0;
    mRecvBufferPos = 0;
    mReplyCode = 0;
}


KMTNntpClient::~KMTNntpClient()
{
    if (mRecvBuffer)
    {
        delete [] mRecvBuffer;
        mRecvBuffer = 0;
    }
    if (mSendBuffer)
    {
        delete [] mSendBuffer;
        mSendBuffer = 0;
    }
}

const char *KMTNntpClient::ProfileName()
{
    return mProfileName;
}

void KMTNntpClient::Stop()
{
    stopnow=true;
}

int KMTNntpClient::Open(const char* aServer, DwUint16 aPort)
{
    busy++;
    debug ("starting KMTNntpClient::Open");
    setStatus(QString("Connecting to server ")+aServer);
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    int err = DwProtocolClient::Open(aServer, aPort);
    if (! err)
    {
        PGetStatusResponse();
    }
    debug ("ending KMTNntpClient::Open");
    clearStatus();
    busy--;
    return mReplyCode;
}


int KMTNntpClient::ReplyCode() const
{
    return mReplyCode;
}


const DwString& KMTNntpClient::StatusResponse() const
{
    return mStatusResponse;
}


const DwString& KMTNntpClient::TextResponse() const
{
    return mTextResponse;
}


int KMTNntpClient::Article(int aArticleNum)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdArticle;
    if (aArticleNum >= 0)
    {
        sprintf(mSendBuffer, "ARTICLE %d\r\n", aArticleNum);
    }
    else
    {
        strcpy(mSendBuffer, "ARTICLE\r\n");
    }
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}

const char *KMTNntpClient::status()
{
    static QString data;
    static long diffs[10];
    static long new_siz=0;
    static long old_siz=0;
    static long diff_avg=0;
    static int count;

    data="server=";
    data+=mServerName;

    if (busy)
    {
        old_siz=new_siz;
        new_siz=mTextResponse.length();
        if (new_siz<old_siz)
            old_siz=0;
        count=(count+1)%10;
        diffs[count]=new_siz-old_siz;
        //WARNING! DON'T LOOK!!! Ugly hack to smooth the speed below.
        diff_avg=(diffs[0]+diffs[1]+diffs[2]+diffs[3]+diffs[4]+diffs[5]+diffs[6]+diffs[7]+diffs[8]+diffs[9])/10;
        //You can look now.

        data+=" ("+speed(diff_avg,500000)+") ";
    }

    for (int i=0;i<curstat;i++)
    {
        data+=" "+__status[i];
    }
    return data.data();
}

int KMTNntpClient::Article(const char* aMsgId)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdArticle;
    if (!aMsgId || !*aMsgId)
    {
        // error!
        return mReplyCode;
    }
    strcpy(mSendBuffer, "ARTICLE ");
    strncat(mSendBuffer, aMsgId, 80);
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Head(int aArticleNum)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdHead;
    if (aArticleNum >= 0)
    {
        sprintf(mSendBuffer, "HEAD %d\r\n", aArticleNum);
    }
    else
    {
        strcpy(mSendBuffer, "HEAD\r\n");
    }
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Head(const char* aMsgId)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdHead;
    if (!aMsgId || !*aMsgId)
    {
        return mReplyCode;
    }
    strcpy(mSendBuffer, "HEAD ");
    strncat(mSendBuffer, aMsgId, 80);
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Body(int articleNum)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdBody;
    if (articleNum >= 0)
    {
        sprintf(mSendBuffer, "BODY %d\r\n", articleNum);
    }
    else
    {
        strcpy(mSendBuffer, "BODY\r\n");
    }
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Body(const char* aMsgId)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdBody;
    if (!aMsgId || !*aMsgId)
    {
        return mReplyCode;
    }
    strcpy(mSendBuffer, "BODY ");
    strncat(mSendBuffer, aMsgId, 80);
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Stat(int articleNum)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdStat;
    if (articleNum >= 0)
    {
        sprintf(mSendBuffer, "STAT %d\r\n", articleNum);
    }
    else
    {
        strcpy(mSendBuffer, "STAT\r\n");
    }
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::Stat(const char* aMsgId)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdStat;
    if (!aMsgId || !*aMsgId)
    {
        return mReplyCode;
    }
    strcpy(mSendBuffer, "STAT ");
    strncat(mSendBuffer, aMsgId, 80);
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::Group(const char* aNewsgroupName)
{
    debug ("entered KMTNntpClient::Group");
    busy++;
    setStatus(QString("Entering ")+aNewsgroupName);
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdGroup;
    if (!aNewsgroupName || !*aNewsgroupName)
    {
        return mReplyCode;
    }
    strcpy(mSendBuffer, "GROUP ");
    strncat(mSendBuffer, aNewsgroupName, SEND_BUFFER_SIZE-32);
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    clearStatus();
    busy--;
    debug ("leaving KMTNntpClient::Group");
    gname=aNewsgroupName;
    return mReplyCode;
}


int KMTNntpClient::Help()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdHelp;
    strcpy(mSendBuffer, "HELP\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 1)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Last()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdLast;
    strcpy(mSendBuffer, "LAST\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::List()
{
    busy++;
    setStatus("Downloading Active File");
    debug ("starting KMTNntpClient::List");
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdList;
    strcpy(mSendBuffer, "LIST\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    debug ("ending KMTNntpClient::List");
    clearStatus();
    busy--;
    return mReplyCode;
}


int KMTNntpClient::Newgroups(const char* aDate, const char* aTime,
                             DwBool aIsGmt, const char* aDistribution)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdNewgroups;
    strcpy(mSendBuffer, "NEWGROUPS ");
    strncat(mSendBuffer, aDate, 16);
    strcat(mSendBuffer, " ");
    strncat(mSendBuffer, aTime, 16);
    if (aIsGmt)
    {
        strcat(mSendBuffer, " GMT");
    }
    if (aDistribution)
    {
        strcat(mSendBuffer, " ");
        strncat(mSendBuffer, aDistribution, SEND_BUFFER_SIZE-64);
    }
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Newnews(const char* aNewsgroups, const char* aDate,
                           const char* aTime, DwBool aIsGmt, const char* aDistribution)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdNewnews;
    strcpy(mSendBuffer, "NEWNEWS ");
    strncat(mSendBuffer, aNewsgroups, SEND_BUFFER_SIZE-64);
    strcat(mSendBuffer, " ");
    strncat(mSendBuffer, aDate, 16);
    strcat(mSendBuffer, " ");
    strncat(mSendBuffer, aTime, 16);
    if (aIsGmt)
    {
        strcat(mSendBuffer, " GMT");
    }
    if (aDistribution)
    {
        strcat(mSendBuffer, " ");
        size_t n = strlen(mSendBuffer);
        strncat(mSendBuffer, aDistribution, SEND_BUFFER_SIZE-n-4);
    }
    strcat(mSendBuffer, "\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    return mReplyCode;
}


int KMTNntpClient::Next()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdNext;
    strcpy(mSendBuffer, "NEXT\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::Post()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdPost;
    strcpy(mSendBuffer, "POST\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::Quit()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdQuit;
    strcpy(mSendBuffer, "QUIT\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::Slave()
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdSlave;
    strcpy(mSendBuffer, "SLAVE\r\n");
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
    }
    return mReplyCode;
}


int KMTNntpClient::SendData(const DwString& aStr)
{
    return SendData(aStr.data(), aStr.length());
}


int KMTNntpClient::SendData(const char* aBuf, int aBufLen)
{
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";

    int pos = 0;
    int len = 0;
    const char* buf = 0;

    int lastLastChar = '\r';
    int lastChar = '\n';

    while (1)
    {
        if (stopnow)
        {
            mReplyCode = 0;
            break;
        }
        len = SEND_BUFFER_SIZE;
        len = (len < aBufLen - pos) ? len : aBufLen - pos;
        if (len == 0) break;

        // Look for CR LF '.'.  If it is found, then we have to copy the buffer
        // and stuff an extra '.'.

        int hasCrLfDot = 0;
        int tLastChar = lastChar;
        int tLastLastChar = lastLastChar;
        for (int i=0; i < len; ++i)
        {
            int ch = aBuf[pos+i];
            if (tLastLastChar == '\r' && tLastChar == '\n' && ch == '.')
            {
                hasCrLfDot = 1;
                break;
            }
            tLastLastChar = tLastChar;
            tLastChar = ch;
        }
        if (! hasCrLfDot)
        {
            lastChar = tLastChar;
            lastLastChar = tLastLastChar;
            buf = &aBuf[pos];
            pos += len;
        }

        // If CR LF '.' was found, copy the chars to a different buffer and stuff
        // the extra '.'.

        else /* (hasCrLfDot) */
        {
            tLastChar = lastChar;
            tLastLastChar = lastLastChar;
            int iDst = 0;
            int iSrc = 0;
            // Implementation note: be careful to avoid overrunning the
            // destination buffer when CR LF '.' are the last three characters
            // of the source buffer.
            while (iDst < SEND_BUFFER_SIZE && iSrc < len)
            {
                int ch = aBuf[pos+iSrc];
                if (tLastLastChar == '\r' && tLastChar == '\n' && ch == '.')
                {
                    if (iDst == SEND_BUFFER_SIZE-1)
                    {
                        break;
                    }
                    mSendBuffer[iDst++] = '.';
                }
                mSendBuffer[iDst++] = (char) ch;
                ++iSrc;
                tLastLastChar = tLastChar;
                tLastChar = ch;
            }
            lastChar = tLastChar;
            lastLastChar = tLastLastChar;
            len = iDst;
            buf = mSendBuffer;
            pos += iSrc;
        }

        // Send the buffer

        int numSent = PSend(buf, len);
        if (numSent != len)
        {
            mReplyCode = 0;
            return mReplyCode;
        }
    }

    // Send final '.' CR LF.  If CR LF are not at the end of the buffer, then
    // send a CR LF '.' CR LF.

    if (lastLastChar == '\r' && lastChar == '\n')
    {
        PSend(".\r\n", 3);
    }
    else
    {
        PSend("\r\n.\r\n", 5);
    }

    // Get the server's response

    PGetStatusResponse();
    return mReplyCode;
}


void KMTNntpClient::PGetStatusResponse()
{
    mReplyCode = 0;
    mStatusResponse = "";
    char* ptr;
    int len;
    int err = PGetLine(&ptr, &len);
    if (! err)
    {
        mReplyCode = strtol(ptr, NULL, 10);
        mStatusResponse.assign(ptr, len);
        DBG_NNTP_STMT(char buffer[256];)
        DBG_NNTP_STMT(strncpy(buffer, ptr, len);)
        DBG_NNTP_STMT(buffer[len] = 0;)
        DBG_NNTP_STMT(cout << "S: " << buffer;)
    }
}


void KMTNntpClient::PGetTextResponse()
{
    mTextResponse = "";

    // Get a line at a time until we get CR LF . CR LF

    while (1)
    {
        if (stopnow)
        {
            mReplyCode = 0;
            break;
        }
        char* ptr;
        int len;
        int err = PGetLine(&ptr, &len);

        // Check for an error

        if (err)
        {
            mReplyCode = 0;
            return;
        }

        // Check for '.' on a line by itself, which indicates end of multiline
        // response

        if (len >= 3 && ptr[0] == '.' && ptr[1] == '\r' && ptr[2] == '\n')
        {
            break;
        }

        // Remove '.' at beginning of line

        if (*ptr == '.') ++ptr;

        mTextResponse.append(ptr, len);
    }
}


int KMTNntpClient::PGetLine(char** aPtr, int* aLen)
{
    // Restore the saved state

    int startPos = mRecvBufferPos;
    int pos = mRecvBufferPos;
    int lastChar = -1;

    // Keep trying until we get a complete line, detect an error, or
    // determine that the connection has been closed

    int isEndOfLineFound = 0;
    while (1)
    {
        if (stopnow)
        {
            return -1;
        }

        // Search buffer for end of line chars. Stop when we find them or when
        // we exhaust the buffer.

        while (pos < mNumRecvBufferChars)
        {
            if (lastChar == '\r' && mRecvBuffer[pos] == '\n')
            {
                isEndOfLineFound = 1;
                ++pos;
                break;
            }
            lastChar = mRecvBuffer[pos];
            ++pos;
        }
        if (isEndOfLineFound)
        {
            *aPtr = &mRecvBuffer[startPos];
            *aLen = pos - startPos;
            mRecvBufferPos = pos;
            return 0;
        }

        // If the buffer has no room, return without an error; otherwise,
        // replenish the buffer.

        // Implementation note: The standard does not allow long lines,
        // however, that does not mean that they won't occur.  The way
        // this function deals with long lines is to return a full buffer's
        // worth of characters as a line.  The next call to this function
        // will start where this call left off.  In essence, we have
        // *forced* a line break, but without putting in CR LF characters.

        if (startPos == 0 && pos == RECV_BUFFER_SIZE)
        {
            *aPtr = mRecvBuffer;
            *aLen = RECV_BUFFER_SIZE;
            mRecvBufferPos = pos;
            return 0;
        }
        memmove(mRecvBuffer, &mRecvBuffer[startPos],
                mNumRecvBufferChars-startPos);
        mNumRecvBufferChars -= startPos;
        mRecvBufferPos = mNumRecvBufferChars;
        int bufFreeSpace = RECV_BUFFER_SIZE - mRecvBufferPos;
        int n = PReceive(&mRecvBuffer[mRecvBufferPos], bufFreeSpace);
        if (n == 0)
        {
            // The connection has been closed or an error occurred
            return -1;
        }
        mNumRecvBufferChars += n;
        startPos = 0;
        pos = mRecvBufferPos;
    }
}

bool KMTNntpClient::getActive()
{
    debug ("entered getActive");
    busy++;
    setStatus("Updating Active");
    //FIXME handle error code better, look at RFC
    int rcode=List();
    if (rcode==0) //No response
    {
        debug ("leaving getActive (no response)");
        clearStatus();
        busy--;
        return false;
    }
    QString resp=TextResponse().c_str();
    QStrList l;
    QStringSplit(resp,'\n',l);
    QListIterator <char> it(l);
    QString data(resp.length());
    char *index,*dindex=data.data();
    int len=0;
    clearStatus();
    setStatus("Saving Active List");
    for ( ; it.current(); ++it )
    {
        index=strchr(it.current(),' ');
        *index=0;
        len=index-it.current();
        strncpy(dindex,it.current(),len);
        *(dindex+len)='\n';
        dindex=dindex+len+1;
        pth_yield(NULL);
    }
    data.resize(dindex-data.data());
    data+="\n";
    //FIXME handle errors
    kStringToFile(data, (KApplication::localkdedir()+"/share/apps/newkrn/servers.active/")+ProfileName(),
                  FALSE,FALSE,TRUE);
    clearStatus();
    busy--;
    debug ("leaving getActive");
    return true;
}

int KMTNntpClient::Xover(int from, int to)
{
    busy++;
    debug ("starting KMTNntpClient::Xover");
    setStatus(QString("Xover data"));
    mReplyCode = 0;
    mStatusResponse = mTextResponse = "";
    mLastCommand = kCmdXover;
    sprintf(mSendBuffer, "XOVER %d-%d\r\n", from,to);
    DBG_NNTP_STMT(cout << "C: " << mSendBuffer << flush;)
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
        }
    }
    clearStatus();
    debug ("leaving KMTNntpClient::Xover");
    busy--;
    return mReplyCode;
}

int KMTNntpClient::ListOverviewFmt()
{
    busy++;
    debug ("starting KMTNntpClient::ListOverviewFmt");
    setStatus("List Overview Format");
    strcpy(mSendBuffer, "LIST overview.fmt\r\n");

    // for debugging
    cout << "C: " << mSendBuffer << endl;

    mReplyCode = -1;
    int bufferLen = strlen(mSendBuffer);
    int numSent = PSend(mSendBuffer, bufferLen);
    if (numSent == bufferLen)
    {
        PGetStatusResponse();
        cout <<"S: " << StatusResponse() << endl;
        if (mReplyCode/100%10 == 2)
        {
            PGetTextResponse();
            QString of=TextResponse().data();
            int index;

            index=of.find ("Subject");
            if (index!=-1)
                OffsetSubject=of.left(index).contains('\n')+1;
            else
                OffsetSubject=0;

            index=of.find ("From");
            if (index!=-1)
                OffsetFrom=of.left(index).contains('\n')+1;
            else
                OffsetFrom=0;

            index=of.find ("Lines");
            if (index!=-1)
                OffsetLines=of.left(index).contains('\n')+1;
            else
                OffsetLines=0;

            index=of.find ("Message-ID");
            if (index!=-1)
                OffsetID=of.left(index).contains('\n')+1;
            else
                OffsetID=0;

            index=of.find ("Date");
            if (index!=-1)
                OffsetDate=of.left(index).contains('\n')+1;
            else
                OffsetDate=0;

            index=of.find ("References");
            if (index!=-1)
                OffsetRef=of.left(index).contains('\n')+1;
            else
                OffsetRef=0;
        }
        else
        {
            OffsetSubject=1;
            OffsetFrom=2;
            OffsetDate=3;
            OffsetID=4;
            OffsetRef=5;
            OffsetLines=7;
        }
        debug ("Offsets:%d,%d,%d,%d,%d,%d",OffsetSubject,OffsetFrom,OffsetLines,OffsetID,OffsetDate,OffsetRef);
        //This one is always the same
        OffsetArticleNumber=0;
    }
    clearStatus();
    busy--;
    debug ("leaving KMTNntpClient::ListOverviewFmt");
    return mReplyCode;
}

struct connect_struct
{
    KMTNntpClient *client;
    QString server;
    int port;
};

bool KMTNntpClient::Connect(const char *profile)
{
    busy++;
    debug ("entered Connect");
    setStatus("Connecting");
    KSimpleConfig conf(KApplication::localkdedir()+"/share/apps/newkrn/servers/"+profile);
    conf.setGroup("ServerOptions");
    QString servername=conf.readEntry("name");
    uint port=conf.readNumEntry("port");
    bool success=false;
    int res1=Open(servername,port);
    if ((res1==200) || (res1==201)) //did connect
    {
        success=true;
        ListOverviewFmt();
    }
    //FIXME authentication not implemented
    clearStatus();
    mProfileName=profile;
    busy--;
    debug ("leaving Connect");
    return success;
}






























