##############################################################################
#                                                                            #
# kftpupdater - Mirror a local directory to an ftp server or vice versa      #
# Copyright (C) 2000 Martin P. Holland  <m.holland@noether.freeserve.co.uk>  #
#                                                                            #
# This program is free software; you can redistribute it and/or modify       #
# it under the terms of the GNU General Public License as published by       #
# the Free Software Foundation; either version 2 of the License, or          #
# (at your option) any later version.                                        #
#                                                                            #
# This program is distributed in the hope that it will be useful,            #
# but WITHOUT ANY WARRANTY; without even the implied warranty of             #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              #
# GNU General Public License for more details.                               #
#                                                                            #
# You should have received a copy of the GNU General Public License          #
# along with this program; if not, write to the Free Software                #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  #
#                                                                            #
##############################################################################

import sys, string
from ftplib import *
from timehelper import *
from listparse import listparse, _IS_BAD, _IS_FILE, _IS_DIR

class Ftp(FTP):
    """
    This class add some extra features to FTP from ftplib.

    In particular,
    MDTM        file modification time
    BINGET      fetch a file in binary mode
    FILES_DIRS  returns list of files and directories in remote dir

    It doesn't catch the exceptions from ftplib.
    It shouldn't raise any new ones of its own but
    may raise those from ftplib.
    """

    #the default storbinary in the ftplib doesn't allow me
    #to refresh between blocks so here is a modifed version

    def storbinary2(self, cmd, fp, blocksize):
        '''Store a file in binary mode.'''
        self.voidcmd('TYPE I')
        conn = self.transfercmd(cmd)
        while 1:
            #New line inserted by me to allow a widget to refresh
            #between blocks
            self.event()
            buf = fp.read(blocksize)
            if not buf: break
            conn.send(buf)
        conn.close()
        return self.voidresp()

    #hacked to show progress

    def storbinary3(self, cmd, fp, blocksize):
        '''Store a file in binary mode.'''
        self.voidcmd('TYPE I')
        conn = self.transfercmd(cmd)
        while 1:
            #New line inserted by me to allow a widget to refresh
            #between blocks
            self.event()
            buf = fp.read(blocksize)
            if not buf: break
            conn.send(buf)
            self.do_progress(len(buf))
#            print "sent",len(buf),"bytes"
        conn.close()
        self.finish_progress()
        return self.voidresp()

    def setProgress(self,do,finish):
        self.do_progress=do
        self.finish_progress=finish

    def do_progress(self):
        pass

    def finish_progress(self):
        pass

    def binget(self,local,remote=None):
        "get REMOTE file and place it on LOCAL file in binary mode."
        if remote==None: remote=local
        self.fd=open(local,'w')
        self.retrbinary("RETR "+remote,self.fd.write)
        self.fd.close()

    def binget2(self,local,remote=None):
        "get REMOTE file and place it on LOCAL file in binary mode."
        if remote==None: remote=local
        self.fd=open(local,'w')
        self.retrbinary("RETR "+remote,self._helper)
        self.fd.close()

    def binget3(self,local,remote=None):
        "get REMOTE file and place it on LOCAL file in binary mode."
        if remote==None: remote=local
        self.fd=open(local,'w')
        self.retrbinary("RETR "+remote,self._newhelper)
        self.fd.close()
        self.finish_progress()

    def _newhelper(self,s):
        self.event()
        self.fd.write(s)
#        print "got",len(s),"bytes"
        self.do_progress(len(s))

    def event(self):
        "default behaviour between blocks is to do nothing"
        pass

    def setEvent(self,f):
        "override default event between blocks"
        self.event=f

    def _helper(self,s):
        self.fd.write(s)
        self.event()

    def mdtm(self,remote):
        "Return modification time in seconds since epoch of REMOTE."
        mdtm_response=self.sendcmd('MDTM '+remote)
        try:
            year=int(mdtm_response[4:8])
            month=int(mdtm_response[8:10])
            day=int(mdtm_response[10:12])
            hours=int(mdtm_response[12:14])
            mins=int(mdtm_response[14:16])
            secs=int(mdtm_response[16:])
        except ValueError:
            print 'Server gave unexpected response to MDTM, darn!'
            return -1
        else:
            try:
                ans=timegm((year,month,day,hours,mins,secs))
            except gmterror,detail:
                print "Bad time stamp",detail
                ans=-1
            return ans

    def filesdirs(self,rdir):
        """
        Return list of files and directories in remote directory RDIR.

        Character devices, block devices, ., .., and
        anything else that is not understood by LISTPARSE is ignored.
        """
        raw_list=[]
        self.dir('-a',rdir,raw_list.append)
        files=[]
        dirs=[]
        for line in raw_list:
            ans,file=self._listparse(line)
            if ans==_IS_DIR:
                dirs.append(file)
            elif ans==_IS_FILE:
                files.append(file)
        return files,dirs

    def emptydir(self,d):
        "rm -rf the remote directory D."
        #First we must empty the directory
        files,dirs=self.filesdirs(d)
        for f in dirs:
            self.emptydir(os.path.join(d,f))
        for f in files:
            self.delete(os.path.join(d,f))
        self.rmd(d)

    def existsremotefile(self,f):
        "Does the remote file F exist?"
        lines=[]
        self.ftp.dir('-a',f,lines.append)
        #should only really be one line here but in case
        #server prints out some preamble parse all lines
        for line in lines:
            ans,filename=self._listparse(line)
            if ans==_IS_FILE:
                return 1
        return 0

    def _listparse(self,line):
        return listparse(line)
        
    def old_listparse(self,line):
        """
        Parse an output LINE from a LIST command on an ftp server.

        Assumes LINE has had the \r\n stripped

        It returns:
        flag, filename
        where flag=_IS_BAD   not a regular file or directory
                   _IS_FILE  a regular file or link
                   _IS_DIR   a directory
        character, block devices etc fall into IS_BAD
        as do . and ..
        In the IS_BAD case None is returned for the filename
        """
        #First split line into 8 pieces
        #the first 7 are seperated by whitespace
        #the last piece is what's left: it can contain spaces
        fields=string.split(line,None,7)
        if len(fields)<8 or fields[7] in ('.','..'):
            #too few fields or .,.. so bogus
            return _IS_BAD,None
        if fields[4] in ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'):
            #must be a file/dir with a blank "hard links" field
            filename=fields[7]
        else:
            #can't be in blank "hard links" case so fields[7] needs to be split in 2
            end=string.split(fields[7],None,1)
            if len(end)<2 or end[1] in ('.','..'): #bogus
                return _IS_BAD,None
            filename=end[1]
        filename=string.split(filename,' -> ')[0]
        if fields[0][0]=='d':
            return _IS_DIR,filename
        elif fields[0][0] in ('-','l'):
            return _IS_FILE,filename
        return _IS_BAD,None

#ftp=Ftp()
#ftp.set_debuglevel(1)
#ftp.connect('localhost')
#ftp.login('martin','******')
#ftp.binget('foo','bar')
#epoch_time=ftp.mdtm('bar')
#print epoch_time
#import calendar
#print calendar.gmtime(epoch_time)
#ftp.quit()
