/* 
    xWinTV using Bt848 frame grabber driver

    Copyright (C) 1998 Moritz Wenk (wenk@mathematik.uni-kl.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "videotxt.h"

#define vtxDEBUG

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
//#include <sys/vtx.h>

vtxClass::vtxClass(char *_vtxdev)
{
  int status;

  vtxdev=(char *)_vtxdev;

#ifdef vtxDEBUG
  printf("vtx: opening %s\n",vtxdev);
#endif

  if ((status = openDevice(REQ_MAJOR, REQ_MINOR, &vtx_info)) < 0) {
    if (status == VTXEVERSION) {
      printf("vtx: can't open videotext-device: incompatible driver version (need %d.%d)\n",
	      REQ_MAJOR, REQ_MINOR);
    } else {
      printf("vtx: can't open videotext-device %s: %s\n",vtxdev,strerror(errno));
    }
    exit(1);
  }

}

/* Open device, initialize SAA 5246 & internal tables; this also checks, if the driver-version
 * is compatible (you have to pass the expected major-version and the lowest possible minor-
 * version); it also fills in the vtx_info_t-struct pointed to by prg_info
 * Return VTXOK if success, VTXEOPEN if device isn't ready, VTXERR if I/O-error occured,
 * VTXEVERSION if driver-version isn't compatible
 */
int vtxClass::openDevice(int major, int minor, vtx_info_t *prg_info) {
  vtx_info_t *info, tmp_info;
  
  if ((fvtx = open(vtxdev, O_RDONLY)) < 0)
    return VTXEOPEN;
  info = (prg_info != NULL ? prg_info : &tmp_info);
  if (ioctl(fvtx, VTXIOCGETINFO, info) < 0)
    return VTXERR;
  if (info->version_major != major || info->version_minor < minor) {
    while (close(fvtx) < 0 && errno == EINTR);
    fvtx = -1;
    return VTXEVERSION;
  }

#ifdef vtxDEBUG
  printf("vtx: info: version: %d.%d\n",info->version_major,info->version_minor);
  printf("vtx: info: numpages: %d\n",info->numpages);
  printf("vtx: info: vtx chipset: %s\n",chipset[info->cct_type].str);
#endif

  return VTXOK;
}


/* Turn off SAA 5246, close device
 */
void vtxClass::closeDevice(void) {
  if (fvtx == -1)
    return;
  while (close(fvtx) < 0 && errno == EINTR);
  fvtx = -1;

#ifdef vtxDEBUG
  printf("vtx: device closed.\n");
#endif
}


/* Clear page-buffer pgbuf in SAA 5246 & wait.
 * Return VTXOK if success, VTXEINVAL if pgbuf is out of range, VTXERR if I/O-error occured
 */
int vtxClass::clearpgbuf(int pgbuf) {
  vtx_pagereq_t pagereq;
  
  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.pgbuf = pgbuf;
  if (ioctl(fvtx, VTXIOCCLRPAGE, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }

#ifdef vtxDEBUG
  printf("vtx: clearpgbuf %d\n",pgbuf);
#endif

  return VTXOK;
}


/* Check if SAA 5246 already found a page in pgbuf. This only checks if the beginning of a page
 * was found, NOT the whole page!
 * Return VTXOK if page was found, VTXNOTFOUND if no page was found, VTXEINVAL if pgbuf is out
 * of range, VTXERR if I/O-error occured
 */
int vtxClass::checkpage(int pgbuf) {
  vtx_pagereq_t pagereq;
  vtx_pageinfo_t pageinfo;
  
  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.pgbuf = pgbuf;
  pagereq.buffer = &pageinfo;
  if (ioctl(fvtx, VTXIOCGETSTAT, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }
  if (pageinfo.notfound || pageinfo.hamming)
    return VTXNOTFOUND;
  return VTXOK;
}


/* Stop data acquisition unit for pgbuf.
 * Return VTXOK if success, VTXEINVAL if pgbuf is out of range, VTXERR if I/O-error occured
 */
int vtxClass::stop_dau(int pgbuf) {
  vtx_pagereq_t pagereq;
  
  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.pgbuf = pgbuf;
  if (ioctl(fvtx, VTXIOCSTOPDAU, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }
  return VTXOK;
}


/* Reset bits in pgbuf used to check if a page was already found
 * Return VTXOK if success, VTXEINVAL if pgbuf is out of range, VTXERR if I/O-error occured
 */
int vtxClass::reset_pgfound(int pgbuf) {
  vtx_pagereq_t pagereq;
  
  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.pgbuf = pgbuf;
  if (ioctl(fvtx, VTXIOCCLRFOUND, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }
  return VTXOK;
}


/* Search for page page/hour/minute (hexadecimal !!!), only regard digits with corresponding bits
 * in pagemask set (constants defined in <sys/vtx.h>), store page in page-buffer pgbuf
 * Return VTXOK if success, VTXEINVAL if page, hour, minute or pgbuf is out of range, VTXERR if
 * I/O-error occured
 */
int vtxClass::searchpage(int page, int hour, int minute, int pagemask, int pgbuf) {
  vtx_pagereq_t pagereq;
  
  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.page = page;
  pagereq.hour = hour;
  pagereq.minute = minute;
  pagereq.pagemask = pagemask;
  pagereq.pgbuf = pgbuf;
  if (ioctl(fvtx, VTXIOCPAGEREQ, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }

#ifdef vtxDEBUG
  printf("vtx: searchpage successfull %d, pgbug %d\n",page,pgbuf);
#endif

  return VTXOK;
}


/* Get page from page-buffer pgbuf of SAA 5246 (first char x1/y1, last char x2/y2), store it in
 * buffer (= ptr to array of byte_t's)
 * Return VTXOK if success, VTXEINVAL if corner-coordinates or pgbuf is invalid, VTXERR if
 * I/O-error occured
 */
int vtxClass::getpage(int pgbuf, int x1, int y1, int x2, int y2, byte_t *buffer, vtx_pageinfo_t *info) {
  vtx_pagereq_t pagereq;

  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.pgbuf = pgbuf;
  pagereq.start = y1 * 40 + x1;
  pagereq.end = y2 * 40 + x2;
  if (info) {
    pagereq.buffer = info;
    if (ioctl(fvtx, VTXIOCGETSTAT, &pagereq) < 0) {
      if (errno == EINVAL)
        return VTXEINVAL;
      return VTXERR;
    }
  }
  if (buffer) {
    pagereq.buffer = buffer;
    if (ioctl(fvtx, VTXIOCGETPAGE, &pagereq) < 0) {
      if (errno == EINVAL)
        return VTXEINVAL;
      return VTXERR;
    }
  }

#ifdef vtxDEBUG
  printf("vtx: getpage: %d\n",pgbuf);
#endif

  return VTXOK;
}


/* Display page in buffer (first char x1/y1, last char x2/y2) on TV-Screen.
 * Return VTXOK if success, VTXEINVAL if corner-coordinates or are invalid, VTXERR if
 * I/O-error occured
 */
int vtxClass::putpage(int x1, int y1, int x2, int y2, const byte_t *buffer, const vtx_pageinfo_t *info) {
  vtx_pagereq_t pagereq;

  if (fvtx == -1)
    return VTXENOTOPEN;
  if (info) {
    pagereq.buffer = (vtx_pageinfo_t*)info;
    if (ioctl(fvtx, VTXIOCPUTSTAT, &pagereq) < 0) {
      if (errno == EINVAL)
        return VTXEINVAL;
      return VTXERR;
    }
  }
  if (buffer) {
    pagereq.start = y1 * 40 + x1;
    pagereq.end = y2 * 40 + x2;
    pagereq.buffer = (byte_t*)buffer;
    if (ioctl(fvtx, VTXIOCPUTPAGE, &pagereq) < 0) {
      if (errno == EINVAL)
        return VTXEINVAL;
      return VTXERR;
    }
  }
  return VTXOK;
}


/* Set mode of video-output of SAA5246 (for displaying pages on TV-screen).
 * Return VTXOK if success of VTXERR if I/O-error occured.
 */
int vtxClass::set_display(vtxdisp_t disp) {
  vtx_pagereq_t pagereq;

  if (fvtx == -1)
    return VTXENOTOPEN;
  pagereq.page = disp;
  if (ioctl(fvtx, VTXIOCSETDISP, &pagereq) < 0) {
    if (errno == EINVAL)
      return VTXEINVAL;
    return VTXERR;
  }
  return VTXOK;
}


/* Clear the page-cache on cards that have one.
 * Return VTXOK if success (or if interface has noch cache) or VTXERR if I/O-error occured.
 */
int vtxClass::clear_cache(void) {
  if (fvtx == -1)
    return VTXENOTOPEN;
  return ((ioctl(fvtx, VTXIOCCLRCACHE) < 0) ? VTXERR : VTXOK);
}


/* Turn on/off virtual mode. Most interfaces can't display a page on TV if virtual mode is on!
 * Return VTXOK if success or VTXERR if I/O-error occured.
 */
int vtxClass::set_vir(int vir) {
  if (fvtx == -1)
    return VTXENOTOPEN;
  return ((ioctl(fvtx, VTXIOCSETVIRT, vir) < 0) ? VTXERR : VTXOK);
}

