/* 
    kirc -- KDE Infrared Remote Control

    Copyright (C) 1998 Moritz Wenk (wenk@mathematik.uni-kl.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_IRC

#include <kirc.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <qpushbutton.h> 
#include <qlabel.h>

extern int h_errno;
extern int errno;

#ifdef myDEBUG
#define kDEBUG
//#define kDEBUG_A
//#define kircWARN
#endif

// =================================================================================================

KIRCWidget::KIRCWidget( KConfig* pconf )
  : aCodeDict(37), aWidgetDict(11)
{
  if (!pconf)
    fatal("kirc: Null KConfig object");
  pConfig = pconf;
  aAvailableId = 1;
  aCodeDict.setAutoDelete(TRUE);
  aWidgetDict.setAutoDelete(TRUE);
  keyUP=KCodeEntry::normal;
  num10=0;

  ircclient= new KIRCClient( LIRCD_DEVICE );
#ifdef kircWARN
  if ( !ircclient->clientStatus() ) {
    warning("kwintv: Unable to open socket connection to lircd!");
    warning("kwintv: Infrared remote control funtionality will be disabled!");
  } 
#endif

  configMode=false;

  kc= new KIRCConfig( &aCodeDict );
  kc->hide();

  connect( kc, SIGNAL(apply()), this, SLOT( applySLOT() ));
  connect( kc, SIGNAL(defaults()), this, SLOT( defaultsSLOT() ));
  connect( kc, SIGNAL(cancel()), this, SLOT( cancelSLOT() ));
  
  connect( ircclient, SIGNAL(sendReceivedDataSIGNAL( ullong, unsigned int, char *, char * )),
	   this, SLOT(receiveDataSLOT( ullong, unsigned int, char *, char * )) );

  connect( this, SIGNAL( configDataReceivedSIGNAL( ullong, char *, char * )), 
	   kc, SLOT(receiveCodeSLOT( ullong, char *, char * )));

}

// =================================================================================================

KIRCWidget::~KIRCWidget()
{
  sync();

  /* remove all widgetEntries */
  aWidgetDict.clear();
  aCodeDict.clear();

  delete ircclient;
  delete kc;
}

// =================================================================================================

void KIRCWidget::applySLOT()
{
  kc->hide();
  configMode=false;

  QDictIterator<KCodeEntry> aKeyIt(aCodeDict); 
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    aKeyIt.current()->aCurrentKeyCode=aKeyIt.current()->aConfigKeyCode;
    aKeyIt.current()->aCurrentKey=aKeyIt.current()->aConfigKey;
#ifdef kDEBUG
    debug("kirc: apply: %d:%d, %s:%s", 
	  (int)aKeyIt.current()->aCurrentKeyCode, (int)aKeyIt.current()->aConfigKeyCode,
	  (const char *)aKeyIt.current()->aCurrentKey, (const char *)aKeyIt.current()->aConfigKey);
#endif
    ++aKeyIt;
  }

  sync();
}

// =================================================================================================

void KIRCWidget::defaultsSLOT()
{
  kc->hide();
  configMode=false;
}

// =================================================================================================

void KIRCWidget::cancelSLOT()
{
  kc->hide();
  configMode=false;
}

// =================================================================================================

void KIRCWidget::receiveDataSLOT( ullong code, unsigned int count, char * key, char * config )
{

  if (configMode) {
    emit configDataReceivedSIGNAL( code, key, config );
    return;
  }

  QDictIterator<KCodeEntry> aKeyIt(aCodeDict);
  aKeyIt.toFirst();
  while ( aKeyIt.current() && !(aKeyIt.current()->aCurrentKeyCode == code) ) {
    ++aKeyIt;
  }

#ifdef kDEBUG
  debug("kirc: data: %d, function: <%s>",(int)code,aKeyIt.current()?aKeyIt.currentKey():"none");
#endif

  if ( aKeyIt.current() && aKeyIt.current()->isEnabled ) {

    if ( aKeyIt.current()->typeOfButton == KCodeEntry::once && count > 0 ) return; 

    KCodeEntry *kecode= aKeyIt.current();

    switch ( kecode->typeOfCode ) {
    case KCodeEntry::twoDigits:
    case KCodeEntry::up10:
    case KCodeEntry::up20:
    case KCodeEntry::up30:
      keyUP= kecode->typeOfCode;
      num10=0;
      emit specialKeyReceivedSIGNAL( kecode->typeOfCode );
      return;

    case KCodeEntry::number:
      switch ( keyUP ) {
      case KCodeEntry::twoDigits:
	if ( num10 == 0 ) {
	  num10= QString( aKeyIt.currentKey() ).toInt();
	  return;
	} else {
	  num= QString( aKeyIt.currentKey() ).toInt() + num10 * 10;
	  keyUP=KCodeEntry::normal;
	}
      case KCodeEntry::up10:
	num= QString( aKeyIt.currentKey() ).toInt() + 10;
	keyUP=KCodeEntry::normal;
	break;
      case KCodeEntry::up20:
	num= QString( aKeyIt.currentKey() ).toInt() + 20;
	keyUP=KCodeEntry::normal;
	break;
      case KCodeEntry::up30:
	num= QString( aKeyIt.currentKey() ).toInt() + 30;
	keyUP=KCodeEntry::normal;
	break;
      case KCodeEntry::normal:
	num= QString( aKeyIt.currentKey() ).toInt();
	break;
      }

      emit numKeyReceivedSIGNAL( num );
      
#ifdef kDEBUG
      debug("kirc: numKeyReceivedSIGNAL( %d )", num );
#endif
      break;

    case KCodeEntry::normal:
      QDictIterator<KCodeConnectEntry> kcce(*kecode->pConnectDict);
      kcce.toFirst();

      while ( kcce.current() ) {
#ifdef kDEBUG
      debug("kirc: send signal");
#endif
	kcce.current()->sig->activate();
	++kcce;
      }
      
      break;
    } // switch
 
  }
}

// =================================================================================================

bool KIRCWidget::addCode( const QString& functionName, uint defaultCode, 
			  KCodeEntry::codeType type, KCodeEntry::buttonType button,
			  bool configurable )
{
#ifdef kDEBUG_A
  debug("kirc: register function %s",(const char *)functionName);
#endif
  /* search for an existing "functionName" entry */
  KCodeEntry *pEntry = aCodeDict[ functionName ];
	
  if ( pEntry ) /* already exists : remove it */
    removeCode(functionName);

  pEntry = new KCodeEntry;
  aCodeDict.insert( functionName, pEntry );
	
  pEntry->aDefaultKeyCode = defaultCode;
  pEntry->aCurrentKeyCode = defaultCode;
  pEntry->bConfigurable = configurable;
  pEntry->aAccelId = 0;
  pEntry->typeOfCode= type;
  pEntry->typeOfButton= button;
  pEntry->pConnectDict = NULL;
  pEntry->isEnabled= true;

  if ( !configurable )
    return false;

	/* search an entry in the KConfig object */
  pConfig->setGroup("CONFIG__IRC_keycodes");
  if ( !pConfig->hasKey(functionName) )
    return false;
	
	/* read and recognize the key */
  QStrList slist;
  QString sel;
  int num=pConfig->readListEntry(functionName,slist);
  sel= slist.first();
  pEntry->aCurrentKeyCode = sel.toULong();
  if ( num > 1) {
    sel= slist.next();
    pEntry->aCurrentKey= sel;
    sel= slist.next();
    pEntry->typeOfButton= sel.toInt() ? KCodeEntry::once : KCodeEntry::often;
  }    

#ifdef kDEBUG_A
  debug("kirc: read %d, %s, %d",(int)pEntry->aCurrentKeyCode,
	(const char*)pEntry->aCurrentKey,pEntry->typeOfButton==KCodeEntry::once?1:0);
#endif

  return false;
}

// =================================================================================================

void KIRCWidget::removeCode( const QString& functionName )
{
  /* search for an existing "functionName" entry */
  KCodeEntry *pEntry = aCodeDict[ functionName ];
	
  if ( !pEntry ) 
    return;
	
  /* disconnect eventuals connections */
  if ( pEntry->aAccelId ) {
    QDictIterator<KCodeConnectEntry> aConnectIt(*pEntry->pConnectDict);
    aConnectIt.toFirst();
    while ( aConnectIt.current() ) {
      disconnectFunction( aConnectIt.currentKey(), functionName );
      ++aConnectIt;
    }
  }
	
  aCodeDict.remove( functionName );
}

// =================================================================================================

unsigned int KIRCWidget::readCurrentKey( const QString& functionName )
{
  KCodeEntry *pEntry = aCodeDict[ functionName ];
	
  if ( !pEntry )
    return 0;
  else
    return pEntry->aCurrentKeyCode;
}

// =================================================================================================

unsigned int KIRCWidget::readDefaultKey( const QString& functionName )
{
  KCodeEntry *pEntry = aCodeDict[ functionName ];
	
  if ( !pEntry )
    return 0;
  else
    return pEntry->aDefaultKeyCode;
}

// =================================================================================================

void KIRCWidget::registerWidget( const QString& widgetName, QWidget* currentWidget )
{
#ifdef kDEBUG
  debug("kirc: register widget %s",(const char *)widgetName);
#endif

  KIRCWidgetEntry *pWEntry = aWidgetDict[ widgetName ];
  if ( pWEntry ) {
    QString str;
    str.sprintf( "initKeyWidget : \"%s\" widget already initialized",
		 (const char *)widgetName );
    warning(str);
    return;
  }

  pWEntry = new KIRCWidgetEntry;
  aWidgetDict.insert( widgetName, pWEntry );
  pWEntry->sWidgetName= widgetName;
  pWEntry->pWidget = currentWidget;
#ifdef kDEBUG
  debug("kirc: register widget %s done",(const char *)widgetName);
#endif
}
// =================================================================================================

void KIRCWidget::connectFunction( const QString& widgetName, 
				  const QString& functionName,
				  const QObject* receiver, const char* member,
				  bool activate )
{
#ifdef kDEBUG_A
  debug("kirc: connect %s -> %s",(const char *)functionName,member);
#endif
  QString str;
	
  /* search for an existing "widgetName" entry */
  KIRCWidgetEntry *pWEntry = aWidgetDict[ widgetName ];
  if ( !pWEntry ) {
    str.sprintf( "connectFunction : \"%s\" widget has not been initialized",
		 (const char *)widgetName );
    warning(str);
    return;
  }

  /* search for an existing "functionName" entry */
  KCodeEntry *pEntry = aCodeDict[ functionName ];
  if ( !pEntry ) {
    str.sprintf( "connectFunction : \"%s\" function does not exist",
		 (const char *)functionName );
    warning(str);
    return;
  }

  if ( pEntry->typeOfCode != KCodeEntry::normal ) {
    str.sprintf( "connectFunction : a special key cannnot be connected",
		 (const char *)functionName );
    warning(str);
    return;
  }
	
  KCodeConnectEntry *pCEntry;
	
  /* if the widget is already connected */
  if ( pEntry->pConnectDict ) {
    pCEntry = (*pEntry->pConnectDict)[ widgetName ];
    if ( pCEntry )
      //      internalDisconnectFunction( widgetName, pWEntry, pEntry, pCEntry );
      pEntry->pConnectDict->remove( widgetName );
  } else {
    pEntry->pConnectDict = new QDict<KCodeConnectEntry>(13);
    pEntry->pConnectDict->setAutoDelete(TRUE);
    pEntry->aAccelId = aAvailableId;
    aAvailableId++;
  }
		
  pCEntry = new KCodeConnectEntry;
  pCEntry->pReceiver = (QObject *)receiver;
  pCEntry->sMember = member;
  pCEntry->sig= new QSignal;
  pCEntry->sig->connect( receiver, member);
  pEntry->pConnectDict->insert( widgetName, pCEntry );

  /*	
  pWEntry->createItem( pEntry->aAccelId, pEntry->aCurrentKeyCode,
		       (QObject *)receiver, member );
		       */

  if ( !activate )
    toggleFunction( widgetName, functionName, FALSE );

}

// =================================================================================================

void KIRCWidget::toggleFunction( const QString& widgetName,const QString& functionName, bool activate )
{	
  QString str;
	
  /* search for an existing "widgetName" entry */
  KIRCWidgetEntry *pWEntry = aWidgetDict[ widgetName ];
  if ( !pWEntry ) {
    str.sprintf( "(dis)activateFunction : \"%s\" widget has not been initialized",
		 (const char *)widgetName );
    warning(str);
    return;
  }
	
  /* search for an existing "functionName" entry */
  KCodeEntry *pEntry = aCodeDict[ functionName ];
  if ( !pEntry ) {
    str.sprintf( "(dis)activateFunction : \"%s\" function does not exist",
		 (const char *)functionName );
    warning(str);
    return;
  }

  pEntry->isEnabled= activate;
}

// =================================================================================================

void KIRCWidget::disconnectFunction( const QString& widgetName,
				     const QString& functionName )
{
  /* search for an existing "widgetName" entry */
  KIRCWidgetEntry *pWEntry = aWidgetDict[ widgetName ];
  if ( !pWEntry ) {
    QString str;
    str.sprintf( "disconnectFunction : \"%s\" widget has not been initialized",
		 (const char *)widgetName );
    warning(str);
    return;
  }
	
  /* search for an existing "functionName" entry */
  KCodeEntry *pEntry = aCodeDict[ functionName ];
  if ( !pEntry ) 
    return;
  /* search for a connection */
  KCodeConnectEntry *pCEntry = (*pEntry->pConnectDict)[ widgetName ];
  if ( !pCEntry ) 
    return;
  pCEntry->sig->disconnect( pCEntry->pReceiver, pCEntry->sMember );
  delete pCEntry->sig;

  //internalDisconnectFunction( widgetName, pWEntry, pEntry, pCEntry );
  // anstatt
  pEntry->pConnectDict->remove( widgetName );

  /* if this was the unique connection of the functionName : delete the
	   pConnectDict */
  if ( pEntry->pConnectDict->count()==0 )
    delete pEntry->pConnectDict;
}

// =================================================================================================
/*
void KIRCWidget::internalDisconnectFunction( const QString& widgetName, 
					     KIRCWidgetEntry *pWEntry, KCodeEntry *pEntry, KCodeConnectEntry *pCEntry )
{
  pWEntry->deleteItem( pEntry->aAccelId, pCEntry->pReceiver,
		       pCEntry->sMember );
  pEntry->pConnectDict->remove( widgetName );
}
*/
// =================================================================================================

void KIRCWidget::disconnectAllFunctions( const QString& widgetName )
{
  QDictIterator<KCodeEntry> aKeyIt(aCodeDict);
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    disconnectFunction( widgetName, aKeyIt.currentKey() );
    ++aKeyIt;
  }
}

// =================================================================================================
void KIRCWidget::destroyWidgetEntry( const QString& widgetName )
{
  aWidgetDict.remove( widgetName );
}
// =================================================================================================

void KIRCWidget::internalDisconnectAll( const QString& widgetName)
{
  QDictIterator<KCodeEntry> aKeyIt(aCodeDict); 
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    if ( aKeyIt.current()->pConnectDict )
      if ( (*aKeyIt.current()->pConnectDict)[ widgetName ] )
	aKeyIt.current()->pConnectDict->remove( widgetName );
    ++aKeyIt;
  }
}

// =================================================================================================

void KIRCWidget::sync()
{
  /* write the current values in aCodeDict to KConfig object */
  pConfig->setGroup("CONFIG__IRC_keycodes");
  QString dum;
  QDictIterator<KCodeEntry> aKeyIt(aCodeDict);
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    if ( aKeyIt.current()->bConfigurable ) {
      dum.sprintf("%d,",aKeyIt.current()->aCurrentKeyCode);
      dum+=(const char*)aKeyIt.current()->aCurrentKey;
      dum+=(aKeyIt.current()->typeOfButton==KCodeEntry::once)?",1":",0";
#ifdef kDEBUG_A
      debug("kirc: %s, %s",(const char *)dum,(const char*)aKeyIt.current()->aCurrentKey);
#endif
      pConfig->writeEntry( aKeyIt.currentKey(),(const char*)dum );
    }
    ++aKeyIt;
  }
#ifdef kDEBUG
  debug("kirc: KIRCWidget::sync()");
#endif
}

// =================================================================================================

bool KIRCWidget::configureCodes( QWidget *parent )
{
#ifdef kDEBUG
  debug("kirc: Config start ");
#endif
  
  QDictIterator<KCodeEntry> aKeyIt(aCodeDict); 
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    aKeyIt.current()->aConfigKeyCode=aKeyIt.current()->aCurrentKeyCode;
    aKeyIt.current()->aConfigKey=aKeyIt.current()->aCurrentKey;
    ++aKeyIt;
  }

  kc->newDictSLOT( &aCodeDict );
  kc->show();
  configMode=true;

#ifdef kDEBUG
  debug("kirc: Config end ");
#endif
  return true;
}

// =================================================================================================
// =================================================================================================

KIRCConfig::KIRCConfig(QDict<KCodeEntry> *_aCodeDict, QWidget * parent = 0, const char * name = 0 )
  : aCodeDict(_aCodeDict), QWidget( parent, name )
{
#ifdef kDEBUG
  debug("kircConfig: KIRCConfig");
#endif
  setCaption("kWinTV - Infrared Remote Control Setup");

  lv= new QListView( this );
  lv->addColumn("Function");
  lv->setColumnWidthMode(0,QListView::Manual);
  lv->setColumnWidth(0,150);
  lv->addColumn("Code");
  lv->setColumnWidthMode(1,QListView::Manual);
  lv->setColumnWidth(1,50);
  lv->addColumn("Button");
  lv->setColumnWidthMode(2,QListView::Manual);
  lv->setColumnWidth(2,125);
  lv->addColumn("Mode");
  lv->setColumnWidthMode(1,QListView::Manual);
  lv->setColumnWidth(3,45);

  lv->setFrameStyle( 51 );
  lv->setLineWidth( 2 );
  lv->setMultiSelection ( false );
  lv->setAllColumnsShowFocus( true );

  newDictSLOT( _aCodeDict );

  resize(400,300);
  setMinimumSize( 400,300 );
  setMaximumSize( 400,300 );

  lv->setGeometry(5,5,390,200);
  connect( lv, SIGNAL( selectionChanged ( QListViewItem * ) ),this, SLOT(selectionChangedSLOT( QListViewItem * )) );

  QFrame* dlgedit_Frame_1;
  dlgedit_Frame_1 = new QFrame( this, "Frame_1" );
  dlgedit_Frame_1->setGeometry( 5, height() - 80 - 10, 390, 5 );
  dlgedit_Frame_1->setMinimumSize( 10, 10 );
  dlgedit_Frame_1->setMaximumSize( 32767, 32767 );
  dlgedit_Frame_1->setFrameStyle( 52 );

  le= new QLineEdit( this );
  le->setGeometry( 230,height() - 80 , 150 ,30);
  connect( le, SIGNAL(textChanged ( const char * ) ), this, SLOT(textChangedSLOT( const char *)));

  QLabel *label= new QLabel( "Enter IRC code or press IRC button:", this );
  label->setGeometry( 20 ,height() - 80 , 200 ,30);
  
  QFrame* dlgedit_Frame_2;
  dlgedit_Frame_2 = new QFrame( this, "Frame_1" );
  dlgedit_Frame_2->setGeometry( 5, height() - 50 , 390, 5 );
  dlgedit_Frame_2->setMinimumSize( 10, 10 );
  dlgedit_Frame_2->setMaximumSize( 32767, 32767 );
  dlgedit_Frame_2->setFrameStyle( 52 );

  QPushButton *b1= new QPushButton( "OK", this );
  b1->setGeometry(  width()-320, height()-30, 100, 25);
  connect( b1,SIGNAL( clicked () ), this, SLOT( applySLOT() ));

  QPushButton *b2= new QPushButton( "Default", this );
  b2->setGeometry(  width()-210,height()-30, 100, 25);
  connect( b2,SIGNAL( clicked () ), this, SLOT( defaultsSLOT() ));

  QPushButton *b3= new QPushButton( "Cancel", this );
  b3->setGeometry( width()-100, height()-30, 90, 25);
  connect( b3,SIGNAL( clicked () ), this, SLOT( cancelSLOT() ));

  newDictInsert=false;
  selChanged=false;
}

// =================================================================================================

KIRCConfig::~KIRCConfig()
{
  delete lv;
}

// =================================================================================================

void KIRCConfig::newDictSLOT( QDict<KCodeEntry> *_aCodeDict )
{
#ifdef kDEBUG_A
  debug("kircConfig: newDict begin");
#endif
  char btype[255];

  newDictInsert=true;

  lv->clear();
  aCodeDict= _aCodeDict;
  QDictIterator<KCodeEntry> aKeyIt(*aCodeDict);
  aKeyIt.toFirst();
  while ( aKeyIt.current() ) {
    QString code;
    code.sprintf("%d",(long unsigned int) aKeyIt.current()->aConfigKeyCode);
    QString key(aKeyIt.currentKey());
    if ( aKeyIt.current()->typeOfCode == KCodeEntry::number )
      key= QString("number key \"")+key+QString("\"");
    if ( ( aKeyIt.current()->typeOfCode == KCodeEntry::twoDigits ) ||
	 ( aKeyIt.current()->typeOfCode == KCodeEntry::up10 ) ||
	 ( aKeyIt.current()->typeOfCode == KCodeEntry::up20 ) ||
	 ( aKeyIt.current()->typeOfCode == KCodeEntry::up30 ) )
      key= QString("special key \"")+key+QString("\"");
    strcpy(btype,aKeyIt.current()->typeOfButton == KCodeEntry::once ? "t" : "c" );
    (void) new QListViewItem( lv, 
			      (const char*)key, 
			      (const char*)code, 
			      (const char*)aKeyIt.current()->aConfigKey,
			      btype);
    ++aKeyIt;
  }

#ifdef kDEBUG_A
  debug("kircConfig: newDict end");
#endif

  newDictInsert=false;

  currentItem= lv->firstChild ();
  lv->setSelected ( currentItem , true );

  aKeyIt.toFirst();
  currentCode= aKeyIt.current();

#ifdef kDEBUG
  debug("kircConfig: newDict ready");
#endif
}

// =================================================================================================

void KIRCConfig::applySLOT()
{
  emit apply();
}

// =================================================================================================

void KIRCConfig::defaultsSLOT()
{
#ifdef kDEBUG_A
  debug("kircConfig: default");
#endif
  currentCode->aConfigKeyCode=currentCode->aCurrentKeyCode;
  currentCode->aConfigKey=currentCode->aCurrentKey;

  QString text( currentCode->aCurrentKey );
  //  text.setNum( (long unsigned int) currentCode->aConfigKeyCode );
  le->setText( (const char*) text );
  
  text.sprintf("%d",(long unsigned int) currentCode->aConfigKeyCode);
  currentItem->setText( 1, text );
  currentItem->setText( 2, (const char *)currentCode->aCurrentKey);
}

// =================================================================================================

void KIRCConfig::cancelSLOT()
{
  emit cancel();
}

// =================================================================================================

void KIRCConfig::receiveCodeSLOT( ullong code, char * key, char * config )
{
#ifdef kDEBUG
  debug("kircConfig: data: %d, %s, %s",(int)code,key,config);
#endif
  currentCode->aConfigKeyCode= code;
  currentCode->aConfigKey= key;

  QString text;
  text.setNum( (long unsigned int) code);
  le->setText( (const char*) text );
  currentItem->setText( 1, (const char*) text );
  currentItem->setText( 2, (const char*) key );
}

// =================================================================================================

void KIRCConfig::selectionChangedSLOT( QListViewItem * item )
{
  if (newDictInsert) return;

#ifdef kDEBUG
  debug("kircConfig: select <%s>",item->text(0));
#endif

  currentItem= item;
  QString sitem( item->text(0) );
  if ( -1 != sitem.find('\"')) {
    sitem.remove(0,sitem.find('\"')+1);
    sitem.remove(sitem.find('\"'),sitem.length()-sitem.find('\"'));
    currentCode= (*aCodeDict)[ (const char*) sitem ];
  } else {
    currentCode= (*aCodeDict)[ (const char*) item->text(0) ];
  }
  QString tmp;
  tmp.setNum( (long unsigned int) currentCode->aConfigKeyCode );
  selChanged=true;
  le->setText( (const char*) tmp );
  selChanged=false;
  //le->setText( (const char*) currentCode->aConfigKey );
}

// =================================================================================================

void KIRCConfig::textChangedSLOT( const char * text )
{
  currentCode->aConfigKeyCode=atol( text );
  QString code;
  code.sprintf("%d",(long unsigned int) currentCode->aConfigKeyCode);
  currentItem->setText( 1, code );
  if ( !selChanged ) currentItem->setText( 2, "" );
}

// =================================================================================================
// =================================================================================================

KIRCClient::KIRCClient( const char * _device ) : device(_device)
{
  key= new char[CODE_LENGTH];
  config= new char[CODE_LENGTH];
  status= connect_client();
}

bool KIRCClient::connect_client()
{
  struct sockaddr_un sa;     // lirc 0.5.3 uses UNIX sockets
  sa.sun_family=AF_UNIX;
  strcpy(sa.sun_path,device);

  sockfd = socket( AF_UNIX, SOCK_STREAM, 0 );
  if ( sockfd == -1 ) {
    warning("kwintv: Unable to create UNIX socket!");
    return false;
  }

  // make the socket non-blocking here, usually using fcntl( O_NONBLOCK )
  if ( ::connect( sockfd, (struct sockaddr *)&sa, sizeof(sa) ) == -1 ) {
#ifdef kircWARN
    warning("kwintv: Unable to connect to socket!");
    warning("kwintv: Device %s:  %s!",device,strerror(errno));
#endif
    return false;
  }
  
  sn = new QSocketNotifier( sockfd, QSocketNotifier::Read, this );
  QObject::connect( sn, SIGNAL(activated(int)),
		    this, SLOT(receiveDataFromSocketSLOT(int)) );

  if ( !sn->isEnabled() ) {
    warning("kwintv: Unable to open socket notifier!");
    return false;
  }

  return true;
}

// =================================================================================================

KIRCClient::~KIRCClient()
{
#ifdef kDEBUG
  debug("kirc: closing socket connection.");
#endif
  delete key;
  delete config;

  if ( -1 == ::close(sockfd) )
    warning("kwintv: Device %s:  %s!",device,strerror(errno));
}

// =================================================================================================

void KIRCClient::receiveDataFromSocketSLOT( int socket )
{
  if ( ::read(socket,buf,128) == -1 ) 
    warning("kwint: Read error: %s",strerror(errno));

  sscanf( buf, "%lx %lx %s %s", &code, &count, key, config);

#ifdef kDEBUG
  debug("kric: receiveDataSLOT: %d %d %s %s",(int)code,(int)count,key,config);
#endif

  emit sendReceivedDataSIGNAL( code, count, key, config );
}

// =================================================================================================

bool KIRCClient::reConnect( const char * _device )
{
  device= _device;
  status= connect_client();
  return status;
}

// =================================================================================================
// =================================================================================================


#include "kirc.moc"

#endif // HAVE_IRC


