#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <dspout.h>

dspout::dspout( int devnum )
	: audio_buf(NULL)
#ifdef USE_FLOATS
	, output_buf(NULL)
#endif
{
	// Create filename from devnum
	if ( devnum == -1 ) {
		strcpy( devname, "/dev/dsp" );
	} else {
		sprintf( devname, "/dev/dsp%d", devnum );
	}

	audio_fd = open( devname, O_WRONLY | O_NONBLOCK );

	if ( audio_fd == -1 ) {
		if ( errno == EBUSY ) {
			lasterror = DeviceBusy;
		} else {
			lasterror = DeviceNoGood;
		}
	} else {
		lasterror = NoError;
	}
}

dspout::dspout( const char *filename )
	: audio_buf(NULL)
#ifdef USE_FLOATS
	, output_buf(NULL)
#endif
{
	audio_fd = open( filename, O_WRONLY | O_NONBLOCK, 0 );

	if ( audio_fd == -1 ) {
		if ( errno == EBUSY ) {
			lasterror = DeviceBusy;
		} else {
			lasterror = DeviceNoGood;
		}
	} else {
		lasterror = NoError;
	}
}
dspout::~dspout() {
	if ( audio_fd != -1 ) {
		close( audio_fd );
	}

	delete[] audio_buf;
#ifdef USE_FLOATS
	delete[] output_buf;
#endif
}

void dspout::setFragmentInfo( unsigned int numfrags, unsigned int size ) {
	audio_buf_info info;
	int frag = ( ( ( numfrags - 1 ) << 16 ) | size );

	if ( ioctl( audio_fd, SNDCTL_DSP_SETFRAGMENT, &frag ) ) {
		printf( "dspout::setFragmentInfo(): Failed to set audio device fragment size (%u)\n", frag);
		exit( 1 );
	}

	if ( ioctl( audio_fd, SNDCTL_DSP_GETOSPACE, &info ) == -1 ) {
		perror( "dspout::writeableFragments(): Couldn't get buffer info" );
		exit( 1 );
	}

	totalfrags = info.fragstotal;
	gotfrags = info.fragments;
	buffsize = info.fragsize;
	buffsize = buffsize / sizeof( sample_t );

	audio_buf = new sample_t[ buffsize ];
#ifdef USE_FLOATS
	output_buf = new SWORD[ buffsize ];
#endif
}

void dspout::setFormat( int fcode ) {
	format = fcode;

	if ( ioctl( audio_fd, SNDCTL_DSP_SETFMT, &format ) == -1 ) {
		perror( "dspout::setFormat(): Failed to set audio format" );
		exit( 1 );
	}

	if ( format != fcode ) {
		printf( "dspout::setFormat(): Format not supported\n" );
		exit( 1 );
	}
}

void dspout::setChannels( int sstereo) {
	int stereo_return = sstereo;

	if ( ioctl( audio_fd, SNDCTL_DSP_STEREO, &stereo_return ) == -1 ) {
		perror( "dspout::setChannels(): Failed to set number of channels" );
		exit( 1 );
	}

	if ( stereo_return != sstereo ) {
		printf( "dspout::setChannels(): Failed to set requested number of channels\n" );
	}

	channels = stereo_return;
}

void dspout::setSampleRate( int rrate ) {
	rate = rrate;

	if ( ioctl( audio_fd, SNDCTL_DSP_SPEED, &rate ) == -1 ) {
		perror( "dspout::setSampleRate(): Failed to set rate" );
		exit( 1 );
	}
}

void dspout::printInfo() {
	int frag_size;

	if ( ioctl( audio_fd, SNDCTL_DSP_GETBLKSIZE, &frag_size ) == -1 ) {
		perror( "dspout::printInfo(): I couldnt check the frag size" );
		exit( 1 );
	}

	printf( "Total frags available is %d\n", totalfrags );
	printf( "Allocated frags %d\n", gotfrags );
	printf( "Frag size in bytes is %u\n", frag_size );
	printf( "Buffer size in samples is %u\n", buffsize );
	printf( "Sample rate is %u\n", rate );
}

void dspout::writeBuffer() {

#ifdef USE_FLOATS
	sample_t *inbuf;
	SWORD *outbuf;
	sample_t iterate;

	inbuf = audio_buf;
	outbuf = output_buf;

	for ( int i = 0; i < buffsize; i++ ) {
		iterate = *inbuf;
		inbuf++;

		if ( iterate > 32767.0 ) { iterate = 32767.0; }
		if ( iterate < -32768.0 ) { iterate = -32768.0; }
		*outbuf = (SWORD) iterate;
		outbuf++;
	}

	for (;;) {
		if ( write( audio_fd, output_buf, buffsize * sizeof( SWORD ) ) != -1 ) {
			break;
		}
	}
#else
	for (;;) {
		if ( write( audio_fd, audio_buf, buffsize * sizeof( sample_t ) ) != -1 ) {
			break;
		}
	}
#endif

}

void dspout::flushBuffer() {
	ioctl( audio_fd, SNDCTL_DSP_POST );
}

unsigned int dspout::writeableFragments() const
{
	audio_buf_info info;

	if ( ioctl( audio_fd, SNDCTL_DSP_GETOSPACE, &info ) == -1 ) {
		perror( "dspout::writeableFragments(): Couldn't get buffer info" );
		exit( 1 );
	}

	return info.fragments;
}

unsigned int dspout::bytesPlayed() const
{
	count_info info;

	ioctl( audio_fd, SNDCTL_DSP_GETOPTR, &info );

	return( info.bytes );
}

int dspout::numDevs()
{
	char devname[ 20 ];
	int i, fd;

	i = 0;
	for (;;) {
		sprintf( devname, "/dev/dsp%d", i++ );
		fd = open( devname, O_WRONLY | O_NONBLOCK );
		if ( fd == -1 ) {
			i--;
			break;
		}
		close( fd );
	}
	return( i );
}

