// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/MacroModule.h,v $
// $Revision: 1.3 $
// $Date: 1999/05/15 22:48:23 $
// $State: Exp $
// **************************************************************

#ifndef MacroModule_h
#define MacroModule_h

#include "Module.h"
#include "Wire.h"
#include "EditorInfo.h"
#include "LookInfo.h"

/** 
 * @short Combines an arbitrary number of modules to form a more complex @ref Module.
 *
 * A MacroModule behaves exactly like any other module. Its only but
 * great difference is how it's implemented. As it's implementation it uses
 * a network of Modules, which are interconnected by Wires. Some of these
 * Sub-Modules may have Connectors with the special property "External". These
 * Connectors appear as the Connectors of the MacroModule and may not be wired
 * internally. 
 *
 * Another property of a MacroModule is that it may be saved to and loaded from
 * a file.
 */

class MacroModule : public Module
{
    /**
     * If embedding_type == ET_FILEREF then the name of the file containing
     * the MacroModule's structure is kept here.
     */
    string filename;

    /**
     * List of all Submodules.
     */
    DLList modules;

    /**
     * List of all Wires. The wires connect the submodules. Each wire goes from
     * the Signal/Slot of one submodule to the Slot/Signal of another (or the same).
     */
    DLList wires;    // Wire live here

    /**
     * Information about the MacroModuleEditor that is contained in the
     * Module file. The class MacroModule doesn't really understand this
     * data, but it will conserve it while loading and saving in files.
     * It provides this information to the MacroModuleEditor. The MMM Player
     * is not interested in this data.  
     */
    EditorInfo editor_info;

    /**
     * Information about how the "frontpanel" of this MacroModule looks.
     * The class MacroModule doesn't really understand this
     * data, but it will conserve it while loading and saving in files.
     * It provides this information to the MacroModuleEditor and to
     * MacroModuleLook. The MMM Player is not interested in this data.
     */
    LookInfo look_info;

    /**
     * After 'loadFromFile' was called, this stores a pointer to the last submodule before the first
     * newly loaded submodule. See also @ref #nextNewModule.
     */
    Module *last_old_module;

    /**
     * After 'loadFromFile' was called, this stores a pointer to the last Wire before the first
     * newly loaded Wire. See also @ref #nextNewWire.
     */
    Wire *last_old_wire;

public:

    /**
     * The embedding type describes how this MacroModule behaves when it is
     * itself part of another MacroModule. This is essential when being saved
     * to a file as part of the saving of the upper MacroModule.
     *
     * ET_TOPLEVEL: This module is not embedded within a MacroModule but has been loaded
     *              directly.
     *
     * ET_PLUGIN:   This module is embedded and has been loaded from a file in the Plugins-
     *              Directory. It's internal structure is refered to by a unique module name.
     *
     * ET_INLINE:   This module is embedded and will be saved completely within the module
     *              it's embedded in. No external references exist. However, it's possible
     *              and likely that the inline embedded module in turn contains MacroModules
     *              of other embedding types and thus MAY indirectly contain external references.
     *
     * ET_FILEREF:  This module is embedded but its internal structure will be referenced by a 
     *              filename. When loading this module, the referenced file will be opened.
     */
    enum EmbeddingType { ET_TOPLEVEL, ET_PLUGIN, ET_INLINE, ET_FILEREF };

private:
    EmbeddingType embedding_type;

public:

    /**
     * Creates a new MacroModule. If imf=0 this will be an empty MacroModule.
     */
    MacroModule(EmbeddingType embedding_type, string Parameters, InputModuleFile *modulefile=0);

    /**
     * Cleans up. Deletes modules and wires.
     */
    ~MacroModule();

    /**
     * Creates a Stringrepresentation of the parameters of this
     * Module. Defined in Module.
     */
    string toString() const;

    /**
     * @return The name of this Module. The name must be unique.
     */
    string getName() const { return look_info.name; };

    /**
     * Alters the name of this Module.
     */
    void setName(string n) { look_info.name = n; };
    
    /**
     * @return A Description what this module does
     */
    string getDescription() const { return look_info.description; };

    /**
     * Inherited from @ref Module. Determines if this module is executable.
     * A MacroModule is executable, iff it contains an executable Module.
     */
    bool isExecutable() const;

    /**
     * Inherited from @ref Module. Prepare this Module for subsequent calls
     * of @ref #executeBlock.
     * @param sampling_interval Sampling interval (1 / sampling rate) with
     * that sound signals will be processed.
     */
    void prepareForExecution(const SamplingConfig *);

    /**
     * Executes the next block of samples. Returns false when the execution
     * has finished.
     */
    bool executeBlock(long start_time, long nsamples);

    /**
     * Tells this module that the execution is to be aborted. This MacroModule
     * tells the new to its executable submodules. They can close down, e.g.
     * close /dev/audio or open files etc.
     */
    void finishExecution();

    bool reportModuleErrors(string&);

    /**
     * Returns true, if this Module is embedded into another MacroModule.
     */
    bool isEmbedded() const;

    LookInfo& lookInfo() { return look_info; };

    EditorInfo& editorInfo() { return editor_info; };

    bool saveToFile(OutputModuleFile& modulefile, bool selection_only);
    void saveSubmodules(OutputModuleFile& modulefile, bool selection_only);
    void saveWires(OutputModuleFile& modulefile, bool selection_only);

    /**
     * Load the contents of this MacroModule from an open InputModuleFile. The pointers
     * last_old_module and last_old_wire are set. See also @ref #nextNewModule and @ref #nextNewWire.
     * @param modulefile load from this file
     * @param parameters Parameters for the submodules
     * @param insert If insert is true, then the modules contained in modulefile will be inserted
     * into the existing MacroModule. Other parameters like look_info and editor_info will remain
     * untouched.
     */
    bool     loadFromFile  (InputModuleFile& modulefile, string parameterlist, bool insert);
    Module **loadSubmodules(InputModuleFile& modulefile, string parameterlist, long& number_of_modules);
    Module  *loadSubmodule (InputModuleFile& modulefile, long index, string& parameterlist);
    void     loadWires     (InputModuleFile& modulefile, Module **module_table, int number_of_modules); 
    bool     loadWire      (InputModuleFile& modulefile, Module **module_table, int number_of_modules); 

    /**
     * This function allows an object or program, that is controlling and editing a MacroModule,
     * to keep track of all submodules this MacroModule is consisting of. The most promiment
     * such programm is the m3editor (class MacroModuleEditor). The method @ref #loadFromFile
     * stores a pointer into the doubly linked list to the first newly loaded submodule in
     * @ref last_old_module. A call to nextNewModule returns this pointer and moves it forward
     * to the next new Module thus allowing to scan all newly loaded submodules.
     * @return A pointer to a new Module or 0, if all new Modules have been scanned.
     */
    Module  *nextNewModule();

    /**
     * Works exaclty like @ref #nextNewModule, but returns all newly loaded Wires in turn.
     * @return A pointer to a new Wire or 0, if all new Wires have been scanned.
     */
    Wire    *nextNewWire();

    /**
     * Adds a new submodule to the module. If the submodule has connectors with
     * the property "external", those connectors become connectors of this
     * MacroModule.
     */
    void addModule(Module *module);

    /**
     * Removes a submodule from this MacroModule and deletes it.
     * @param module Module to remove. The module is assumed to be indeed a submodule
     * of this MacroModule. If module is 0, then an arbitrary Module will be removed.
     * @return true if successful. false is the module can not be removed,
     * e.g. because one of its Slots or Signals is wired.
     */
    void deleteModule(Module *module=0);

    /**
     * Removes and deletes all submodules and all wires.
     */
    void deleteWiresAndModules();

    /**
     * Returns the number of Submodules this MacroModule contains.
     */
    int numberOfModules() const;

    /**
     * Returns the number of submodules of this MacroModule that are
     * selected. (@ref Module#isSelected).
     */
    int numberOfSelectedModules() const;

    int numberOfWires() const;
			      
    int numberOfIntraSelectionWires() const; 

    /**
     * Checks, if a subsequent call to @ref addWire with the same arguments
     * would succeed.
     */
    bool checkWire(Connector *, Connector *);

    /**
     * Adds a wire between to Connectors. The Connectors must belong to 
     * submodules of this MacroModule. No error checking will be performed.
     * That must be done by calling @ref checkWire immediately before this
     * call. If both connectors are the same Slot than no Wire is created
     * but the slot is defined to be a parameter slot.
     * @param signal Signal to wire
     * @param slot Slot to wire
     * @return A pointer to the new @ref Wire or 0, if a parameter slot was
     * defined.
     */
    Wire *addWire(Connector *signal, Connector *slot, Module *signal_module, Module *slot_module);

    void deleteWire(Wire *);

    /**
     * Scans the plugin directory for valid Module files and
     * creates a ModuleGenerator for each.
     */
    static void scanPlugins(const char *pluginsdir);

private:

    bool isInline()   const;

    bool isFileRef()  const;

    bool isPlugin()   const;

};


#endif // MacroModule_h
