/*  
  start external programms an connect the stdin/stdout slots  
  Copyright (C) 1999  Martin Vogt,Christian Czezatke

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

*/


/*

  This file was originally written by Christian Czezatke, but
  was modified to abstract the signal mechanism (gtk port)

*/


#include <signal/process/yafProcess.h>
#define _MAY_INCLUDE_KPROCESSCONTROLLER_
#include <signal/process/yafProcessController.h>


#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/socket.h>
#include <fcntl.h>

#include <stdio.h>

extern YafProcessController *theYafProcessController;


YafProcess::YafProcess() {
  arguments.setAutoDelete(TRUE);
  
  if (0 == theYafProcessController) {
    theYafProcessController= new YafProcessController();
    CHECK_PTR(theYafProcessController);
  }
  
  run_mode = NotifyOnExit;
  runs = FALSE;
  pid = 0;
  status = 0;
  innot = outnot = errnot = 0;
  communication = NoCommunication;
  input_data = 0;
  input_sent = 0;
  input_total = 0;
  
  theYafProcessController->processList->append(this);
}



YafProcess::~YafProcess() {
  // destroying the YafProcess instance sends a SIGKILL to the
  // child process (if it is running) after removing it from the
  // list of valid processes (if the process is not started as
  // "dont_care")
  if (runs) {
    commClose(); // cleanup communication sockets
  }

  theYafProcessController->processList->remove(this);
  // this must happen before we kill the child
  // TODO: block the signal while removing the current process from the process list
  
  if (runs && (run_mode != DontCare))
    kill(SIGKILL);
}



bool YafProcess::setExecutable(const char *proc) {
  char *hlp;
  
  if (runs) return FALSE;

  arguments.removeFirst();
  if (0 != proc) {
    hlp = strdup(proc);
    CHECK_PTR(hlp);
    arguments.insert(0,hlp);
  }

  return TRUE;
}

 
 



YafProcess &YafProcess::operator<<(const char *arg) {
  char *new_arg= strdup(arg);

  CHECK_PTR(new_arg);
  arguments.append(new_arg);
  return *this;
}



void YafProcess::clearArguments() {
  if (0 != arguments.first()) {
    while (arguments.remove())
      ;
  }
}



bool YafProcess::start(RunMode runmode, Communication comm) {
  uint i;
  uint n = arguments.count();
  char **arglist;

  if (runs || (0 == n)) {
    return FALSE;  // cannot start a process that is already running
    // or if no executable has been assigned
  }
  run_mode = runmode; 
  status = 0;

  arglist = (char **)malloc( (n+1)*sizeof(char *));
  CHECK_PTR(arglist);
  for (i=0; i < n; i++)
    arglist[i] = arguments.at(i);
  arglist[n]= 0;

  if (!setupCommunication(comm))
    cout<<"Could not setup Communication!"<<endl;

  runs = TRUE;
  pid = fork();

  if (0 == pid) {
    // The child process
    
    if(!commSetupDoneC())
      cout<<"Could not finish comm setup in child!"<<endl;
    
    // Matthias
    if (run_mode == DontCare) 
      setpgid(0,0);
    
    if (execvp(arglist[0], arglist) < 0) {
      perror("execvp in yafProcess");
    }
    exit(-1);
    
  } else if (-1 == pid) {
    cout << "forking failed"<<endl;
    runs = FALSE;
    free(arglist);
    commClose();
    return FALSE;
    
  } else {
    // the parent continues here
    // finish communication socket setup for the parent   
    if (!commSetupDoneP())  
      cout<<"Could not finish comm setup in parent!"<<endl;
    
    // Discard any data for stdin that might still be there
    input_data = 0;
    
    if (run_mode == Block) {
      waitpid(pid, &status, 0);
      processHasExited(status);
    }
  }
  free(arglist);    
  return TRUE;
}



bool YafProcess::kill(int signo) {
  bool rv=FALSE;

  if (0 != pid)
    rv= (-1 != ::kill(pid, signo));
  // probably store errno somewhere...
  return rv;
}



bool YafProcess::isRunning() {
  return runs;
}



pid_t YafProcess::getPid() {
  return pid;
}



bool YafProcess::normalExit() {
  int _status = status;
  return (pid != 0) && (!runs) && (WIFEXITED(_status));
}



int YafProcess::exitStatus() {
  int _status = status;
  return WEXITSTATUS(_status);
}



bool YafProcess::writeStdin(char *buffer, int buflen) {
  bool rv;

  // if there is still data pending, writing new data
  // to stdout is not allowed (since it could also confuse
  // YafProcess... 
  if (0 != input_data)
    return FALSE;

  if ( runs && communication) {
    input_data = buffer;
    input_sent = 0;
    input_total = buflen;
    slotSendData(0);
    innot->setEnabled(TRUE);
    rv = TRUE;
  } else
    rv = FALSE;
  return rv;
}



bool YafProcess::closeStdin() {
  bool rv;

  if (communication & Stdin) {
    innot->setEnabled(FALSE);
    close(in[1]);
    rv = TRUE;
  } else
    rv = FALSE;
  return rv;
}


/////////////////////////////
// protected slots         //
/////////////////////////////



void YafProcess::slotChildOutput(int fdno) {
  int back=childOutput(fdno);
  if (!back) {
    outnot->setEnabled(FALSE);	
  }
}



void YafProcess::slotChildError(int fdno) {
  int back=childError(fdno);
  if (!back) {
    errnot->setEnabled(FALSE);
  }
}



void YafProcess::slotSendData(int) {
  if (input_sent == input_total) {
    innot->setEnabled(FALSE);
    input_data = 0;
    emit wroteStdin(this);
  } else {
    input_sent += ::write(in[1], input_data+input_sent, 
			  input_total-input_sent);
    if (input_sent == input_total) {
      slotSendData(0);
    }
  }
}



//////////////////////////////
// private member functions //
//////////////////////////////



void YafProcess::processHasExited(int state) {
  runs = FALSE;
  status = state;

  commClose(); // cleanup communication sockets
  
  // also emit a signal if the process was run Blocking
  if (DontCare != run_mode)
    emit processExited(this);
}



int YafProcess::childOutput(int fdno) {
  int len;

  len = ::read(fdno, childOutputBuffer, 1024);

  if ( 0 > len) {
    perror("YafProcess::childOutput");
    return 0;
  }

  if ( 0 < len) {
    emit receivedStdout(this, childOutputBuffer, len);
  }
  return len;
}



int YafProcess::childError(int fdno) {
  int len;

  len = ::read(fdno, childErrorBuffer, 1024);
  if ( 0 > len) {
    perror("YafProcess::childError");
    return 0;
  }
  if ( 0 < len) {
    emit receivedStderr(this,childErrorBuffer , len);
  }
  return len;
}



int YafProcess::setupCommunication(Communication comm) {
  int ok;

  communication = comm;
 
  ok = 1;
  if (comm & Stdin) 
	ok &= socketpair(AF_UNIX, SOCK_STREAM, 0, in) >= 0;
 
  if (comm & Stdout) 
	ok &= socketpair(AF_UNIX, SOCK_STREAM, 0, out) >= 0;

  if (comm & Stderr) 
	ok &= socketpair(AF_UNIX, SOCK_STREAM, 0, err) >= 0;

  return ok;
}



int YafProcess::commSetupDoneP() {
  int ok = 1;

  if (communication != NoCommunication) {
    if (communication & Stdin)
      close(in[0]);
    if (communication & Stdout)
      close(out[1]);
    if (communication & Stderr)
      close(err[1]); 
    
    if (communication & Stdin) {
      ok &= (-1 != fcntl(in[1], F_SETFL, O_NONBLOCK));
      innot =  new QSocketNotifier(in[1], QSocketNotifier::Write, this);
      CHECK_PTR(innot);
      innot->setEnabled(FALSE); // will be enabled when data has to be sent
      QObject::connect(innot, SIGNAL(activated(int)),
		       this, SLOT(slotSendData(int)));
    }
    
    if (communication & Stdout) {
      ok &= (-1 != fcntl(out[0], F_SETFL, O_NONBLOCK));
      outnot = new QSocketNotifier(out[0], QSocketNotifier::Read, this);
      CHECK_PTR(outnot);
      QObject::connect(outnot, SIGNAL(activated(int)),
		       this, SLOT(slotChildOutput(int)));
    }
    
    if (communication & Stderr) {
      ok &= (-1 != fcntl(err[0], F_SETFL, O_NONBLOCK));
      errnot = new QSocketNotifier(err[0], QSocketNotifier::Read, this );    
      CHECK_PTR(errnot);
      QObject::connect(errnot, SIGNAL(activated(int)),
		       this, SLOT(slotChildError(int)));
    }
  }
  return ok;
}



int YafProcess::commSetupDoneC() {
  int ok = 1;
  struct linger so;

  if (communication != NoCommunication) {
    if (communication & Stdin)
      close(in[1]);
    if (communication & Stdout)
      close(out[0]);
    if (communication & Stderr)
      close(err[0]);
    
    if (communication & Stdin) {
      ok &= dup2(in[0],  STDIN_FILENO) != -1;
    }
    
    if (communication & Stdout) {
      ok &= dup2(out[1], STDOUT_FILENO) != -1;    
      ok &= !setsockopt(out[1], SOL_SOCKET, SO_LINGER, (char*)&so, sizeof(so));
    }
    if (communication & Stderr) {
      ok &= dup2(err[1], STDERR_FILENO) != -1;
      ok &= !setsockopt(err[1], SOL_SOCKET, SO_LINGER, (char*)&so, sizeof(so));
    }
  }
  return ok;
}  



void YafProcess::commClose() {
  if (NoCommunication != communication) {
    if (communication & Stdin)
      delete innot;
    
    if (communication & Stdout) {
      delete outnot;   
      while(childOutput(out[0])> 0 )
	;
    }
    
    if (communication & Stderr) {
      delete errnot;
      while(childError(err[0]) > 0)
	;
    }      
    if (communication & Stdin)
      close(in[1]);
    if (communication & Stdout)
      close(out[0]);
    if (communication & Stderr)
      close(err[0]);
  }
}




