/*
  wrapper for X11 Window
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "x11Window.h"



static void *ditherThread(void *arg){
  ((X11Window*)arg)->ditherLoop();
  return NULL;
}   
static int lXerror;


static int dummy(Display *dpy, XErrorEvent *event) {
  lXerror=true;
  return true;
}



const char *ERR_XI_STR[] = {
  "X initialisation OK!",
  "No Shared Memory available", 
  "cannot open Display", 
  "bad color depth",
  "can't create Window", 
  "can't alloc memory for virtual screen",
  "cannot create XImage",
  "can't alloc memory for Shared memory segment info",
  "cannot create Shared Memory XImage",
  "Shared memory segment info error",
  "Shared memory virtual screen allocation failed",
  "cannot attach Shared Memory segment to display"
};



X11Window::X11Window(int lthreadDither) {
  XWnd=(XWindow *)malloc(sizeof(XWindow));
  XWnd->lOpen=false;
  currentPic=NULL;
  initDither();
  lDitherLoop=true;
  linDitherLoop=false;
  lPut=false;
  ditherTime=new TimeStamp();
  currentTime=new TimeStamp();
  spendTime=new TimeStamp();
  threadWaitTime=new TimeStamp();
  currentTimeMain=new TimeStamp();
  putTime=new TimeStamp();
  startTime=new TimeStamp();
  endTime=new TimeStamp();
  totalWaitTime=new TimeStamp();
  restWaitTime=new TimeStamp();

  pthread_mutex_init(&ditherMut,NULL);
  pthread_mutex_init(&changeMut,NULL);
  pthread_cond_init(&changeCond,NULL);
  this->lthreadDither=lthreadDither;
  if (lthreadDither) {
    pthread_create(&tr,NULL,ditherThread,this);
    while(linDitherLoop == false) {
      usleep(500);
    }
  }

#ifdef INTEL
  cout << "check mm support for dither"<<endl;
  lmmx=mm_support();
  if (lmmx) {
    cout << "Jau! Intel MMX in dither"<<endl;
  } else {
    cout << "Sorry folks, no MMX. in dither"<<endl;
  }
#else
  lmmx=false;
  cout << "no INTEL arch- disable MMX"<<endl;
#endif

}


X11Window::~X11Window() {
  void* ret;
  ditherLock();
  lDitherLoop=false;
  ditherUnlock();
  pthread_join(tr,&ret);
  delete ditherTime;
  delete currentTime;
  delete currentTimeMain;
  delete spendTime;
  delete putTime;

  cout << "~X11Window -s"<<endl;
  if (XWnd->lOpen) {
    closeWindow();
  }
  cout << "~X11Window -e"<<endl;

}

int X11Window::openWindow(int width, int height, 
			  char *title){

  int err;
  if (XWnd->lOpen) {
    // can we use the same window?
    if ((width == XWnd->width) && (height == XWnd->height)) {
      return true;
    }
    closeWindow();
  }  
  err=InitXWindow(width,height,title);
  if(err==true) {
    printf("\nX initialisation error:\n *** %s\n",ERR_XI_STR[err]);
    printf("check ipcs and delete resources with ipcrm\n");
    return ERR_XI_FAILURE;
  }



  // now check step by step the accessMethod
  int access=VIDEO_XI_STANDARD | VIDEO_XI_SHMSTD | VIDEO_XI_SHMPIXMAP;
  if (createImage(access) != ERR_XI_OK) {
    access=access & VIDEO_XI_SHMPIXMAP;
    if (createImage(access) != ERR_XI_OK) {
      access=access & VIDEO_XI_SHMSTD;
      if (createImage(access) != ERR_XI_OK) {
	cout << "could not create image"<<endl;
	return ERR_XI_FAILURE;
      }
    }
  }

  printf("\nX initialisation OK\n");
  switch(XWnd->videoaccesstype)
    {
    case VIDEO_XI_STANDARD:
      printf("  # using conventional Xlib calls.\n\n");
      break;
    case VIDEO_XI_SHMSTD:
      printf("  # Using Xlib shared memory extension %d.%d\n\n",
	     XShmMajor,XShmMinor);
      break;
    case VIDEO_XI_SHMPIXMAP:
      printf("  # Using shared memory pixmaps extension %d.%d\n\n",
	     XShmMajor,XShmMinor);
    }

  return ERR_XI_OK;
}


int X11Window::InitXWindow(int width, int height,char *title) {

  XWnd->width=width;
  XWnd->height=height;

  XWnd->display=XOpenDisplay(NULL);
  if(!XWnd->display)
    return ERR_XI_DISPLAY;

  XWnd->screennum=DefaultScreen(XWnd->display);
  XWnd->screenptr=DefaultScreenOfDisplay(XWnd->display);
  XWnd->visual=DefaultVisualOfScreen(XWnd->screenptr);
  XWnd->depth=DefaultDepth(XWnd->display,XWnd->screennum);

  switch(XWnd->depth) {
    case 8:
      XWnd->pixelsize=1;
      break;
    case 16:
      XWnd->pixelsize=2;
      break;
    case 24:
      XWnd->pixelsize=4;
      break;
    case 32:
      XWnd->pixelsize=4;
      break;
   default:
     cout << "unknown pixelsize for depth:"<<XWnd->depth<<endl;
     exit(0);
  }
  XSetWindowAttributes attributes;
  attributes.backing_store=NotUseful;

  XWnd->window=XCreateWindow(XWnd->display,RootWindowOfScreen(XWnd->screenptr),
			     0,0,XWnd->width, XWnd->height,0,XWnd->depth,
			     InputOutput, XWnd->visual,
			     CWBackingStore ,
			     &attributes);
 
  if(!XWnd->window)
    return ERR_XI_WINDOW;
  {
    XSizeHints Hints;

    Hints.flags=PSize|PMinSize|PMaxSize;
    Hints.min_width=Hints.max_width=Hints.base_width=width;
    Hints.min_height=Hints.max_height=Hints.base_height=height;
    XSetWMNormalHints(XWnd->display,XWnd->window,&Hints);
    XStoreName(XWnd->display,XWnd->window,title);
  }
  XSelectInput(XWnd->display,XWnd->window,
	       ExposureMask|KeyPressMask|KeyReleaseMask|ButtonPressMask);
  XWnd->gc=XCreateGC(XWnd->display,XWnd->window,0,NULL);
  XMapRaised(XWnd->display,XWnd->window);



  if (XWnd->depth >= 16) {
    initColorDisplay(XWnd);
    initColorDither(XWnd->depth>=24);
  } else {
    // depth is <= 8
    initColor();
    initOrderedDither();
    initSimpleDisplay(XWnd);
  }

  XWnd->palette=NULL;
  XWnd->screensize=XWnd->height*XWnd->width*XWnd->pixelsize;
  return ERR_XI_OK;
}


int X11Window::createImage(int accessAllow) {


  XWnd->videoaccesstype=VIDEO_XI_STANDARD;
  XWnd->videomemory=NULL;

#ifdef __USE_X_SHAREDMEMORY__

  if(XShmQueryVersion(XWnd->display,&XShmMajor,&XShmMinor,&XShmPixmaps)) {
    if(XShmPixmaps==True && XShmPixmapFormat(XWnd->display)==ZPixmap) {
      if (accessAllow & VIDEO_XI_SHMPIXMAP) {
	XWnd->videoaccesstype=VIDEO_XI_SHMPIXMAP;
      } else {
	if (accessAllow & VIDEO_XI_SHMPIXMAP) {
	  XWnd->videoaccesstype=VIDEO_XI_SHMSTD;
	}
      }
    } else {
      if (XShmPixmaps==True) {
	if (accessAllow & VIDEO_XI_SHMPIXMAP) {
	  XWnd->videoaccesstype=VIDEO_XI_SHMSTD;
	}
      }
    }
  }
#endif

  switch(XWnd->videoaccesstype)
    {
#ifdef __USE_X_SHAREDMEMORY__

    case VIDEO_XI_SHMPIXMAP:

      XWnd->shmseginfo=(XShmSegmentInfo *)malloc(sizeof(XShmSegmentInfo));
      if(!XWnd->shmseginfo)
	return ERR_XI_SHMALLOC;

      memset(XWnd->shmseginfo,0,sizeof(XShmSegmentInfo));
      cout << "XWnd->screensize:"<<XWnd->screensize<<endl;
      XWnd->shmseginfo->shmid=shmget(IPC_PRIVATE,
				     XWnd->screensize,IPC_CREAT|0777);
      if(XWnd->shmseginfo->shmid<0)
	return ERR_XI_SHMSEGINFO;

      XWnd->shmseginfo->shmaddr=(char*)shmat(XWnd->shmseginfo->shmid,0,0);
      XWnd->videomemory=XWnd->shmseginfo->shmaddr;

      XWnd->virtualscreen=XWnd->videomemory;

      XWnd->shmseginfo->readOnly=False;
      lXerror=false;
      XSetErrorHandler(dummy);
      XShmAttach(XWnd->display,XWnd->shmseginfo);
      XSync(XWnd->display, False);
      XSetErrorHandler(NULL);
      XFlush(XWnd->display);
      if (lXerror) {
	cout << "ERR_XI_SHMATTACH"<<endl;
	return ERR_XI_SHMATTACH;
      }

      XWnd->pixmap=XShmCreatePixmap(XWnd->display,XWnd->window,
				    XWnd->videomemory,XWnd->shmseginfo,
				    XWnd->width,XWnd->height,XWnd->depth);
      XSetWindowBackgroundPixmap(XWnd->display,XWnd->window,XWnd->pixmap);
      break;

    case VIDEO_XI_SHMSTD:
      XWnd->shmseginfo=(XShmSegmentInfo *)malloc(sizeof(XShmSegmentInfo));
      if(!XWnd->shmseginfo)
	return ERR_XI_SHMALLOC;

      memset(XWnd->shmseginfo,0, sizeof(XShmSegmentInfo));

      XWnd->ximage=XShmCreateImage(XWnd->display,XWnd->visual,XWnd->depth,
				   ZPixmap,NULL,XWnd->shmseginfo,XWnd->width,
				   XWnd->height);
      if(!XWnd->ximage)
	return ERR_XI_SHMXIMAGE;

      cout << "XWnd->ximage->bytes_per_line:"
	   << XWnd->ximage->bytes_per_line*XWnd->ximage->height<<endl;

      XWnd->shmseginfo->shmid=shmget(IPC_PRIVATE,
				     XWnd->ximage->bytes_per_line*
				     XWnd->ximage->height,IPC_CREAT|0777);
      if(XWnd->shmseginfo->shmid<0)
	return ERR_XI_SHMSEGINFO;
      
      XWnd->shmseginfo->shmaddr=(char*)shmat(XWnd->shmseginfo->shmid,NULL,0);
      XWnd->ximage->data=XWnd->shmseginfo->shmaddr;
      XWnd->virtualscreen=XWnd->ximage->data;

      if(!XWnd->virtualscreen)
	return ERR_XI_SHMVIRTALLOC;

      XWnd->shmseginfo->readOnly=False;
      
      XSetErrorHandler(dummy);
      XShmAttach(XWnd->display,XWnd->shmseginfo);
      XSync(XWnd->display, False);
      XSetErrorHandler(NULL);
      XFlush(XWnd->display);
      if (lXerror) {
	cout << "ERR_XI_SHMATTACH -2"<<endl;
	return ERR_XI_SHMATTACH;
      }

      break;
#endif

    case VIDEO_XI_STANDARD:
      XWnd->virtualscreen=(char*)malloc(XWnd->screensize*sizeof(char));
      if(!XWnd->virtualscreen)
	return ERR_XI_VIRTALLOC;
      XWnd->ximage=XCreateImage(XWnd->display,XWnd->visual,XWnd->depth,ZPixmap,
				0,XWnd->virtualscreen,XWnd->width,XWnd->height,
				32,XWnd->width*XWnd->pixelsize);
      if(!XWnd->ximage)
	return ERR_XI_XIMAGE;
      break;
    }
  if ( (XWnd->videoaccesstype == VIDEO_XI_STANDARD) ||
       (XWnd->videoaccesstype == VIDEO_XI_SHMSTD) ) {
#ifndef WORDS_BIGENDIAN
    XWnd->ximage->byte_order = LSBFirst;
    XWnd->ximage->bitmap_bit_order = LSBFirst;
#else
    XWnd->ximage->byte_order = MSBFirst;
    XWnd->ximage->bitmap_bit_order = MSBFirst;
#endif
    
  }
  XWnd->lOpen=true;
  return ERR_XI_OK;
}

void X11Window::flushWindow() {
  cout << "X11Window::flushWindow"<<endl;
  ditherLock();
  ditherUnlock();
}

void X11Window::closeWindow() {
  ditherLock();
  if (XWnd->lOpen==false) {
    cout << "X11Window::closeWindow already closed"<<endl;
    ditherUnlock();
    return;
  }

  if(XWnd && XWnd->display && XWnd->window)
    {
      switch(XWnd->videoaccesstype)
	{
#ifdef __USE_X_SHAREDMEMORY__
	case VIDEO_XI_SHMSTD:
	  XShmDetach(XWnd->display,XWnd->shmseginfo);
	  if(XWnd->ximage)
	    XDestroyImage(XWnd->ximage);
	  if(XWnd->shmseginfo->shmaddr)
	    shmdt(XWnd->shmseginfo->shmaddr);
	  if(XWnd->shmseginfo->shmid>=0)
	    shmctl(XWnd->shmseginfo->shmid,IPC_RMID,NULL);
	  free(XWnd->shmseginfo);
	  break;

	case VIDEO_XI_SHMPIXMAP:
	  XShmDetach(XWnd->display,XWnd->shmseginfo);
	  XFreePixmap(XWnd->display,XWnd->pixmap);
	  if(XWnd->shmseginfo->shmaddr)
	    shmdt(XWnd->shmseginfo->shmaddr);
	  if(XWnd->shmseginfo->shmid>=0)
	    shmctl(XWnd->shmseginfo->shmid,IPC_RMID,NULL);
	  free(XWnd->shmseginfo);
	  break;
#endif
	case VIDEO_XI_STANDARD:
	  if(XWnd->ximage)
	    XDestroyImage(XWnd->ximage);
	  if(XWnd->virtualscreen)
	    free(XWnd->virtualscreen);
	  break;
	}
      XFreeGC(XWnd->display,XWnd->gc);
      XDestroyWindow(XWnd->display,XWnd->window);
      XCloseDisplay(XWnd->display);
    }
  XWnd->lOpen=false;
  ditherUnlock();
}




void X11Window::putImage(Picture* pic,TimeStamp* waitTime) {
  spendTime->gettimeofday();
  if (pic == NULL) {
    cout << "pic is null"<<endl;
    return;
  }
  ditherLock();
  
  startTime->gettimeofday();
  startTime->addOffset(waitTime);

  // threaded dither activated?
  if (lthreadDither==true) {
    if (pic->getThreadedDither()) {
      currentPic=pic;
      lPut=true;
    } else {
      ditherPicture(pic);
      lPut=true;
      currentPic=NULL;
    }
  } else {
    ditherPicture(pic);
    PutXImage();
    waitRestTime();
  } 



  ditherUnlock();
  
}


void X11Window::waitRestTime() {
  endTime->gettimeofday();
  startTime->minus(endTime,endTime);
  endTime->waitForIt();
}



void* X11Window::ditherLoop() {
  pthread_mutex_lock(&ditherMut);
  linDitherLoop=true;
  while(lDitherLoop) {
    if (pthread_mutex_trylock(&changeMut) == EBUSY) {
      pthread_cond_wait(&changeCond,&ditherMut);
      continue;
    }

    pthread_mutex_unlock(&changeMut);
    if (currentPic != NULL) {
      ditherPicture(currentPic);
      currentPic=NULL;
    }
    if (lPut) {
      PutXImage();
      lPut=false;
      waitRestTime();
    }
    pthread_cond_wait(&changeCond,&ditherMut);
  }
  pthread_mutex_unlock(&ditherMut);
  return NULL;
}


void X11Window::ditherLock() {
  pthread_mutex_lock(&changeMut);
  pthread_mutex_lock(&ditherMut);
}

void X11Window::ditherUnlock() {

  pthread_mutex_unlock(&changeMut);
  pthread_cond_signal(&changeCond);
  pthread_mutex_unlock(&ditherMut);

}

 
void X11Window::ditherPicture(struct Picture* pic) {


  int h=pic->getHeight();
  int w=pic->getWidth();
  unsigned char* lum=pic->getLuminancePtr();
  unsigned char* cr=pic->getCrPtr();
  unsigned char* cb=pic->getCbPtr();
  unsigned char* dest=(unsigned char* )XWnd->virtualscreen;
  totalWaitTime->gettimeofday();

  switch (XWnd->depth) {
  case 8:
    ditherImageOrdered(lum, cr, cb,dest , h, w);
    break;
  case 16:
    if (lmmx) {
      ditherBlock(lum,cr,cb,dest,h,w,0);
      ditherImageColor16(lum,cr,cb,dest,h,w,0);
    } else {
      ditherImageColor16(lum,cr,cb,dest,h,w,0);
    }

    break;
  case 24:
  case 32:
    if (lmmx) {
      dither32_mmx(lum, cr, cb,dest ,h,w,0);
      //ditherImageColor32(lum, cr, cb,dest ,h/2,w,100);
    } else {
      ditherImageColor32(lum, cr, cb,dest ,h,w,0);
    }


    break;
  default:
    cout << "cannot dither depth:"<<XWnd->depth<<endl;
  }
  currentTime->gettimeofday();
  currentTime->minus(totalWaitTime,totalWaitTime);
  //totalWaitTime->print("dither");
}


void X11Window::PutXImage() {



#ifdef __USE_X_SHAREDMEMORY__
  switch(XWnd->videoaccesstype)
    {
    case VIDEO_XI_SHMSTD:
      XShmPutImage(XWnd->display, XWnd->window, XWnd->gc, XWnd->ximage,
		   0, 0, 0, 0, XWnd->width, XWnd->height, False);
      XFlush(XWnd->display); /* Not needed, done by XPending */
      break;
      
    case VIDEO_XI_SHMPIXMAP:
      XClearWindow(XWnd->display,XWnd->window);
      XSync(XWnd->display,True); /* Not needed, done by XPending */
      break;
      
    case VIDEO_XI_STANDARD:
#endif
      XPutImage(XWnd->display, XWnd->window, XWnd->gc, XWnd->ximage,
		0, 0, 0, 0, XWnd->width, XWnd->height);      
      XFlush(XWnd->display); /* Not needed, done by XPending */
#ifdef __USE_X_SHAREDMEMORY__
      break;
    }
#endif

}


void X11Window::resizeWindow(unsigned int w,unsigned int h) {

  XResizeWindow(XWnd->display, XWnd->window, w, h);
  XFlush(XWnd->display);
}


int X11Window::getDepth() {
  return XWnd->depth;
}
