/*
  process skipped frames
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include "skipped.h"

#define NDEBUG
#include <assert.h>


Skipped::Skipped() {
}


Skipped::~Skipped() {
}



/*
 *--------------------------------------------------------------
 *
 * ProcessSkippedPFrameMBlocks --
 *
 *	Processes skipped macroblocks in P frames.
 *
 * Results:
 *	Calculates pixel values for luminance, Cr, and Cb planes
 *      in current pict image for skipped macroblocks.
 *
 * Side effects:
 *	Pixel values in pict image changed.
 *
 *--------------------------------------------------------------
 */
void Skipped::ProcessSkippedPFrameMBlocks(Picture* current,
					  Picture* future,
					  Macroblock* mblock,
					  int mb_width) {

  int row_size, half_row, mb_row, mb_col, row, col, rr;
  int addr, row_incr, half_row_incr, crow, ccol;
  int *dest, *src, *dest1, *src1;

  /* For each row in macroblock luminance plane... */
  if (mb_width == 0) {
    cout << "mb_width in skipped is 0"<<endl;
    return;
  }


  /* Calculate row sizes for luminance and Cr/Cb macroblock areas. */

  row_size = mb_width << 4;
  half_row = (row_size >> 1);
  row_incr = row_size >> 2;
  half_row_incr = half_row >> 2;

  /* For each skipped macroblock, do... */
  int lumEnd=current->getLumLength();
  int colorEnd=current->getColorLength();
  
  unsigned char *picDest;
  unsigned char *picSrc;
 
  unsigned char *picDestStart;
  unsigned char *picSrcStart;



  for (addr = mblock->past_mb_addr + 1;
       addr < mblock->mb_address; addr++) {

    /* Calculate macroblock row and col. */

    mb_row = addr / mb_width;
    mb_col = addr % mb_width;

    /* Calculate upper left pixel row,col for luminance plane. */

    row = mb_row << 4;
    col = mb_col << 4;
    
    picDest=current->getLuminancePtr();
    picSrc=future->getLuminancePtr();

    picDestStart=(picDest+(row*row_size)+col);
    picSrcStart=(picSrc+(row*row_size)+col);
   
    if ((picDestStart+7*row_size+7 >= picDest+lumEnd) ||
	(picDestStart < picDest)) {
      cout << "urg! last resort caught before sigsev skipped -1"<<endl;
      break;
    }
    if ((picSrcStart+7*row_size+7 >= picSrc+lumEnd) ||
	(picSrcStart < picSrc)) {
      cout << "urg! last resort caught before sigsev skipped -2"<<endl;
      break;
    }

    dest=(int*)picDestStart;
    src=(int*)picSrcStart;



    for (rr = 0; rr < 8; rr++) {

      /* Copy pixel values from last I or P picture. */
      memcpy(dest,src,sizeof(int)*4);

      dest += row_incr;
      src += row_incr;
      memcpy(dest,src,sizeof(int)*4);

      dest += row_incr;
      src += row_incr;
    }

    /*
     * Divide row,col to get upper left pixel of macroblock in Cr and Cb
     * planes.
     */

    crow = row >> 1;
    ccol = col >> 1;

    /* For each row in Cr, and Cb planes... */
    picDest=current->getCrPtr();
    picDestStart=(picDest+(crow*half_row)+ccol);
    if ((picDestStart+7*half_row_incr+7 >= picDest+colorEnd) ||
	(picDestStart < picDest)) {
      cout << "urg! last resort caught before sigsev skipped -3"<<endl;
      break;
    }


    dest=(int*)(current->getCrPtr()+(crow*half_row)+ccol);
    src=(int*)(future->getCrPtr()+(crow*half_row)+ccol);
    dest1=(int*)(current->getCbPtr()+(crow*half_row)+ccol);
    src1=(int*)(future->getCbPtr()+(crow*half_row)+ccol);

    for (rr = 0; rr < 4; rr++) {

      /* Copy pixel values from last I or P picture. */
      memcpy(dest,src,sizeof(int)*2);
      memcpy(dest1,src1,sizeof(int)*2);


      dest += half_row_incr;
      src += half_row_incr;
      dest1 += half_row_incr;
      src1 += half_row_incr;

      memcpy(dest,src,sizeof(int)*2);
      memcpy(dest1,src1,sizeof(int)*2);

      dest += half_row_incr;
      src += half_row_incr;
      dest1 += half_row_incr;
      src1 += half_row_incr;
    }

  }

  mblock->recon_right_for_prev = 0;
  mblock->recon_down_for_prev = 0;

}
  




/*
 *--------------------------------------------------------------
 *
 * ProcessSkippedBFrameMBlocks --
 *
 *	Processes skipped macroblocks in B frames.
 *
 * Results:
 *	Calculates pixel values for luminance, Cr, and Cb planes
 *      in current pict image for skipped macroblocks.
 *
 * Side effects:
 *	Pixel values in pict image changed.
 *
 *--------------------------------------------------------------
 */

void Skipped::ProcessSkippedBFrameMBlocks(Pict* picture,
					  Picture* past,
					  Picture* current,
					  Picture* future,
					  Macroblock* mblock,
					  int mb_width) {
  int row_size, half_row, mb_row, mb_col, row, col, rr;
  int right_half_for = 0, down_half_for = 0;
  int c_right_half_for = 0, c_down_half_for = 0;
  int right_half_back = 0, down_half_back = 0;
  int c_right_half_back = 0, c_down_half_back = 0;
  int addr, right_for = 0, down_for = 0;
  int recon_right_for, recon_down_for;
  int recon_right_back, recon_down_back;
  int right_back = 0, down_back = 0;
  int c_right_for = 0, c_down_for = 0;
  int c_right_back = 0, c_down_back = 0;
  unsigned char forw_lum[256];
  unsigned char forw_cr[64], forw_cb[64];
  unsigned char back_lum[256], back_cr[64], back_cb[64];
  int row_incr, half_row_incr;
  int ccol, crow;


  /* Calculate row sizes for luminance and Cr/Cb macroblock areas. */

  if (mb_width == 0) {
    cout << "mb_width in skipped is 0 (2)"<<endl;
    return;
  }

  row_size = mb_width << 4;
  half_row = (row_size >> 1);
  row_incr = row_size >> 2;
  half_row_incr =  half_row >> 2;

  /* Establish motion vector codes based on full pixel flag. */

  if (picture->full_pel_forw_vector) {
    recon_right_for = mblock->recon_right_for_prev << 1;
    recon_down_for = mblock->recon_down_for_prev << 1;
  } else {
    recon_right_for = mblock->recon_right_for_prev;
    recon_down_for = mblock->recon_down_for_prev;
  }

  if (picture->full_pel_back_vector) {
    recon_right_back = mblock->recon_right_back_prev << 1;
    recon_down_back = mblock->recon_down_back_prev << 1;
  } else {
    recon_right_back = mblock->recon_right_back_prev;
    recon_down_back = mblock->recon_down_back_prev;
  }


  /* If only one motion vector, do display copy, else do full
     calculation. 
  */

  /* Calculate motion vectors. */
  
  if (mblock->bpict_past_forw) {
    right_for = recon_right_for >> 1;
    down_for = recon_down_for >> 1;
    right_half_for = recon_right_for & 0x1;
    down_half_for = recon_down_for & 0x1;
    
    recon_right_for /= 2;
    recon_down_for /= 2;
    c_right_for = recon_right_for >> 1;
    c_down_for = recon_down_for >> 1;
    c_right_half_for = recon_right_for & 0x1;
    c_down_half_for = recon_down_for & 0x1;
    
  }
  if (mblock->bpict_past_back) {
    right_back = recon_right_back >> 1;
    down_back = recon_down_back >> 1;
    right_half_back = recon_right_back & 0x1;
    down_half_back = recon_down_back & 0x1;
    
    recon_right_back /= 2;
    recon_down_back /= 2;
    c_right_back = recon_right_back >> 1;
    c_down_back = recon_down_back >> 1;
    c_right_half_back = recon_right_back & 0x1;
    c_down_half_back = recon_down_back & 0x1;
    
  }
  /* For each skipped macroblock, do... */
  
  for (addr = mblock->past_mb_addr + 1;
       addr < mblock->mb_address; addr++) {
    
    /* Calculate macroblock row and col. */
    
    mb_row = addr / mb_width;
    mb_col = addr % mb_width;
    
    /* Calculate upper left pixel row,col for luminance plane. */
    
    row = mb_row << 4;
    col = mb_col << 4;
    crow = row / 2;
    ccol = col / 2;
    
    /* If forward predicted, calculate prediction values. */
    if (mblock->bpict_past_forw) {
      int lumEnd=current->getLumLength();
      int colorEnd=current->getColorLength();
      Skipped::ReconSkippedBlock(past->getLuminancePtr(),
				 forw_lum,row,col,row_size,
				 right_for,down_for,
				 right_half_for,
				 down_half_for,16,lumEnd);
      Skipped::ReconSkippedBlock(past->getCrPtr(),
				 forw_cr,crow,ccol, half_row,
				 c_right_for,c_down_for,
				 c_right_half_for,
				 c_down_half_for,8,colorEnd);
      Skipped::ReconSkippedBlock(past->getCbPtr(),
				 forw_cb,crow,ccol,half_row,
				 c_right_for,c_down_for,
				 c_right_half_for,
				 c_down_half_for,8,colorEnd);
    }
    /* If back predicted, calculate prediction values. */
    
    if (mblock->bpict_past_back) {
      int lumEnd=current->getLumLength();
      int colorEnd=current->getColorLength();
      Skipped::ReconSkippedBlock(future->getLuminancePtr(),
				 back_lum,row,col,row_size,
				 right_back,down_back,
				 right_half_back,down_half_back,
				 16,lumEnd);
      Skipped::ReconSkippedBlock(future->getCrPtr(),
				 back_cr,crow,ccol,
				 half_row,c_right_back,
				 c_down_back,c_right_half_back,
				 c_down_half_back,8,colorEnd);
      Skipped::ReconSkippedBlock(future->getCbPtr(),
				 back_cb,crow,ccol,half_row,
				 c_right_back,c_down_back,
				 c_right_half_back,
				 c_down_half_back,8,colorEnd);
    }
    unsigned char* picDest=current->getLuminancePtr();
    int lumEnd=current->getLumLength();
    int colorEnd=current->getColorLength();
 
    unsigned char* picDestStart=(picDest+(row*row_size)+col);

   
    if ((picDestStart+7*row_size+7 >= picDest+lumEnd) ||
	(picDestStart < picDest)) {
      cout << "urg! last resort caught before sigsev skipped -4"<<endl;
      return;
    }

    picDest=current->getCrPtr();
    picDestStart=(picDest+(crow*half_row)+ccol);
    if ((picDestStart+7*half_row_incr+7 >= picDest+colorEnd) ||
	(picDestStart < picDest)) {
      cout << "urg! last resort caught before sigsev skipped -5"<<endl;
      exit(0);
    }
    

    if (mblock->bpict_past_forw &&
	!mblock->bpict_past_back) {
      
      int *dest, *dest1;
      int *src, *src1;
      dest=(int*)(current->getLuminancePtr()+(row*row_size)+col);
      src=(int*)forw_lum;
      
      for (rr = 0; rr < 16; rr++) {
	
	/* memcpy(dest, forw_lum+(rr<<4), 16);  */
	
	dest[0] = src[0];
	dest[1] = src[1];
	dest[2] = src[2];
	dest[3] = src[3];
	dest += row_incr;
	src += 4;
      }
      
      dest = (int*)(current->getCrPtr()+(crow*half_row)+ccol);
      dest1 = (int*)(current->getCbPtr()+(crow*half_row)+ccol);
      src = (int*)forw_cr;
      src1 = (int*)forw_cb;
      
      for (rr = 0; rr < 8; rr++) {
	/*
	 * memcpy(dest, forw_cr+(rr<<3), 8); memcpy(dest1, forw_cb+(rr<<3),
	 * 8);
	 */
	
	dest[0] = src[0];
	dest[1] = src[1];
	
	dest1[0] = src1[0];
	dest1[1] = src1[1];
	
	dest += half_row_incr;
	dest1 += half_row_incr;
	src += 2;
	src1 += 2;
      }
    } else if (mblock->bpict_past_back &&
	       !mblock->bpict_past_forw) {
      
      int *src, *src1;
      int *dest, *dest1;
      dest=(int*)(current->getLuminancePtr()+(row*row_size)+col);
      src = (int*)back_lum;
      
      for (rr = 0; rr < 16; rr++) {
	dest[0] = src[0];
	dest[1] = src[1];
	dest[2] = src[2];
	dest[3] = src[3];
	dest += row_incr;
	src += 4;
      }
      
      
      dest = (int *)(current->getCrPtr()+(crow*half_row)+ccol);
      dest1 = (int *)(current->getCbPtr()+(crow*half_row)+ccol);
      src = (int *)back_cr;
      src1 = (int *)back_cb;
      
      for (rr = 0; rr < 8; rr++) {
	/*
	 * memcpy(dest, back_cr+(rr<<3), 8); memcpy(dest1, back_cb+(rr<<3),
	 * 8);
	 */
	
	dest[0] = src[0];
	dest[1] = src[1];
	
	dest1[0] = src1[0];
	dest1[1] = src1[1];
	
	dest += half_row_incr;
	dest1 += half_row_incr;
	src += 2;
	src1 += 2;
      }
    } else {
      
      unsigned char *src1, *src2, *src1a, *src2a;
      unsigned char *dest, *dest1;
      dest = current->getLuminancePtr()+(row*row_size)+col;
      src1 = forw_lum;
      src2 = back_lum;
      
      for (rr = 0; rr < 16; rr++) {
        dest[0] = (int) (src1[0] + src2[0]) >> 1;
        dest[1] = (int) (src1[1] + src2[1]) >> 1;
        dest[2] = (int) (src1[2] + src2[2]) >> 1;
        dest[3] = (int) (src1[3] + src2[3]) >> 1;
        dest[4] = (int) (src1[4] + src2[4]) >> 1;
        dest[5] = (int) (src1[5] + src2[5]) >> 1;
        dest[6] = (int) (src1[6] + src2[6]) >> 1;
        dest[7] = (int) (src1[7] + src2[7]) >> 1;
        dest[8] = (int) (src1[8] + src2[8]) >> 1;
        dest[9] = (int) (src1[9] + src2[9]) >> 1;
        dest[10] = (int) (src1[10] + src2[10]) >> 1;
        dest[11] = (int) (src1[11] + src2[11]) >> 1;
        dest[12] = (int) (src1[12] + src2[12]) >> 1;
        dest[13] = (int) (src1[13] + src2[13]) >> 1;
        dest[14] = (int) (src1[14] + src2[14]) >> 1;
        dest[15] = (int) (src1[15] + src2[15]) >> 1;
        dest += row_size;
        src1 += 16;
        src2 += 16;
      }
      
      
      dest = current->getCrPtr() + (crow * half_row) + ccol;
      dest1 = current->getCbPtr() + (crow * half_row) + ccol;
      src1 = forw_cr;
      src2 = back_cr;
      src1a = forw_cb;
      src2a = back_cb;
      
      for (rr = 0; rr < 8; rr++) {
        dest[0] = (int) (src1[0] + src2[0]) >> 1;
        dest[1] = (int) (src1[1] + src2[1]) >> 1;
        dest[2] = (int) (src1[2] + src2[2]) >> 1;
        dest[3] = (int) (src1[3] + src2[3]) >> 1;
        dest[4] = (int) (src1[4] + src2[4]) >> 1;
        dest[5] = (int) (src1[5] + src2[5]) >> 1;
        dest[6] = (int) (src1[6] + src2[6]) >> 1;
        dest[7] = (int) (src1[7] + src2[7]) >> 1;
        dest += half_row;
        src1 += 8;
        src2 += 8;
	
        dest1[0] = (int) (src1a[0] + src2a[0]) >> 1;
        dest1[1] = (int) (src1a[1] + src2a[1]) >> 1;
        dest1[2] = (int) (src1a[2] + src2a[2]) >> 1;
        dest1[3] = (int) (src1a[3] + src2a[3]) >> 1;
        dest1[4] = (int) (src1a[4] + src2a[4]) >> 1;
        dest1[5] = (int) (src1a[5] + src2a[5]) >> 1;
        dest1[6] = (int) (src1a[6] + src2a[6]) >> 1;
        dest1[7] = (int) (src1a[7] + src2a[7]) >> 1;
        dest1 += half_row;
        src1a += 8;
        src2a += 8;
      }
    }
    
  }
}







/*
 *--------------------------------------------------------------
 *
 * ReconSkippedBlock --
 *
 *	Reconstructs predictive block for skipped macroblocks
 *      in B Frames.
 *
 * Results:
 *	No return values.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void Skipped::ReconSkippedBlock(unsigned char* source,
				unsigned char* dest,
				int row,
				int col,
				int row_size,
				int right,
				int down,
				int right_half,
				int down_half,
				int width,int maxLen) {
  int rr;
  unsigned char *source2;
  unsigned char *tmp;

  tmp = source+((row + down) * row_size) + col + right;
  //printf("tmp:%8x tmpEnd:%8x\n",tmp,tmp+7*row_size+7);
  //printf("source:%8x sourceEnd:%8x\n",source,source+maxLen);

  if ((tmp+7*row_size+7 >= source+maxLen) ||
      (tmp < source)) {
    cout << "urg! last resort caught before sigsev skipped -6"<<endl;
    return;
  } 

  source=tmp;
  if (width == 16) {
    if ((!right_half) && (!down_half)) {
	if (right & 0x1) {
	  /* No alignment, use bye copy */
	  for (rr = 0; rr < 16; rr++) {

	    memcpy(dest,source,sizeof(char)*16);

	    dest += 16;
	    source += row_size;
	  }
	} else if (right & 0x2) {
	  /* Half-word bit aligned, use 16 bit copy */
	  short *src = (short *)source;
	  short *d = (short *)dest;
	  row_size >>= 1;
	  for (rr = 0; rr < 16; rr++) {

	    memcpy(d,src,sizeof(short)*8);
    
	    d += 8;
	    src += row_size;
	  }
	} else {
	  /* Word aligned, use 32 bit copy */
	  int *src = (int *)source;
	  int *d = (int *)dest;
	  row_size >>= 2;
	  for (rr = 0; rr < 16; rr++) {
	    d[0] = src[0];
	    d[1] = src[1];
	    d[2] = src[2];
	    d[3] = src[3];
	    d += 4;
	    src += row_size;
	  }
	}
    } else {
      source2 = source + right_half + (row_size * down_half);
      for (rr = 0; rr < width; rr++) {
        dest[0] = (int) (source[0] + source2[0]) >> 1;
        dest[1] = (int) (source[1] + source2[1]) >> 1;
        dest[2] = (int) (source[2] + source2[2]) >> 1;
        dest[3] = (int) (source[3] + source2[3]) >> 1;
        dest[4] = (int) (source[4] + source2[4]) >> 1;
        dest[5] = (int) (source[5] + source2[5]) >> 1;
        dest[6] = (int) (source[6] + source2[6]) >> 1;
        dest[7] = (int) (source[7] + source2[7]) >> 1;
        dest[8] = (int) (source[8] + source2[8]) >> 1;
        dest[9] = (int) (source[9] + source2[9]) >> 1;
        dest[10] = (int) (source[10] + source2[10]) >> 1;
        dest[11] = (int) (source[11] + source2[11]) >> 1;
        dest[12] = (int) (source[12] + source2[12]) >> 1;
        dest[13] = (int) (source[13] + source2[13]) >> 1;
        dest[14] = (int) (source[14] + source2[14]) >> 1;
        dest[15] = (int) (source[15] + source2[15]) >> 1;
        dest += width;
        source += row_size;
        source2 += row_size;
      }
    }
  } else {			/* (width == 8) */
    assert(width == 8);
    if ((!right_half) && (!down_half)) {
      if (right & 0x1) {
	for (rr = 0; rr < width; rr++) {

	  memcpy(dest,source,sizeof(char)*8);

	  dest += 8;
	  source += row_size;
	}
      } else if (right & 0x02) {
	short *d = (short *)dest;
	short *src = (short *)source;
	row_size >>= 1;
	for (rr = 0; rr < width; rr++) {
	  d[0] = src[0];
	  d[1] = src[1];
	  d[2] = src[2];
	  d[3] = src[3];
	  d += 4;
	  src += row_size;
	}
      } else {
	int *d = (int *)dest;
	int *src = (int *)source;
	row_size >>= 2;
	for (rr = 0; rr < width; rr++) {
	  d[0] = src[0];
	  d[1] = src[1];
	  d += 2;
	  src += row_size;
	}
      }
    } else {
      source2 = source + right_half + (row_size * down_half);
      for (rr = 0; rr < width; rr++) {
        dest[0] = (int) (source[0] + source2[0]) >> 1;
        dest[1] = (int) (source[1] + source2[1]) >> 1;
        dest[2] = (int) (source[2] + source2[2]) >> 1;
        dest[3] = (int) (source[3] + source2[3]) >> 1;
        dest[4] = (int) (source[4] + source2[4]) >> 1;
        dest[5] = (int) (source[5] + source2[5]) >> 1;
        dest[6] = (int) (source[6] + source2[6]) >> 1;
        dest[7] = (int) (source[7] + source2[7]) >> 1;
        dest += width;
        source += row_size;
        source2 += row_size;
      }
    }
  }
}

