/*
  stores sequence header info, for reinit of stream
  Copyright (C) 2000  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "sequenceParse.h"

/* Video rates table */
/* Cheat on Vid rates, round to 30, and use 30 if illegal value 
   Except for 9, where Xing means 15, and given their popularity, we'll
   be nice and do it */

static float VidRateNum[16]={29.97, 24, 24, 25, 29.97, 29.97, 50, 59.94, 
			     59.94, 14.985, 29.97, 29.97, 29.97, 29.97,
			     29.97, 29.97};

/*
static float VidRateNum[16]={30, 24, 24, 25, 30, 30, 50, 60, 
			     60, 15, 30, 30, 30, 30,
			     30, 30};
*/

static const unsigned char default_intra_matrix[64] = {
  8, 16, 19, 22, 26, 27, 29, 34,
  16, 16, 22, 24, 27, 29, 34, 37,
  19, 22, 26, 27, 29, 34, 34, 38,
  22, 22, 26, 27, 29, 34, 37, 40,
  22, 26, 27, 29, 32, 35, 40, 48,
  26, 27, 29, 32, 35, 40, 48, 58,
  26, 27, 29, 34, 38, 46, 56, 69,
  27, 29, 35, 38, 46, 56, 69, 83};
  


/* Set up array for fast conversion from zig zag order to row/column
   coordinates.
*/

const int zigzag[64][2] = {
  {0,0},{1,0},{0,1},{0,2},{1,1},{2,0},{3,0},{2,1},{1,2},{0,3},{0,4},{1,3},
  {2,2},{3,1},{4,0},{5,0},{4,1},{3,2},{2,3},{1,4},{0,5},{0,6},{1,5},{2,4},
  {3,3},{4,2},{5,1},{6,0},{7,0},{6,1},{5,2},{4,3},{3,4},{2,5},{1,6},{0,7},
  {1,7},{2,6},{3,5},{4,4},{5,3},{6,2},{7,1},{7,2},{6,3},{5,4},{4,5},{3,6},
  {2,7},{3,7},{4,6},{5,5},{6,4},{7,3},{7,4},{6,5},{5,6},{4,7},{5,7},{6,6},
  {7,5},{7,6},{6,7},{7,7} };





SequenceParse::SequenceParse(OutputStream* output) {
  init(&sequence,output);
}


void SequenceParse::init(Sequence* sequence,OutputStream* output) {
  sequence->ext_data = NULL;
  sequence->user_data = NULL;
  sequence->h_size=0;
  sequence->v_size=0;
  sequence->mb_height=0;
  sequence->mb_width=0;
  sequence->mb_size=0;
  sequence->aspect_ratio=0;
  sequence->bit_rate=0;
  sequence->vbv_buffer_size=0;
  sequence->const_param_flag=0;
  sequence->picture_rate=0.0;
  SequenceParse::init_quanttables(sequence);

  sequence->output=output;
  sequence->pictureArray=NULL; 
}

SequenceParse::~SequenceParse() {
  if (sequence.pictureArray != NULL) {
    cout << "delete pictureArray"<<endl;
    delete sequence.pictureArray;
  }
  if (sequence.ext_data != NULL) {
    cout << "delete sequence.ext_dat"<<endl;
    delete sequence.ext_data;
  }
  if (sequence.user_data != NULL) {
    cout << "delete sequence.user_data"<<endl;
    delete sequence.user_data;
  }
}



/*
 *--------------------------------------------------------------
 *
 * ParseSeqHead --
 *
 *      Assumes bit stream is at the begining of the sequence
 *      header start code. Parses off the sequence header.
 *
 * Results:
 *      Fills the vid_stream structure with values derived and
 *      decoded from the sequence header. Allocates the pict image
 *      structures based on the dimensions of the image space
 *      found in the sequence header.
 *
 * Side effects:
 *      Bit stream irreversibly parsed off.
 *
 *--------------------------------------------------------------
 */
int SequenceParse::parseSeq(MpegPlayBitWindow* bitwindow,Sequence* dest) {
  unsigned int data;
  int i ;

  /* Flush off sequence start code. */

  bitwindow->flushBits(32);

  /* Get horizontal size of image space. */

  data=bitwindow->getBits(12);
  dest->h_size = data;

  /* Get vertical size of image space. */

  data=bitwindow->getBits(12);
  dest->v_size = data;

  


  /* Calculate macroblock width and height of image space. */

  dest->mb_width = (dest->h_size + 15) / 16;
  dest->mb_height = (dest->v_size + 15) / 16;
  dest->mb_size=dest->mb_height * dest->mb_width-1;
  

  /*
   * Initialize ring buffer of pict images now that dimensions of image space
   * are known.
   */
  createPictureArray(dest);


  /* Parse of aspect ratio code. */

  data=bitwindow->getBits(4);

  dest->aspect_ratio = (unsigned char) data;

  /* Parse off picture rate code. */

  data=bitwindow->getBits(4);
  dest->pictureArray->setPicturePerSecond(VidRateNum[data]);
  dest->picture_rate=VidRateNum[data];

  /* Parse off bit rate. */

  data=bitwindow->getBits(18);
  dest->bit_rate = data;

  /* Flush marker bit. */

  bitwindow->flushBits(1);

  /* Parse off vbv buffer size. */

  data=bitwindow->getBits(10);
  dest->vbv_buffer_size = data;

  /* Parse off contrained parameter flag. */

  data=bitwindow->getBits(1);
  if (data) {
    dest->const_param_flag = true;
  } else
    dest->const_param_flag = false;

  /*
   * If intra_quant_matrix_flag set, parse off intra quant matrix values.
   */
  data=bitwindow->getBits(1);
  if (data) {
    for (i = 0; i < 64; i++) {
      data=bitwindow->getBits(8);
      dest->intra_quant_matrix[zigzag[i][1]][zigzag[i][0]]=(unsigned char)data;
    }
  }
  /*
   * If non intra quant matrix flag set, parse off non intra quant matrix
   * values.
   */

  data=bitwindow->getBits(1);
  if (data) {
    for (i = 0; i < 64; i++) {
      data=bitwindow->getBits(8);

      dest->non_intra_quant_matrix[zigzag[i&0x3f][1]][zigzag[i&0x3f][0]]=
	(unsigned char) data;
    }
  }
  /* Go to next start code. */

  DeMux::next_start_code(bitwindow);

  /*
   * If next start code is extension/user start code, 
   * parse off extension data.
   */
  Util::check_ext_data(bitwindow,dest->ext_data);
  Util::check_user_data(bitwindow,dest->user_data);
  return PARSE_OK;

}



void SequenceParse::init_quanttables(Sequence* dest) {
  int i;
  int j;


   /* Copy default intra matrix. */

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      dest->intra_quant_matrix[i][j]=default_intra_matrix[i*8+j];
    }
  }

  /* Initialize non intra quantization matrix. */

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      dest->non_intra_quant_matrix[i][j] = 16;
    }
  }
 

}

void SequenceParse::copy(Sequence* dest,Sequence* src) {
  dest->ext_data = NULL;
  dest->user_data = NULL;

  dest->h_size=src->h_size;
  dest->v_size=src->v_size;
  dest->mb_height=src->mb_height;
  dest->mb_width=src->mb_width;
  dest->mb_size=src->mb_size;
  dest->aspect_ratio=src->aspect_ratio;
  dest->bit_rate=src->bit_rate;
  dest->vbv_buffer_size=src->vbv_buffer_size;
  dest->const_param_flag=src->const_param_flag;
  dest->picture_rate=src->picture_rate;

  int i;
  int j;


   /* Copy default intra matrix. */

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      dest->intra_quant_matrix[i][j]=src->intra_quant_matrix[i][j];

    }
  }

  /* Initialize non intra quantization matrix. */

  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      dest->non_intra_quant_matrix[i][j] =src->non_intra_quant_matrix[i][j] ;
    }
  }
  
  dest->output=src->output;
  createPictureArray(dest);

}

Sequence* SequenceParse::getSequence() {
  return &sequence;
}

void SequenceParse::recreate(Sequence* info) {
  SequenceParse::copy(&(this->sequence),info);
  sequence.ext_data = NULL;
  sequence.user_data = NULL;
  
  createPictureArray(&sequence);
}


void SequenceParse::createPictureArray(Sequence* dest) {
  if (dest->pictureArray == NULL) {
    int width=dest->mb_width*16;
    int height=dest->mb_height*16;
    if (dest->output == NULL) {
      cout << "output is NULL"<<endl;
      exit(0);
    }
    cout << "width:"<<width<<" height:"<<height<<endl;

    dest->output->openWindow(width,height,(char*)"kmpg");
    dest->pictureArray=new PictureArray(width,height,
					dest->output->getDepth());
    dest->pictureArray->setPicturePerSecond(dest->picture_rate);
  }  
}

void SequenceParse::print(Sequence* seq) {

  cout <<"h_size:"<<seq->h_size<<endl;
  cout <<"v_size:"<<seq->v_size<<endl;
  cout <<"mb_height:"<<seq->mb_height<<endl;
  cout <<"mb_width:"<<seq->mb_width<<endl;
  cout <<"mb_size:"<<seq->mb_size<<endl;
  cout <<"aspect_ratio:"<<seq->aspect_ratio<<endl;
  cout <<"bit_rate:"<<seq->bit_rate<<endl;
  cout <<"vbv_buffer_size:"<<seq->vbv_buffer_size<<endl;
  cout <<"const_param_flag:"<<seq->const_param_flag<<endl;
 
}
