/*
  fileoperations with the playlist model
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include <qdir.h>
#include <kfiledialog.h>
#include <qregexp.h> 
#include <playlistOperation.h>

// carfull you cant set i18n(..) around static vars!

static const char* selectorStringFile="*.[Mm][Pp]?|mp3 files\n"\
                                      "*.[Mm]3[uU]| m3u Playlist files\n"\
			              "*.*| All files";

static const char* selectorStringM3U="*.[Mm]3[uU]| m3u Playlist files\n"\
                                     "*.[Mm][Pp]?|mp3 files\n"\
			             "*.*| All files";

static char* lastPath=NULL;

char* PlaylistOperation::getLastPath() {
  return lastPath;
}


void PlaylistOperation::setLastPath(const char* filename) {
  if (lastPath != NULL) {
    delete lastPath;
  }
  KURL kurlfile(filename);
  const char* path=kurlfile.path();
  lastPath=strdup(path);

}

 

PlaylistOperation::PlaylistOperation() {
}


PlaylistOperation::~PlaylistOperation() {
}


void PlaylistOperation::printPlaylist(Playlist* playlist) {
  Song* song;
  int i;
  int n=playlist->count();
  QString string;
  const char* kurl;
  cout << "Printing Playlist [START]"<<endl;
  for( i=0; i<n; i++ ) {
    song=playlist->at(i);
    kurl=song->getSongLocation();
    cout << kurl<<endl;
  }
  cout << "Printing Playlist [END]"<<endl;
}  


int PlaylistOperation::writePlaylist(Playlist* playlist) {
  QString filename;
  QFileInfo fileInfo;
  filename=KFileDialog::getSaveFileName(PlaylistOperation::getLastPath(),
					i18n(selectorStringM3U));

  if( filename.isNull() ) {
     return false;
  }
  PlaylistOperation::setLastPath(filename);
  fileInfo.setFile(filename.data());
  if (fileInfo.extension()=="") {
    filename=filename+".m3u";
  }

  QTextStream stream;
  QFile file;
  file.setName( filename );
  if (file.open( IO_WriteOnly ) == false) {
    cout << "cannot open file"<<endl;
  }
  stream.setDevice( &file );
  Song* song;
  int i;
  int n=playlist->count();
  QString string;
  QString dir=fileInfo.dirPath();

  QString rep1=QString("^")+dir+"/";
  QString rep2=QString("^file:")+dir+"/";
  QString rep3=QString("^file:");

  for( i=0; i<n; i++ ) {
    song=playlist->at(i);
    QString location(song->getSongLocation());
    location.replace(QRegExp(rep1),"");
    location.replace(QRegExp(rep2),"");
    location.replace(QRegExp(rep3),"");
    if (location.length () > 0) {
      stream << location << endl;
    }
  }
  file.close();
  KURL kurlfile(filename);
  playlist->setName((char*)kurlfile.filename());
  return true;
}


Playlist* PlaylistOperation::readPlaylist() {
   QString filename;
   Playlist* playlist=new Playlist();
   QFileInfo fileInfo;
  
   filename=KFileDialog::getOpenFileName(PlaylistOperation::getLastPath(),
					 i18n(selectorStringM3U));


   if (filename.isNull()) {
     return NULL;
   }
   PlaylistOperation::setLastPath(filename);
   fileInfo.setFile(filename.data());
   if (fileInfo.extension().upper() != QString("m3u").upper()) {
     // ok treat as normal file
     KURL kurl(filename.data());
     PlaylistOperation::rootInsert(filename.data(),playlist);
     return playlist;
   }
   KURL kurl(filename.data());

   PlaylistOperation::readPlaylist((char*)kurl.path(),playlist);
   return playlist;
}


int PlaylistOperation::insert(Playlist* src,Playlist* dest,int pos=-1) {
  int i;
  int n=src->count();
  int back=0;
  QStrList strlist;
  Song* song;
  const char* kurl;

  for( i=0; i<n; i++ ) {
    song=src->at(i);
    kurl=song->getSongLocation();
    strlist.append(kurl);
  }
  back=PlaylistOperation::insert(&strlist,dest,pos);
  return back;
} 



int PlaylistOperation::insert(QStrList* src,Playlist* dest,int pos=-1) {
  char* str;
  Song* song;
  Buffer* targetFilename=new Buffer(30);
  int i=0;
 
  if (pos <= -1) {
    for( str=src->first(); str!=0; str=src->next() ){
      if (PlaylistOperation::canProcess(str,dest,targetFilename)) {
	song=new Song(targetFilename->getData());
	dest->addSong(song);
	i++;
      }
    }
    delete targetFilename;
    return i;
  }
  // otherwise insert at the position.
  for( str=src->first(); str!=0; str=src->next() ){
    if (PlaylistOperation::canProcess(str,dest,targetFilename)) {
      song=new Song(targetFilename->getData());
      dest->addSong(song,pos+i);
      i++;
    }
  }
  return i;
}



void PlaylistOperation::rootInsert(KDNDDropZone* src,
			       Playlist* dest,int pos=-1) {
  QStrList strlist;
  strlist = src->getURLList();
  PlaylistOperation::rootInsert(&strlist,dest,pos);
}


void PlaylistOperation::rootInsert(QStrList* src,
				   Playlist* dest,int pos=-1) {

  // Here we check if we have to process m3u
  // on deeper level m3u are not resolved
  // Reason: when dropping directories (in which may be m3u)
  // it is (in almost all cases) stupid to resolve them as well
  // Thus we check here and later insert m3u only as "file"
  char* str=NULL;
  int i=0;
  for( str=src->first(); str!=0; str=src->next() ){
    KURL kurl(str);
    QFileInfo fileInfo(kurl.path());
    if (fileInfo.extension().upper() == QString("M3U")) {
      Playlist* m3uList=new Playlist();
      PlaylistOperation::readPlaylist((char*)kurl.path(),m3uList);
      if (pos <= -1) {
	PlaylistOperation::insert(m3uList,dest,-1);
      } else {
	i=i+PlaylistOperation::insert(m3uList,dest,pos+i);
      }
      delete m3uList;
    } else {
      QStrList singleFileList;
      singleFileList.append(str);
      if (pos <= -1) {
	PlaylistOperation::insert(&singleFileList,dest,-1);
      } else {
	PlaylistOperation::insert(&singleFileList,dest,pos+i);
	i++;
      }
    }
  }
}


void PlaylistOperation::rootInsert(char* filename,
				   Playlist* dest,int pos=-1) {
  QStrList strlist;
  KURL kurl(filename);
  strlist.append(kurl.url());
  
  PlaylistOperation::rootInsert(&strlist,dest,pos);
} 
 


int PlaylistOperation::loadSong(Playlist* playlist) {
  QString filename;
  
  filename = KFileDialog::getOpenFileName(PlaylistOperation::getLastPath(),
					  i18n(selectorStringFile) );
  if( filename.isNull() ) {
    cout << "filenam null"<<endl;
    return false;
  }
  
  PlaylistOperation::setLastPath(filename);
  PlaylistOperation::rootInsert(filename.data(),playlist);
  return true;
}

 
void PlaylistOperation::playNext(Playlist* playlist) {
  int play=0;
  int currentSong=playlist->getPlaySongPos();
  int count=playlist->count();
  if (playlist->getShuffle() == false) {
    if (count <= 0) {
      return;
    }
    play=currentSong+1;
    if (play >(count-1) ) {
      if (playlist->getRepeat()) {
	playlist->setCurrentPos(0);
	PlaylistOperation::playNextNotCorrupt(playlist);
      }
      return;
    }
    playlist->setCurrentPos(play);
    PlaylistOperation::playNextNotCorrupt(playlist);
    return;
  }
  PlaylistOperation::playShuffle(playlist);
}


void PlaylistOperation::playPrev(Playlist* playlist) {
  int play=0;
  int currentSong=playlist->getPlaySongPos();
  int count=playlist->count();

  if (playlist->getShuffle() == false) {
    if (count <= 0) {
      return;
    }
    play=currentSong-1;
    if (play < 0) {
      return;
    } 
    playlist->setCurrentPos(play);
    PlaylistOperation::playPrevNotCorrupt(playlist);
    return;
  }
  PlaylistOperation::playShuffle(playlist);

}


void PlaylistOperation::playCurrent(Playlist* playlist) {
  int pos=playlist->getCurrentPos();
  if (pos != -1) {
    playlist->playSong(pos);
    Song* song=playlist->at(pos);
    if (song != NULL) {
      song->setPlayed(true);
      if (playlist->getShuffle()) {
	song->setShufflePlay(true);
      }
    }
  }
}


int PlaylistOperation::getRandom(Playlist* playlist) {
  int back=-1;
  float percent=(float)random()/(float)RAND_MAX;
  float count=(float)playlist->count();
  if (count == 0.0) {
    return -1;
  }
  back=(int)(percent*count);
  return back;
}




void PlaylistOperation::playShuffle(Playlist* playlist) {
  Playlist* playlistShuffle=new Playlist();
  int i;
  Song* song;
  int n=playlist->count();

  if (n == 0) {
    return;
  }

  for(i=0;i<n;i++) {
    song=playlist->at(i);
    if (song->getShufflePlay() == false) {
      if (song->getCorrupt() == false) {
	playlistShuffle->addSong(song);
      }
    }
  }
  if (playlistShuffle->count() == 0) {
    if (playlist->getRepeat()) {
      playlist->setShuffle(false);
      playlist->setShuffle(true);
      PlaylistOperation::playShuffle(playlist);
      return;
    }
  }
  int index=PlaylistOperation::getRandom(playlistShuffle);
  if (index >= 0) {
    song=playlistShuffle->at(index);
    int indexPlaylist=playlist->find(song);
    if (song != NULL) {
      playlist->setCurrentPos(indexPlaylist);
      PlaylistOperation::playCurrent(playlist);
    }
  }
  delete playlistShuffle;
}


void PlaylistOperation::readPlaylist(char* filename,
				     Playlist* playlist,int pos=-1) {
  QFile file;
  QTextStream stream;
  QString string;
  QString base_path;
  QStrList strlist;

  file.setName( filename );
  if (file.open( IO_ReadOnly ) == false) {
    cout << "cannot open file:"<<filename<<endl;
    return;
  }

  base_path = QFileInfo(filename).dirPath( true );
  

  stream.setDevice( &file );
  do {
    string = stream.readLine();
    // If already a starting slash we do not add another
    // then we have a (hopefully) valid unix path.
    // we do not substitue
    if (string.left(1) == "/") {
      // nothing
    } else {
      string.replace( QRegExp("\\") ,"/" );
      string = QString(base_path + "/" + string);
    }
    strlist.append(string);
  } while(!stream.eof());
  file.close();
  PlaylistOperation::insert(&strlist,playlist,pos);
  KURL kurl(filename);
  
  playlist->setName((char*)kurl.filename());
}



int PlaylistOperation::canProcess(char* file,Playlist* playlist,Buffer* dest){
  KURL kurl(file);
  QFileInfo fileInfo(kurl.path());
  dest->clear();
  if (fileInfo.isDir()) {
    // Throw away . and .. or things could get out of hand...
    if(fileInfo.fileName() == "." || fileInfo.fileName() == ".."){
      return false;
    }
    // get location of directory and append the contents of it to a strList
    // Note the funky bug in QT that forces us to to string concatenation of a
    // QDir::absPath to a QDir []...

    QDir dir = QDir::QDir(kurl.path(), "");
    QStrList filesInPath;
    unsigned int i;
    for (i=0;i<dir.count();i++) {
      QString fullPath(dir.absPath());
      fullPath.append ("/");
      fullPath.append (dir[i]);
      filesInPath.insert(i, fullPath);
    }

    PlaylistOperation::insert(&filesInPath, playlist, -1);
    return false;
  }
  
  // here should be a check for valid/registered file types/extensions
  // but for now... we accept everything
  // Note: The test for is mp3 extension won't work anymore if kmpg plays .wav
  // files. Is there a KDE class for this which checks if this
  // player is somewhere registered for this file extension??
  dest->append(file);
  return true;
}
      
    


 

void PlaylistOperation::playNextNotCorrupt(Playlist* playlist) {
  int count=playlist->count();
  int play=playlist->getCurrentPos();
  Song* song;
  for(int i=play;i<count;i++) {
    song=playlist->at(i);
    if (song->getCorrupt()) {
      continue;
    }
    playlist->setCurrentPos(i);
    PlaylistOperation::playCurrent(playlist);  
    break;
  }	
}



void PlaylistOperation::playPrevNotCorrupt(Playlist* playlist) {
  int play=playlist->getCurrentPos();
  Song* song;
  for(int i=play;i>=0;i--) {
    song=playlist->at(i);
    if (song->getCorrupt()) {
      continue;
    }
    playlist->setCurrentPos(i);
    PlaylistOperation::playCurrent(playlist);  
    break;
  }
}

 
