/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include <stdio.h>
#include "songTree.h"
#include <iostream.h>
#include <qprogressdialog.h>
#include <qkeycode.h>



SongTree::SongTree(QWidget *parent, const char *name)
  :KDNDWidget(parent,name) {
  setupSurface("");
  pressed = false;
}

SongTree::SongTree(QString bName, QWidget *parent, const char *name)
  :KDNDWidget(parent,name) {
  setupSurface(bName);
}

void SongTree::setupSurface(QString bName){
  setMinimumSize(50,100);
  resize(100,200);

  displayMode = _START_SORT_ORDER;

  icon_loader = KApplication::getKApplication()->getIconLoader();                 
  if(strcmp(bName,"")==0)
    box = new QGroupBox(this,"box");
  else
    box = new QGroupBox(bName, this,"box");

  songTree = new KTreeView(this,"songs");
  songTree->setIndentSpacing(15);
  songTree->setFocusPolicy(QWidget::ClickFocus);
  songTree->setExpandButtonDrawing (true);
  setFocusPolicy(QWidget::StrongFocus);
  setFocusProxy (songTree);
  
  rightButtonMenu = new KPopupMenu(0, "rightbuttonmenu");
  rightButtonMenu->setTitle(i18n("What do you want to do?"));
  rightButtonMenu->insertItem(i18n("Show song info"),1);
  rightButtonMenu->insertItem(i18n("Play song immediately"),2);

  connect(songTree,SIGNAL(selected(int)),this,SLOT(doubleClick(int)));
  connect(songTree,SIGNAL(highlighted(int)),this,SLOT(selectionChanged(int)));
  connect(songTree,SIGNAL(rightPressed(int,const QPoint &)), this, SLOT(rightButtonPressed(int,const QPoint &)));
}

void SongTree::selectionChanged(int){
  emit selectionChanged();
}

void SongTree::clearSelection(){
  if(songTree->count() <= 0) return;
  songTree->setCurrentItem(-1);
}

void SongTree::doubleClick(int index){
  if(songTree->count() <= 0) return;
  if(songTree->itemAt(index)->hasChild()){
    if (songTree->getCurrentItem()->isExpanded())                
      songTree->collapseItem(songTree->currentItem(),false);
    else
      songTree->expandItem(songTree->currentItem(),false);
  }
  else
    emit doubleClick();
}


void SongTree::rightButtonPressed(int row, const QPoint &pos){
  if(row==-1) return;
  Song song;
  KTreeViewItem *item = songTree->itemAt(row);
  if(item->hasChild()) return ;
  song = getSelection(item);

  songTree->setCurrentItem(row);

  int result = rightButtonMenu->exec(mapToGlobal(pos));
  if(! ((result==1) || (result==2)) ) return ;

  if(result == 1) emit showMP3Info(song);
  else emit playSong(song);
}


void SongTree::addSongs(QSongList songs){
  KApplication::setOverrideCursor( waitCursor );
  songTree->setAutoUpdate(false);

  if (displayMode == _SORT_BY_ARTIST) addSongsSortByArtist(songs);
  if (displayMode == _SORT_BY_TITLE ) addSongsSortByTitle (songs);

  songTree->setAutoUpdate(true);
  songTree->repaint();
  KApplication::restoreOverrideCursor();
}

void SongTree::addSongsSortByArtist(QSongList songs){
  KPath *path;
  Song *song;
  QString artist = "";
#include <pics/mini/mini_artist.xpm>
  const QPixmap artist_pix((const char**) artist_pixmap);

  if(songs.count() <= 0) return;

  path = new KPath();
  path->push(&artist);
  for(song=songs.first(); song!=NULL; song=songs.next()){
    if(strcmp(artist,song->getArtist())!=0){
      artist = song->getArtist();
      path->pop();
      path->push(&artist);
      songTree->insertItem(artist.data(),  artist_pix);
      addChildAlbumTitle_SortByArtist(song, path, true);
    }
    else
      addChildAlbumTitle_SortByArtist(song, path, false);
  }
  delete (path);
}

void SongTree::addChildAlbumTitle_SortByArtist(Song *song, KPath *path, bool newArtist){
  static QString album  = "";
#include <pics/mini/mini_cd.xpm>
  const QPixmap album_pix((const char**) album_pixmap);
#include <pics/mini/mini_sound.xpm>
  const QPixmap title_pix((const char**) title_pixmap);

  if( newArtist || (strcmp(album,song->getAlbum())!=0) ){
    album = song->getAlbum();
    songTree->appendChildItem(song->getAlbum(), album_pix, *path);
  }
  path->push(&album);
  songTree->appendChildItem(song->getTitle(), title_pix, *path);
  path->pop();
}

void SongTree::addSongsSortByTitle(QSongList songs){
  KPath *path;
  Song *song;
  QString title = "";
  path = new KPath();
  path->push(&title);
#include <pics/mini/mini_sound.xpm>
  const QPixmap title_pix((const char**) title_pixmap);
#include <pics/mini/mini_artist.xpm>
  const QPixmap artist_pix((const char**) artist_pixmap);

  if(songs.count()<= 0) return;

  for(song=songs.first(); song!=NULL; song=songs.next()){
    if(strcmp(title,song->getTitle())!=0){
      title = song->getTitle();
      path->pop();
      path->push(&title);
      songTree->insertItem(song->getTitle(), title_pix);
      songTree->appendChildItem(song->getArtist(), artist_pix, *path);
    }
    else{
      songTree->appendChildItem(song->getArtist(), artist_pix, *path);
    }
  }
  delete (path);
}

void SongTree::addChildArtistAlbum_SortByTitle(Song *song, KPath *path, bool newTitle){
#include <pics/mini/mini_artist.xpm>
  const QPixmap artist_pix((const char**) artist_pixmap);
  static QString artist  = "";
  if(newTitle || (strcmp(artist,song->getArtist())!=0)){
    artist = song->getArtist();
    songTree->appendChildItem(song->getArtist(), artist_pix, *path);
  }
}


Song SongTree::getSelection(){
  Song song;
  KTreeViewItem *item;

  if(songTree->count() <= 0) return song;
  if(songTree->currentItem() == -1) { 
    return song; 
  }
  
  item = songTree->itemAt(songTree->currentItem());
  return getSelection(item);
}


Song SongTree::getSelection(KTreeViewItem *item){
  Song song;
  const char *level1 = NULL;
  const char *level2 = NULL;
  const char *level3 = NULL;

  if(item == 0) return song;
  
  level1 = item->getText();

  item = item->getParent();
  if (item != 0){
    level2 = item->getText();
    item = item->getParent();
    if (item != 0){
      level3 = item->getText();
    }
  }  
  if (displayMode == _SORT_BY_ARTIST) { /* Artist Album Title */
    if (level3 != NULL) {
      song.setArtist(level3,false);
      song.setAlbum (level2,false);
      song.setTitle (level1);      
    }
    else {
      if (level2 != NULL) {
      song.setArtist(level2,false);
      song.setAlbum (level1);      
      }
      else 
	song.setArtist(level1);
    }
  }

  if (displayMode == _SORT_BY_TITLE) { /* Title Artist Album */
    if (level2 != NULL) {
      song.setTitle (level2,false);
      song.setArtist (level1);      
    }
    else 
      song.setTitle(level1);
  }
  return song;
}

void SongTree::clear(){
  clearSelection();
  songTree->clear();
}

void SongTree::setDisplayOrder(int mode){
  displayMode = mode;
}


void SongTree::dndMouseReleaseEvent ( QMouseEvent * ){
  pressed = false;
}

void SongTree::mousePressEvent ( QMouseEvent * _mouse ){
  pressed = true;
  press_x = _mouse->pos().x();
  press_y = _mouse->pos().y();
  debugOutput( cout << "pressed: " << pressed << " x: " << press_x << " Y: " << press_y << endl );
}

void SongTree::mouseReleaseEvent ( QMouseEvent * _mouse ){
  pressed = false;
  debugOutput( cout << "pressed: " << pressed << endl );
}

void SongTree::dndMouseMoveEvent( QMouseEvent * _mouse ){
  // 'pressed' is set in mousePressedEvent(...)
  debugOutput( cout << "dndMouseMoveEvent" << endl ); 

  if ( !pressed )  return;

  int x = _mouse->pos().x();
  int y = _mouse->pos().y();

  if ( abs( x - press_x ) > Dnd_X_Precision || abs( y - press_y ) > Dnd_Y_Precision ){
    QString data = "KJukeBox: Transfer me";
    QPoint p = mapToGlobal( _mouse->pos() );

#include <pics/dnd_mp3_info.xpm>
    
    QPixmap pixmap((const char **)dnd_mp3_info);
    int dx = - pixmap.width() / 2;
    int dy = - pixmap.height() / 2;
    
    startDrag( new KDNDIcon( pixmap, p.x() + dx, p.y() + dy ), data.data(), data.length(), DndText, dx, dy );
  }  
}

void SongTree::resizeEvent(QResizeEvent *){
  box->setGeometry(2,0,width()-2,height());
  songTree->setGeometry(5,20,box->width()-10,box->height()-25);
}

