    /*

    Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */


#include "structures.h"
#include "sequenceutils.h"
#include "debug.h"
#include <vector>
#include <algorithm>

#define dname(dir) ((dir)==Arts::input?"input":"output")
#define pstat \
	printf("port name %s, direction %s, id %d\n",_Name.c_str(),dname(_Type.Direction),_ID);

PortDesc_impl::PortDesc_impl(Arts::ModuleDesc_ptr parent, string name,
															Arts::PortType type)
{
	if(parent)
	{
		char * pname = parent->Name();
		describe("PortDesc."+string(pname)+string(".")+name);
	}
	else
	{
		describe("PortDesc.Structure."+name);
	}
	_Name = name;
	_Type = type;
	_Parent = parent;
	_isConnected = false;
	_hasValue = false;
	_Connections = new Arts::PortDescSeq;
	_FloatValue = 0.0;
	_StringValue = "";

	if(parent)
	{
		Arts::StructureDesc_var sd = parent->Parent();
		_ID = sd->obtainID();
	}
	// else: assume that some smart object which derives from us will set the ID accordingly
	//  -> for instance StructurePortDesc_impl does so
}

void PortDesc_impl::cleanUp()
{
	disconnectAll();
	delete _Connections;
}

void PortDesc_impl::disconnectAll()
{
	// disconnect all connected ports
	while(_Connections->length())
		(*_Connections)[0]->disconnectFrom(this);
}

PortDesc_impl::~PortDesc_impl()
{
}

// Implementation for interface PortDesc
CORBA::Long PortDesc_impl::ID()
{
	return _ID; 
}

Arts::ModuleDesc_ptr PortDesc_impl::Parent()
{
	return Arts::ModuleDesc::_duplicate(_Parent); 
}

char* PortDesc_impl::Name()
{
	return CORBA::string_dup(_Name.c_str()); 
}

Arts::PortType PortDesc_impl::Type()
{
	return _Type; 
}

CORBA::Boolean PortDesc_impl::isConnected()
{
	return _isConnected; 
}

CORBA::Boolean PortDesc_impl::hasValue()
{
	return _hasValue; 
}

void PortDesc_impl::hasValue(CORBA::Boolean newvalue)
{
	if(_hasValue != newvalue)
	{
		assert(newvalue == false);
		_hasValue = newvalue;
	}
}

Arts::PortDescSeq* PortDesc_impl::Connections()
{
	Arts::PortDescSeq* retval = new Arts::PortDescSeq(*_Connections);
	return retval; 
}

CORBA::Float PortDesc_impl::FloatValue()
{
	assert(_hasValue);
	assert(_Type.DataType == Arts::audio_data);
	return _FloatValue; 
}

void PortDesc_impl::FloatValue( CORBA::Float _new_value )
{
	assert(!_isConnected);
	assert(_Type.Direction == Arts::input);
	assert(_Type.DataType == Arts::audio_data);
	_FloatValue = _new_value;
	_hasValue = true;
}

char* PortDesc_impl::StringValue()
{
	assert(_hasValue);
	assert(_Type.DataType == Arts::string_data);
	return CORBA::string_dup(_StringValue.c_str());
}

void PortDesc_impl::StringValue( const char* _new_value )
{
	assert(!_isConnected);	// shouldn't happen, but check anyway
	assert(_Type.Direction == Arts::input);
	assert(_Type.DataType == Arts::string_data);
	_StringValue = _new_value;
	_hasValue = true;
}

CORBA::Boolean
PortDesc_impl::connectTo( Arts::PortDesc_ptr port )
{
	// check if we are already connected to that port:

	unsigned long i;
	for(i=0;i<_Connections->length();i++)
		if((*_Connections)[i]->ID() == port->ID()) return true;

	Arts::PortType rType = port->Type();

	// only stream or event channels may be connected
	if( _Type.ConnType != rType.ConnType )
		return false;

	// TODO: eventually check conditions when it is legal to connect property
	// ports, and when it is insane (_Type.ConnType == Arts::property)
	//
	// for incoming structure ports, for instance, it is perfectly allright

	// only same data type connections allowed
	if( _Type.DataType != rType.DataType ) return false;

	// only opposite directions
	if( _Type.Direction == rType.Direction ) return false;

	// always first connect the input port to the output port and
	// then the other direction

	if( _Type.Direction == Arts::input )
	{
		if(!_isConnected)
		{
			assert(_Connections->length() == 0);
			_Connections->length(1);
			(*_Connections)[0] = Arts::PortDesc::_duplicate(port);

			port->internalConnectInput(this);

			_isConnected = true;
			_hasValue = false;
			return true;
		}
	}
	if( _Type.Direction == Arts::output )
	{
		return port->connectTo(this);
	}
	return false;
}

void PortDesc_impl::internalConnectInput( Arts::PortDesc_ptr port )
{
	unsigned long l = _Connections->length();

	_Connections->length(l+1);
	(*_Connections)[l] = Arts::PortDesc::_duplicate(port);
	_isConnected = true;
}

void
PortDesc_impl::disconnectFrom( Arts::PortDesc_ptr port )
{
	unsigned long i,found = 0,newlen = 0;
	Arts::PortDescSeq *_NConnections = new Arts::PortDescSeq;

	artsdebug("port %ld disconnecting from port %ld\n",ID(),port->ID());

	for(i=0;i<_Connections->length();i++)
	{
		Arts::PortDesc_ptr other = (*_Connections)[i];
		if(other->ID() != port->ID())
		{
			_NConnections->length(newlen+1);
			(*_NConnections)[newlen] =
				Arts::PortDesc::_duplicate((*_Connections)[i]);
			newlen++;
		}
		else found++;
	}

	delete _Connections;
	_Connections = _NConnections;

	_isConnected = (_Connections->length()>0);
	if(_Parent == 0)
		artsdebug("_Parent = <some structure>, isConnected = %d\n",_isConnected);
	else
		artsdebug("_Parent = %s, isConnected = %d\n",_Parent->Name(),_isConnected);
	if(found) port->disconnectFrom(this);
}

void ModuleDesc_impl::cleanUp()
{
	unsigned long i;
	for(i=0;i<_Ports->length();i++)
		(*_Ports)[i]->decRef();
	delete _Ports;
}

// Implementation for interface ModuleDesc
CORBA::Long ModuleDesc_impl::ID()
{
	return _ID; 
}

Arts::StructureDesc_ptr ModuleDesc_impl::Parent()
{
	return Arts::StructureDesc::_duplicate(_Parent); 
}

char* ModuleDesc_impl::Name()
{
	return CORBA::string_dup(_Name.c_str()); 
}

Arts::PortDescSeq* ModuleDesc_impl::Ports()
{
	Arts::PortDescSeq* retval = new Arts::PortDescSeq(*_Ports);
	return retval; 
}

CORBA::Long ModuleDesc_impl::X()
{
	return _X;
}

CORBA::Long ModuleDesc_impl::Y()
{
	return _Y;
}

CORBA::Long ModuleDesc_impl::Width()
{
	assert(false);
	return 0; 
}

CORBA::Long ModuleDesc_impl::Height()
{
	assert(false);
	return 0; 
}


CORBA::Boolean
ModuleDesc_impl::moveTo( CORBA::Long X, CORBA::Long Y )
{
	// FIXME: collision checking!
	_X = X;
	_Y = Y;

	return(true);
}


// Implementation for interface StructureDesc
CORBA::Long StructureDesc_impl::Width()
{
	assert(false);
	return 0; 
}

CORBA::Long StructureDesc_impl::Height()
{
	assert(false);
	return 0; 
}

/*
 * Query the module for it's paramenters
 */

ModuleDesc_impl::ModuleDesc_impl( Arts::StructureDesc_ptr parent,
									Arts::ModuleInfo *info )
{
	describe(string("ModuleDesc ")+string(info->name));
	_Name = info->name;
	_X = -1;		// no position assigned
	_Y = -1;
	_ID = parent->obtainID();
	_Parent = parent;
	_Ports = new Arts::PortDescSeq;
	_Ports->length(0);
	_isInterface = info->isInterface;
	_isStructure = info->isStructure;

	collectPorts(info);
}

ModuleDesc_impl::~ModuleDesc_impl()
{
}

CORBA::Boolean ModuleDesc_impl::isInterface()
{
	return _isInterface;
}

CORBA::Boolean ModuleDesc_impl::isStructure()
{
	return _isStructure;
}

Arts::PortDesc *ModuleDesc_impl::findPort(const char *name)
{
	unsigned long p;

	for(p=0;p<_Ports->length();p++)
	{
		Arts::PortDesc_ptr pd = (*_Ports)[p];
		CORBA::String_var pname = pd->Name();

		if(strcmp((const char *)pname,name) == 0)
			return(Arts::PortDesc::_duplicate(pd));
	}
	return(0);
}

long ModuleDesc_impl::collectPorts( Arts::ModuleInfo *info )
{
	string portname;
	long portcount = 0;
	long len;

	unsigned long i;

	Arts::PortDesc_ptr port;

	for(i=0;i<info->ports.length();i++)
	{
		Arts::PortType porttype = info->ports[i];
		portname = info->portnames[i];

		artsdebug("#%d: %s\n",portcount,portname.c_str());

		port = new PortDesc_impl(this, portname, porttype);
		port->incRef();			// this is my port

		len = _Ports->length();
		_Ports->length(len+1);
		(*_Ports)[len] = Arts::PortDesc::_duplicate(port);

		portcount++;
	};
	return(portcount);
}

Arts::ModuleDesc_ptr
StructureDesc_impl::createModuleDesc( const Arts::ModuleInfo& info )
{
	Arts::ModuleDesc_ptr result = createModuleDesc(info.name);

	assert(result);
	return result;
}

Arts::ModuleDesc_ptr StructureDesc_impl::createModuleDesc( const char *name )
{
	/* FIXME: need new comment
	 * to create a representation of a specified synth module, we
	 *
	 * - create an instance of this synth module by contacting the
	 *   module server and telling him to do so (result is a C++ class)
	 *
	 * - create an instance of a ModuleDesc, and tell it to query the
	 *   module for it's parameters (result is a CORBA object)
	 *
	 * - destroy the synth module (C++ class) again and return a reference
	 *   to the CORBA object
	 */
/*
	ModuleServer<SynthModule> *MS_SynthModule;
	MS_SynthModule = (ModuleServer<SynthModule> *)SynthModule::get_MS();

	SynthModule *m = (SynthModule *)MS_SynthModule->getModule(name);
*/
	Arts::ModuleInfo_var info = ModuleBroker->lookupModule(name);
	if(!info) return 0;

  	Arts::ModuleDesc_ptr moduledesc =
		Arts::ModuleDesc::_duplicate(new ModuleDesc_impl(this, info));

	long mcount = _Modules->length();
	_Modules->length(mcount+1);
	(*_Modules)[mcount++] = Arts::ModuleDesc::_duplicate(moduledesc);
	moduledesc->incRef();	// this is my! module now ;)

	if(moduledesc->isStructure())
		subStructureCount++;

	return moduledesc;
}

void StructureDesc_impl::freeModuleDesc(Arts::ModuleDesc_ptr moduledesc)
{
	Arts::ModuleDescSeq *new_Modules = new Arts::ModuleDescSeq;
	unsigned long i,newcount = 0;

	for(i=0;i<_Modules->length();i++)
	{
		Arts::ModuleDesc *current = (*_Modules)[i];

		if(current->ID() != moduledesc->ID())
		{
			// ok, this should not be deleted

			new_Modules->length(newcount+1);
			(*new_Modules)[newcount] =
				Arts::ModuleDesc::_duplicate((*_Modules)[i]);
			newcount++;
		}
		else
		{
			if(current->isStructure()) subStructureCount--;
			current->decRef();		// will get freed automagically
		}
	}
	delete _Modules;
	_Modules = new_Modules;
}

Arts::ModuleDescSeq *StructureDesc_impl::Modules()
{
	Arts::ModuleDescSeq *retval = new Arts::ModuleDescSeq(*_Modules);
	return(retval);
}

StructureDesc_impl::StructureDesc_impl(Arts::ModuleBroker *myModuleBroker)
{
	describe("StructureDesc");
	nextID = 0;
	subStructureCount = 0;
	_valid = true;
	_ExternalInterface.name = (const char *)"unknown";		// FIXME
	_ExternalInterface.isStructure = true;
	_ExternalInterface.isInterface = false;
	_Modules = new Arts::ModuleDescSeq;
	_Ports = new Arts::StructurePortDescSeq;
	ModuleBroker = myModuleBroker;
	ModuleBroker->incRef();
}

StructureDesc_impl::~StructureDesc_impl()
{
	ModuleBroker->decRef();
	artsdebug("StructureDesc released...\n");
}

CORBA::Long StructureDesc_impl::obtainID()
{
	return(nextID++);
}

CORBA::Boolean StructureDesc_impl::valid()
{
	return(_valid);
}

void StructureDesc_impl::cleanUp()
{	
	artsdebug("cleanUp...\n");
	clear();							// frees all modules
	delete _Modules;
	delete _Ports;
}

void StructureDesc_impl::clear()
{
	unsigned long i;

	for(i=0;i<_Modules->length();i++)
		(*_Modules)[i]->decRef();		// I don't need that module any longer
	_Modules->length(0);				// should remove every module

	for(i=0;i<_Ports->length();i++)		// Same freeing for ports
		(*_Ports)[i]->decRef();			// ... all structure ports should be
	_Ports->length(0);					// now

	_valid = true;
	subStructureCount = 0;
}

CORBA::Boolean StructureDesc_impl::containsStructures()
{
	return (subStructureCount != 0);
}

// "file" management

Arts::StringSeq *PortDesc_impl::saveToList()
{
	Arts::StringSeq *list = new Arts::StringSeq;

	sqprintf(list,"id=%ld",_ID);
	if(_hasValue)
	{
		if(_Type.DataType == Arts::string_data)
		{
			sqprintf(list,"string_data=%s",_StringValue.c_str());
		}
		if(_Type.DataType == Arts::audio_data)
		{
			sqprintf(list,"audio_data=%2.5f",_FloatValue);
		}
	}

	if(_isConnected)
	{
		unsigned long i;
		for(i=0;i<_Connections->length();i++)
		{
			Arts::PortDesc_ptr port = (*_Connections)[i];
			
			sqprintf(list,"connect_to=%ld",port->ID());
		}
	}
	return list;
}

Arts::StringSeq *ModuleDesc_impl::saveToList()
{
	Arts::StringSeq *list = new Arts::StringSeq;
	Arts::PortDesc *pd;

	unsigned long i;

	sqprintf(list,"id=%ld",_ID);
	sqprintf(list,"x=%ld",_X);
	sqprintf(list,"y=%ld",_Y);
	for(i=0;i<_Ports->length();i++)
	{
		pd = (*_Ports)[i];
		sqprintf(list,"port=%s",pd->Name());

		Arts::StringSeq_var portlist = pd->saveToList();
		addSubStringSeq(list,portlist);
	}
	return list;
}

Arts::StringSeq *StructureDesc_impl::saveToList()
{
	Arts::StringSeq *list = new Arts::StringSeq;
	Arts::ModuleDesc *md;

	unsigned long i;

	sqprintf(list,"name=%s",(const char *)_ExternalInterface.name);
	for(i=0;i<_Modules->length();i++)
	{
		md = (*_Modules)[i];
		sqprintf(list,"module=%s",md->Name());

		Arts::StringSeq_var modulelist = md->saveToList();
		addSubStringSeq(list,modulelist);
	}
	for(i=0;i<_Ports->length();i++)
	{
		Arts::StructurePortDesc *spd = (*_Ports)[i];
		sqprintf(list,"structureport");

		Arts::StringSeq_var modulelist = spd->saveToList();
		addSubStringSeq(list,modulelist);
	}
	return list;
}

void PortDesc_impl::internalReConnect( const Arts::PortDescSeq& allports )
{
	unsigned long i;
	for(i=0;i<allports.length();i++)
	{
		long oid = allports[i]->internalOldID();

		if(find(oldConnections.begin(),oldConnections.end(),oid)
													!= oldConnections.end())
		{
			connectTo(allports[i]);
		}
	}
}

CORBA::Long PortDesc_impl::internalOldID()
{
	return _OldID;
}

void PortDesc_impl::loadFromList(const Arts::StringSeq& list)
{
	unsigned long i;
	char *cmd,*param;
	for(i=0;i<list.length();i++)
	{
		if(parse_line(list[i],cmd,param))	// otherwise: empty or comment
		{
			if(strcmp(cmd,"audio_data") == 0) {
				FloatValue(atof(param));
			} else if(strcmp(cmd,"string_data") == 0) {
				StringValue(param);
			} else if(strcmp(cmd,"id") == 0) {
				_OldID = atol(param);
			} else if(strcmp(cmd,"connect_to") == 0) {
				oldConnections.push_back(atol(param));
			}
		}
	}
}

void ModuleDesc_impl::loadFromList(const Arts::StringSeq& list)
{
	artsdebug("mlist-----------\n");
	unsigned long i;
	char *cmd,*param;
	for(i=0;i<list.length();i++)
	{
		if(parse_line(list[i],cmd,param))	// otherwise: empty or comment
		{
			artsdebug("MD: load-> cmd was %s\n",cmd);
			if(strcmp(cmd,"port") == 0)
			{
				Arts::PortDesc_ptr pd = 0,pdi;

				unsigned long pi;
				for(pi=0;pi<_Ports->length();pi++)
				{
					pdi = (*_Ports)[pi];
					artsdebug("pdi = %s, param = %s\n",pdi->Name(),param);
					if(strcmp(pdi->Name(),param) == 0) pd = pdi;
				}
				assert(pd);

				Arts::StringSeq_var plist = getSubStringSeq(&list,i);

				pd->loadFromList(*plist);
			} else if(strcmp(cmd,"x") == 0) {
				_X = atol(param);
				artsdebug("X set to %ld (param was %s)\n",_X,param);
			} else if(strcmp(cmd,"y") == 0) {
				_Y = atol(param);
				artsdebug("Y set to %ld (param was %s)\n",_Y,param);
			}
		}
	}
	artsdebug("-----------mlist\n");
}

void StructureDesc_impl::loadFromList(const Arts::StringSeq& list)
{
	char *cmd,*param;
	unsigned long pi,i,portcount = 0;
	Arts::PortDescSeq allports;

	_ExternalInterface.name = (const char *)"unknown";

	artsdebug("loadFromList; listlen = %ld\n",list.length());
	for(i=0;i<list.length();i++)
	{
		if(parse_line(list[i],cmd,param))	// otherwise: empty or comment
		{
			artsdebug("SD: load-> cmd was %s\n",cmd);
			if(strcmp(cmd,"module") == 0)
			{
				char *module_name = strdup(param);
				// the strdup is a special trick, since reentrancy will
				// happen here: since creating a ModuleDesc may mean
				// loading a structure from artspath, another structure
				// may get loaded and overwrite the contents of param
				// (as param is static of parse_line)
				Arts::ModuleDesc_var md = createModuleDesc(module_name);
				free(module_name);

				Arts::StringSeq_var mlist = getSubStringSeq(&list,i);

				if(md)
				{
					Arts::PortDescSeq_var pd = md->Ports();

					md->loadFromList(*mlist);

					for(pi = 0; pi < pd->length();pi++) 
					{
						allports.length(portcount+1);
						allports[portcount++] =
							Arts::PortDesc::_duplicate((*pd)[pi]);
					}
				}
				else
				{
					// module couldn't be found
					_valid = false;
				}
			}
			else if(strcmp(cmd,"name") == 0)
			{
				_ExternalInterface.name = (const char *)param;
			}
			else if(strcmp(cmd,"structureport") == 0)
			{
				// just to have valid values to pass to the new (to be loaded)
				// port:
				Arts::PortType type;
				type.Direction = Arts::input;
				type.DataType = Arts::audio_data;
				type.ConnType = Arts::stream;
				const char *newportname = "unknown";

				Arts::StringSeq_var splist = getSubStringSeq(&list,i);
				Arts::StructurePortDesc_var spd =
					createStructurePortDesc(type,newportname);

				spd->loadFromList(*splist);

				// yes; this is a port as well
				allports.length(portcount+1);
				allports[portcount++]=Arts::PortDesc::_duplicate(spd);
			}
		}
	}

	for(i=0;i<allports.length();i++)
		allports[i]->internalReConnect(allports);
}

void StructureDesc_impl::Name(const char *name)
{
	_ExternalInterface.name = name;
}

char *StructureDesc_impl::Name()
{
	return CORBA::string_dup(_ExternalInterface.name);
}

long extint_pscore(Arts::StructurePortDesc_ptr p)
{
	long result = p->Position(); //p->X()*1000+p->Y();
	if(p->Type().Direction == Arts::input) result += 5000000;

	return result;
}

bool extint_port_compare(Arts::StructurePortDesc_ptr p1, Arts::StructurePortDesc_ptr p2)
{
	long p1s = extint_pscore(p1);
	long p2s = extint_pscore(p2);

	CORBA::String_var p1n = p1->Name(), p2n = p2->Name();

	artsdebug("compare; [%s] = %d  ;  [%s] = %d\n",	(const char *)p1n,p1s,(const char *)p2n,p2s);
	return (p1s < p2s);
// return -1;
	//if(p1s == p2s) return 0;
	//return 1;
}

Arts::ModuleInfo *StructureDesc_impl::ExternalInterface()
{
	Arts::ModuleInfo *result = new Arts::ModuleInfo(_ExternalInterface);
	vector<Arts::StructurePortDesc_ptr> sorted_ports;
	vector<Arts::StructurePortDesc_ptr>::iterator p;
	unsigned long l;

	for(l=0;l<_Ports->length();l++) sorted_ports.push_back((*_Ports)[l]);

	sort(sorted_ports.begin(),sorted_ports.end(),extint_port_compare);

	result->ports.length(_Ports->length());
	result->portnames.length(_Ports->length());

	l = 0;
	for(p=sorted_ports.begin();p != sorted_ports.end();p++)
	{
		CORBA::String_var pname = (*p)->Name();
		Arts::PortType ptype = (*p)->Type();

		if(ptype.Direction == Arts::input)
			ptype.Direction = Arts::output;
		else
			ptype.Direction = Arts::input;

		artsdebug("ExternalInterface; sorted ports: %d => %s\n",l,(const char *)pname);
		result->ports[l] = ptype;
		result->portnames[l] = CORBA::string_dup(pname);
		l++;
	}
	return result;
}

Arts::StructurePortDescSeq *StructureDesc_impl::Ports()
{
	return new Arts::StructurePortDescSeq(*_Ports);
}

Arts::StructurePortDesc *StructureDesc_impl::createStructurePortDesc(
		const Arts::PortType& type, const char *name)
{
	unsigned long len;

	artsdebug("creating new port %s\n",name);
	Arts::StructurePortDesc_ptr port = new StructurePortDesc_impl(this, string(name), type);
	port->incRef();			// this is my port

	len = _Ports->length();
	_Ports->length(len+1);
	(*_Ports)[len] = Arts::StructurePortDesc::_duplicate(port);

	// set the Position (put it at the end of the ports)
	unsigned long i, count = 0;
	for(i=0;i<_Ports->length();i++)
	{
		if((*_Ports)[i]->Type().Direction == type.Direction) count++;
	}
	assert(count > 0);	// we just inserted one ;)
	port->internalSetPosition(count-1);

	// FIXME: duplicate necessary?
	return Arts::StructurePortDesc::_duplicate(port);
}

void StructureDesc_impl::freeStructurePortDesc(Arts::StructurePortDesc
																	*portdesc)
{
	Arts::StructurePortDescSeq *new_Ports = new Arts::StructurePortDescSeq;
	unsigned long i,newcount = 0;

	for(i=0;i<_Ports->length();i++)
	{
		if((*_Ports)[i]->ID() != portdesc->ID())
		{
			// ok, this should not be deleted

			new_Ports->length(newcount+1);
			(*new_Ports)[newcount] =
				Arts::StructurePortDesc::_duplicate((*_Ports)[i]);
			newcount++;
		}
		else
		{
			(*_Ports)[i]->decRef();		// will get freed automagically
		}
	}
	delete _Ports;
	_Ports = new_Ports;
}

void StructureDesc_impl::moveStructurePortDesc(Arts::StructurePortDesc
											*portdesc, long newposition)
{
	Arts::PortType type = portdesc->Type();

	long count = 0;
	unsigned long i;

	for(i=0;i<_Ports->length();i++)
	{
		if((*_Ports)[i]->Type().Direction == type.Direction) count++;
	}

	if(newposition < 0) newposition = 0;
	if(newposition > count-1) newposition = count-1;

	if(newposition == portdesc->Position()) return;

	int delta, lower, upper;

	if(newposition > portdesc->Position())
	{
		// if the port gets a higher position, move all ports that
		// are between it's current position and its new position down one
		lower = portdesc->Position();
		upper = newposition;
		delta = -1;
	}
	else
	{
		// if the port gets a lower position, move all ports that
		// are between it's current position and its new position up one
		lower = newposition;
		upper = portdesc->Position();
		delta = 1;
	}

	for(i=0;i<_Ports->length();i++)
	{
		Arts::StructurePortDesc *pd  = (*_Ports)[i];
		if(pd->Type().Direction == type.Direction)
		{
			if(pd->ID() != portdesc->ID() &&
				pd->Position() >= lower && pd->Position() <= upper)
			{
				pd->internalSetPosition(pd->Position()+delta);
			}
		}
	}
	portdesc->internalSetPosition(newposition);
}

StructurePortDesc_impl::StructurePortDesc_impl(Arts::StructureDesc_ptr parent, string name,
																			Arts::PortType type)
								: PortDesc_impl(0,name,type)
{
	_ParentStructure = parent;
	_ID = parent->obtainID();
	_X = 0;
	_Y = 0;
	_Position = 0;
}

StructurePortDesc_impl::~StructurePortDesc_impl()
{
}

CORBA::Long StructurePortDesc_impl::X()
{	
	return _X;
}

CORBA::Long StructurePortDesc_impl::Y()
{
	return _Y;
}

CORBA::Long StructurePortDesc_impl::Position()
{
	return _Position;
}

void StructurePortDesc_impl::lowerPosition()
{
	_ParentStructure->moveStructurePortDesc(this, _Position-1);
}

void StructurePortDesc_impl::raisePosition()
{
	_ParentStructure->moveStructurePortDesc(this, _Position+1);
}

void StructurePortDesc_impl::rename(const char *newname)
{
	_Name = newname;
}
 
// only used by the structure to reorder the ports
void StructurePortDesc_impl::internalSetPosition(long Position)
{
	_Position = Position;
}

Arts::StructureDesc_ptr StructurePortDesc_impl::ParentStructure()
{
	return Arts::StructureDesc::_duplicate(_ParentStructure);
}

CORBA::Boolean StructurePortDesc_impl::moveTo( CORBA::Long X, CORBA::Long Y )
{
	// FIXME: check space
	_X = X;
	_Y = Y;

	return true;
}

/*
  override load & save behaviour this kind of port requires that we save the type
  of the port as well, that means all of the porttype:

	enum PortDirection {input, output};
	enum PortDataType {audio_data, string_data};
	enum PortConnType {stream, event, property};
	struct PortType {
		PortDirection Direction;
		PortDataType DataType;
		PortConnType ConnType;
	};

  so when saved, it will look like that: 

	{
      name=fasel
      x=4
      y=2
	  type
	  {
	    direction=input/output
	    datatype=audio/string
	    conntype=stream/event/property
	  }
	  data
      {
	    [original port saves here]
	  }
	}
*/

Arts::PortType loadTypeFromList(const Arts::StringSeq& list)
{
	unsigned long i,loadstate = 0;
	char *cmd,*param;
	Arts::PortType result;

	for(i=0;i<list.length();i++)
	{
		if(parse_line(list[i],cmd,param))	// otherwise: empty or comment
		{
			artsdebug("PortType: load-> cmd was %s\n",cmd);
			if(strcmp(cmd,"direction") == 0)
			{
				if(strcmp(param,"input") == 0) {
					result.Direction = Arts::input;
				}
				else if(strcmp(param,"output") == 0) {
					result.Direction = Arts::output;
				}
				else assert(false);

				loadstate += 1;
			} else if(strcmp(cmd,"datatype") == 0) {
				if(strcmp(param,"audio") == 0) {
					result.DataType = Arts::audio_data;
				}
				else if(strcmp(param,"string") == 0) {
					result.DataType = Arts::string_data;
				}
				else assert(false);

				loadstate += 100;
			} else if(strcmp(cmd,"conntype") == 0) {
				if(strcmp(param,"stream") == 0) {
					result.ConnType = Arts::stream;
				}
				else if(strcmp(param,"event") == 0) {
					result.ConnType = Arts::event;
				}
				else if(strcmp(param,"property") == 0) {
					result.ConnType = Arts::property;
					artsdebug("got property stuff\n");
				}
				else assert(false);

				loadstate += 10000;
			}
		}
	}
	assert(loadstate == 10101); // should see every member exactly once
	return result;
}

void StructurePortDesc_impl::loadFromList(const Arts::StringSeq& list)
{
	artsdebug("structureportlist-----------\n");
	unsigned long i;
	char *cmd,*param;
	Arts::StringSeq_var typelist,datalist;
	bool haveType = false, haveData = false;
		// need both to do restore, type first

	for(i=0;i<list.length();i++)
	{
		if(parse_line(list[i],cmd,param))	// otherwise: empty or comment
		{
			artsdebug("StructurePortDesc: load-> cmd was %s\n",cmd);
			if(strcmp(cmd,"type") == 0)
			{
				assert(!haveType);	// only allowed once
				haveType = true;
				typelist = getSubStringSeq(&list,i);
			} else if(strcmp(cmd,"data") == 0) {
				assert(!haveData);	// only allowed once
				haveData = true;
				datalist = getSubStringSeq(&list,i);
			} else if(strcmp(cmd,"x") == 0) {
				_X = atol(param);
				artsdebug("X set to %ld (param was %s)\n",_X,param);
			} else if(strcmp(cmd,"y") == 0) {
				_Y = atol(param);
				artsdebug("Y set to %ld (param was %s)\n",_Y,param);
			} else if(strcmp(cmd,"position") == 0) {
				_Position = atol(param);
				artsdebug("Position set to %ld (param was %s)\n",_Position,param);
			} else if(strcmp(cmd,"name") == 0) {
				_Name = param;
				artsdebug("Name set to %s\n",_Name.c_str());
			}
		}
	}
	assert(haveType && haveData);

	_Type = loadTypeFromList(typelist);

	if(_Type.ConnType == Arts::property) artsdebug("have property here\n");
	PortDesc_impl::loadFromList(datalist);	
	artsdebug("-----------structureportlist\n");
}

Arts::StringSeq *saveTypeToList(Arts::PortType type)
{
	Arts::StringSeq *list = new Arts::StringSeq;

	switch(type.Direction)
	{
		case Arts::input: sqprintf(list,"direction=input");
					break;
		case Arts::output: sqprintf(list,"direction=output");
					break;
		default: assert(false);		// should never happen!
	}
	switch(type.DataType)
	{
		case Arts::audio_data: sqprintf(list,"datatype=audio");
					break;
		case Arts::string_data: sqprintf(list,"datatype=string");
					break;
		default: assert(false);		// should never happen!
	}
	switch(type.ConnType)
	{
		case Arts::stream: sqprintf(list,"conntype=stream");
					break;
		case Arts::event: sqprintf(list,"conntype=event");
					break;
		case Arts::property: sqprintf(list,"conntype=property");
					break;
		default: assert(false);		// should never happen!
	}
	
	return list;
}

Arts::StringSeq *StructurePortDesc_impl::saveToList()
{
	Arts::StringSeq *list = new Arts::StringSeq;
	sqprintf(list,"name=%s",_Name.c_str());
	sqprintf(list,"x=%ld",_X);
	sqprintf(list,"y=%ld",_Y);
	sqprintf(list,"position=%ld",_Position);
	sqprintf(list,"type");

	Arts::StringSeq_var typelist = saveTypeToList(_Type);
	addSubStringSeq(list,typelist);

	sqprintf(list,"data");
	Arts::StringSeq_var portlist = PortDesc_impl::saveToList();
	addSubStringSeq(list,portlist);
	return list;
}
