	/*

	Copyright (C) 1998-1999 Stefan Westerfeld
                            stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "bus.h"

static BusManager *the_BusManager = 0;

BusManager::BusManager()
{
	// this cosntructor isn't public (Singleton)
}

BusManager *BusManager::the()
{
	if(!the_BusManager) the_BusManager = new BusManager;
	return(the_BusManager);
}

BusManager::Bus *BusManager::findBus(string name)
{
	list<Bus *>::iterator bi;

	for(bi = BusList.begin(); bi != BusList.end(); bi++)
	{
		if((*bi)->name == name) return(*bi);
	}
	Bus *bus = new Bus;
	bus->name = name;
	BusList.push_back(bus);
	return(bus);
}

Arts::StringSeq *BusManager::busList()
{
	set<string> names;
	set<string>::iterator si;
	
	list<Bus *>::iterator bi;
	for(bi = BusList.begin(); bi != BusList.end(); bi++)
		names.insert((*bi)->name);

	Arts::StringSeq *bl = new Arts::StringSeq;

	unsigned long n = 0;
	bl->length(names.size());
	for(si=names.begin();si != names.end();si++)
		(*bl)[n++] = CORBA::string_dup((*si).c_str());

	return bl; 
}

void BusManager::reBuild(Bus *bus)
{
	list<BusClient *>::iterator client,server;
	long channels = 2;

	artsdebug("rebuilding bus %s\n",bus->name.c_str());
	artsdebug(" - %ld channels\n",channels);
	artsdebug(" - %d clients\n",bus->clients.size());
	artsdebug(" - %d servers\n",bus->servers.size());

	for(client = bus->clients.begin(); client != bus->clients.end(); client++)
	{
		artsdebug(" => client %s\n",(*client)->Module()->getClassName());
	}

	for(server = bus->servers.begin(); server != bus->servers.end(); server++)
	{
		artsdebug(" => server %s\n",(*server)->Module()->getClassName());
		(*server)->configureBus(bus->clients, channels);
	}
}

void BusManager::erase(BusClient *busclient)
{
	list<Bus *>::iterator bi;

	for(bi = BusList.begin(); bi != BusList.end(); bi++)
	{
		list<BusClient *>::iterator client,server;
		int hits = 0;
		Bus *bus = *bi;

	for(client = bus->clients.begin(); client != bus->clients.end(); client++)
		if((*client) == busclient)
		{
			bus->clients.erase(client);
			client = bus->clients.begin();
			hits++;
		}

	for(server = bus->servers.begin(); server != bus->servers.end(); server++)
		if((*server) == busclient)
		{
			bus->clients.erase(server);
			server = bus->servers.begin();
			hits++;
		}
		if(hits != 0) reBuild(bus);
	}
}

void BusManager::addClient(string busname, BusClient *client)
{
	Bus *bus = findBus(busname);
	bus->clients.push_back(client);
	reBuild(bus);
}

void BusManager::removeClient(BusClient *client)
{
	erase(client);
}

void BusManager::addServer(string busname, BusClient *server)
{
	Bus *bus = findBus(busname);
	bus->servers.push_back(server);
	reBuild(bus);
}

void BusManager::removeServer(BusClient *server)
{
	erase(server);
}

class Synth_BUS_UPLINK :public SynthModule,BusClient
{
	// inputs
	enum { LEFT, RIGHT, PROP_BUSNAME };
	BusManager *bm;
	bool active, relink;
public:
	void Initialize();
	void DeInitialize();
	void Calculate() { /* dont calculate */ }
	void CalculateBlock(unsigned long samples) { /* dont calculate */ }
	string getParams() { return("left,right,_busname;"); }
	static void *Creator() { return new Synth_BUS_UPLINK; };
	SynthModule *Module();

	void propertyChanged(unsigned long index);
	void CallBack();

	void connect();
	void disconnect();
};

void Synth_BUS_UPLINK::Initialize()
{
	bm = BusManager::the();
	haveCalculateBlock = true;
	active = false;
	relink = false;

	connect();	// connect to the BusManager
}

void Synth_BUS_UPLINK::propertyChanged(unsigned long index)
{
	assert(index == PROP_BUSNAME);

	// to be sure that reconnection happens when outside the scheduling cycle
	relink = true;
	Synthesizer->requestCallBack(this);
}

void Synth_BUS_UPLINK::connect()
{
	const char *bname = getStringProperty(PROP_BUSNAME);

	assert(active == false);
	if(strlen(bname))
	{
		active = true;
		bm->addClient(bname, this);
	}
}

void Synth_BUS_UPLINK::disconnect()
{
	if(active == true)
	{
		bm->removeClient(this);
		active = false;
	}
}

void Synth_BUS_UPLINK::CallBack()
{
	if(relink)
	{
		disconnect();
		connect();
		relink = false;
	}
}

void Synth_BUS_UPLINK::DeInitialize()
{
	disconnect();
}

SynthModule *Synth_BUS_UPLINK::Module()
{
	return(this);
}

ModuleClient MC_Synth_BUS_UPLINK(SynthModule::get_MS,"Synth_BUS_UPLINK",Synth_BUS_UPLINK::Creator);

class Synth_BUS_DOWNLINK :public SynthModule,BusClient
{
	// inputs
	enum { CLIENTS, PROP_BUSNAME };

	// outputs
	enum { LEFT, RIGHT };

	bool active, relink;
	BusManager *bm;

	unsigned long bus_clients;
	unsigned long bus_channels;

	void connect();
	void disconnect();

public:
	void Initialize();
	void DeInitialize();
	void Calculate() { assert(false); /* only have CalculateBlock */ };
	void CallBack();
	void CalculateBlock(unsigned long samples);
	string getParams() { return("clients,_busname;left,right"); }
	static void *Creator() { return new Synth_BUS_DOWNLINK; };
	void configureBus(list<BusClient *>& clientlist, unsigned long channels);
	SynthModule *Module();

	void propertyChanged(unsigned long index);
};

void Synth_BUS_DOWNLINK::Initialize()
{
	bm = BusManager::the();
	active = relink = false;
	haveCalculateBlock = true;
	connect();
}

void Synth_BUS_DOWNLINK::DeInitialize()
{
	disconnect();
}

void Synth_BUS_DOWNLINK::connect()
{
	const char *bname = getStringProperty(PROP_BUSNAME);

	assert(active == false);
	if(strlen(bname))
	{
		active = true;
		bm->addServer(bname, this);
	}
}

void Synth_BUS_DOWNLINK::disconnect()
{
	if(active == true)
	{
		Synthesizer->removeDynamicConnections(this);
		bm->removeServer(this);
		active = false;
	}
}

void Synth_BUS_DOWNLINK::CallBack()
{
	if(relink)
	{
		disconnect();
		connect();
		relink = false;
	}
}

void Synth_BUS_DOWNLINK::propertyChanged(unsigned long index)
{
	assert(index == PROP_BUSNAME);

	// to be sure that reconnection happens when outside the scheduling cycle
	relink = true;
	Synthesizer->requestCallBack(this);
}

SynthModule *Synth_BUS_DOWNLINK::Module()
{
	return(this);
}

void Synth_BUS_DOWNLINK::configureBus(list<BusClient *>& clientlist, unsigned long channels)
{
	Synthesizer->removeDynamicConnections(this);

	list<BusClient *>::iterator i;
	for(i=clientlist.begin();i != clientlist.end();i++)
	{
		BusClient *client = *i;
		SynthModule *smod = client->Module();
		artsdebug("1 client is: %s\n",smod->getClassName());
		Synthesizer->addDynamicConnection(this,smod,channels);
	}
	bus_channels = channels;
	bus_clients = clientlist.size();
}

// sum up the data coming from the bus

void Synth_BUS_DOWNLINK::CalculateBlock(unsigned long samples)
{
	float *sig;

	unsigned long channel, client;

	for(channel = 0; channel < bus_channels; channel++)
	{
		float *out_start = out[channel], *out_end = out[channel]+samples;

		sig = out_start;
		while(sig < out_end) *sig++ = 0.0;		// first zero the fragment

		if(active)
		{
			for(client=0; client<bus_clients; client++)
			{
				float *input=in[1 + client*bus_channels + channel];
					// 1+ is ugly ;)
	
				assert(inConnCount >= 1 + client*bus_channels + channel);
				sig = out_start;
				while(sig < out_end) *sig++ += *input++;
												// add input data from client
			}

			float *gain = in[CLIENTS];
			sig = out_start;
			while(sig < out_end) *sig++ /= *gain++;  // scale down data
		}
	}
}

ModuleClient MC_Synth_BUS_DOWNLINK(SynthModule::get_MS,"Synth_BUS_DOWNLINK",Synth_BUS_DOWNLINK::Creator);
