    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>

#include <list>

#include "audioman_impl.h"
#include "artsorb.h"
#include "debug.h"
#include "synthmodule.h"
#include "structbuilder.h"
#include "synth_impl.h"
#include "convert.h"

// informations for potential telnet users (just forget them as "normal" client)
#define ARTS_MAN_INFO 800

// welcome
#define ARTS_MAN_WELCOME 200

// ior connection established
#define ARTS_MAN_CONNECT_IOR 100

AudioManager_impl::AudioManager_impl(Arts::Synthesizer_ptr Synthesizer)
{
	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);

	describe("AudioManager");

	maxClient = 1;
	_changes = 1;

	thePort = DEFAULT_AUDIOMANAGER_PORT;
	theSocket = open_listen_socket(thePort);
	assert(theSocket > 0);

	socketIOEvent = new ArtsIOEvent(this,theSocket,ArtsIOEvent::ioRead,0);
	socketIOEvent->start();
}

AudioManager_impl::~AudioManager_impl()
{
	delete socketIOEvent;
	close(theSocket);
}

void AudioManager_impl::notifyIO(int fd,long)
{
	assert(fd == theSocket);
	artsdebug("AudioManager: got client connect\n");

	int clientfd;
	struct sockaddr_in incoming;
	size_t size_in = sizeof(struct sockaddr_in);                                
	
	do {
		clientfd = accept(theSocket, (struct sockaddr*) &incoming, &size_in );
		if(clientfd > 0)
		{
			int nbl = 0; // set to 1?
			assert ( ioctl( clientfd, FIONBIO, &nbl ) >= 0 );

			// send welcome message

			char welcome[strlen(VERSION)+200];
			sprintf(welcome,"%d aRts audio server (version %s) ready.\n",
					ARTS_MAN_WELCOME,VERSION);
			write(clientfd,welcome,strlen(welcome));

			// TODO: multiple services should be possible,
			//   authentifications should be made

			// create object (this will handle all communication from now)

			AudioUplink_impl *uplink =
				new AudioUplink_impl(clientfd,Synthesizer,maxClient++,this);

			_uplinks.push_back(uplink);
			_changes++;
		}
	} while(clientfd > 0);
}

/* same as in esd (thanks esd people!) */

int AudioManager_impl::open_listen_socket( int port )
{
    /*********************/
    /* socket test setup */
    struct sockaddr_in socket_addr;
    int socket_listen;
    struct linger lin;
 
    /* create the socket, and set for non-blocking */
    socket_listen=socket(AF_INET,SOCK_STREAM,0);
    if (socket_listen<0)
    {
        fprintf(stderr,"Unable to create socket\n");
        return( -1 );
    }
    if (fcntl(socket_listen,F_SETFL,O_NONBLOCK)<0)
    {
        fprintf(stderr,"Unable to set socket to non-blocking\n");
        return( -1 );                                                           
    }
 
    /* set socket for linger? */
    lin.l_onoff=1;      /* block a closing socket for 1 second */
    lin.l_linger=100;   /* if data is waiting to be sent */
    if ( setsockopt( socket_listen, SOL_SOCKET, SO_LINGER,
                     &lin, sizeof(struct linger) ) < 0 )
    {
        fprintf(stderr,"Unable to set socket linger value to %d\n",
                lin.l_linger);
        return( -1 );
    }
 
    /* set the listening information */
    socket_addr.sin_family = AF_INET;
    socket_addr.sin_port = htons( port );
    socket_addr.sin_addr.s_addr = htonl( inet_addr("0.0.0.0") );
    if ( bind( socket_listen,
               (struct sockaddr *) &socket_addr,
               sizeof(struct sockaddr_in) ) < 0 )
    {                                                                           
        fprintf(stderr,"Unable to bind port %d\n",
                socket_addr.sin_port );
        return(-1);
    }
    if (listen(socket_listen,16)<0)
    {
        fprintf(stderr,"Unable to set socket listen buffer length\n");
        return(-1);
    }
 
    return socket_listen;                                                       
}

// Implementation for interface AudioManager

char* AudioManager_impl::url()
{
	char _url[1024];

	sprintf(_url,"tcp:localhost:%d",thePort);
  	return CORBA::string_dup(_url);
}

// the management functions:

void AudioManager_impl::unregisterUplink(AudioUplink_impl *uplink)
{
	list<AudioUplink_impl *>::iterator i;

	for(i=_uplinks.begin();i != _uplinks.end();i++)
	{
		AudioUplink_impl *u = *i;
		if(u->clientID() == uplink->clientID())
		{
			_uplinks.erase(i);
			_changes++;
			return;
		}
	}
	assert(false);
} 

unsigned long AudioManager_impl::changes()
{
	return _changes;
}

Arts::AudioClientInfoSeq* AudioManager_impl::clients()
{
	Arts::AudioClientInfoSeq* acs = new Arts::AudioClientInfoSeq();
	unsigned long l = 0;

	acs->length(_uplinks.size());
	list<AudioUplink_impl *>::iterator i;
	for(i=_uplinks.begin(); i != _uplinks.end();i++)
	{
		AudioUplink_impl *u = *i;
		(*acs)[l].ID = u->clientID();
		(*acs)[l].description = u->description();
		(*acs)[l].type = u->type();
		(*acs)[l].destination = u->bus();
		(*acs)[l].status = u->status();
		(*acs)[l].direction = u->direction();
		l++;
	}
	return acs;
}
 
// CORBA exported functions:
void AudioManager_impl::configureClient(long ID, const char *destination)
{
	list<AudioUplink_impl *>::iterator i;
	for(i=_uplinks.begin(); i != _uplinks.end();i++)
	{
		AudioUplink_impl *u = *i;

		if(u->clientID() == ID)
		{
			u->setBus(destination);
			u->setStatus(Arts::acRunning);
		}
	}
}

//------------------------------------------------------------------------------
// Implementation for interface AudioUplink

list<AudioUplink_impl *> AudioUplink_impl::instances;

AudioUplink_impl::AudioUplink_impl(int fd, Arts::Synthesizer_ptr Synthesizer,
					long clientID, AudioManager_impl *manager)
{
	instances.push_back(this);

	this->fd = fd;
	this->Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
	this->AudioManager = manager;

	gotBus = gotStart = gotClose = gotDirection = playing = false;

	_underruns = 0;
	_clientID = clientID;
	theClient = 0;
	_description = "unknown";
	_type = "unknown";
	_status = Arts::acInitializing;

	artsdebug("AudioUplink: sending welcome messages\n");

	char ior[1024];
	sprintf(ior,"%d aRts IOR connection starting,"
				" here is the IOR of the CORBA service\n",ARTS_MAN_INFO);
	write(fd,ior,strlen(ior));

	sprintf(ior,"%d %s\n",ARTS_MAN_CONNECT_IOR,ArtsOrb->object_to_string(this));
	write(fd,ior,strlen(ior));

	readIOEvent = new ArtsIOEvent(this,fd,ArtsIOEvent::ioRead,0);
	readIOEvent->start();

	writeIOEvent = new ArtsIOEvent(this,fd,ArtsIOEvent::ioWrite,0);
	writeRunning = false;

	AudioManager->incRef();
	Synthesizer->incRef();
}

AudioUplink_impl::~AudioUplink_impl()
{
	// we *must* stay here as long as the file descriptor is opened (the
	// client must be able to use this object as long as he is connected)
	assert(gotClose);

	AudioManager->unregisterUplink(this);
	AudioManager->decRef();
	Synthesizer->decRef();
	instances.remove(this);
}

long AudioUplink_impl::clientID()
{
	return _clientID;
}

AudioUplink_impl *AudioUplink_impl::find(long ID)
{
	list<AudioUplink_impl *>::iterator i;

	for(i=instances.begin(); i != instances.end(); i++)
		if((*i)->clientID() == ID) return *i;

	return 0;
}

#define BUFSIZE 16*1024

void AudioUplink_impl::notifyIO(int notifyfd, long type)
{
	char block[BUFSIZE];
	bool ioError = false;

	assert(notifyfd == fd);

	if(type == ArtsIOEvent::ioRead)
	{
		int r = read(fd,block,BUFSIZE);
		if(r > 0)
		{
			if(_direction == Arts::acPlay)
			{
				buffer.write(r,block);
			}
			else
			{
				// the client is stupid and has written something to the
				// record file descriptor, just ignore that
			}
		}
		else
		{
			if(r <= 0)		// TODO: figure out what are the exact conditions
			{				//   of (a) a real error or (b) connection closed
				ioError = true;
			}
		}
	}
	if(type == ArtsIOEvent::ioWrite && _direction == Arts::acRecord)
	{
		int r = BUFSIZE;		// smarter to take smaller value here?
		if(r > buffer.size()) r = buffer.size();

		if(r == 0)
		{
			artsdebug("* well ioWrite: nothing to write here, so leaving\n");
			writeIOEvent->stop();
			writeRunning = false;
			return;
		}

		int readbytes = buffer.read(r,block);
		assert(readbytes == r);	// we checked that before

		int w = write(fd,block,r);

		artsdebug("* ioWrite: got %d bytes from buffer, wrote %d bytes\n",readbytes,w);
		if(w == r) return;		// fine

		if(w <= 0)
		{
			ioError = true;
		}
		else
		{
			// well, then store the part we couldn't write for later
			buffer.unRead(r-w,&block[w]);
		}
	}

	if(ioError)
	{
		artsdebug("AudioUplink: read error, assuming remote end closed "
									"connection\n");
		
		readIOEvent->stop();
		delete readIOEvent;

		if(writeRunning)
			writeIOEvent->stop();
		delete writeIOEvent;

		// clean up
		close(fd);
		gotClose = true;

		artsdebug("AudioUplink: closed connection\n");

		// When we aren't playing, we can leave here, nobody will care
		if(!playing)
		{
			delete this;
			return;
		}
		// When we are, the inject module will ask us for more data
		// sooner or later (as soon as he has consumed everything) -
		// then we'll terminate ourselves and then remove this object
	}
}

// CORBA interface:

char* AudioUplink_impl::description()
{
	return CORBA::string_dup(_description.c_str()); 
}

void AudioUplink_impl::description( const char* _new_value )
{
	_description = _new_value;
}

char* AudioUplink_impl::type()
{
	return CORBA::string_dup(_type.c_str()); 
}

void AudioUplink_impl::type( const char* _new_value )
{
	_type = _new_value;
}

Arts::AudioClientDirection AudioUplink_impl::direction()
{
	return _direction;
}

void AudioUplink_impl::direction( Arts::AudioClientDirection _new_value )
{
	_direction = _new_value;
	gotDirection = true;
	play();
}

CORBA::Long AudioUplink_impl::bufferedSamples()
{
	// TODO: nicer calculation;
	return buffer.size()/4;
}

CORBA::Long AudioUplink_impl::underruns()
{
	return _underruns; 
}

void AudioUplink_impl::start()
{
	gotStart = true;
	play();
}

void AudioUplink_impl::play()
{
	if(playing || !gotBus || !gotStart || !gotDirection) return;
	playing = true;

	StructureBuilder sb(Synthesizer);

	char source[1024];
	sprintf(source,"%ld",clientID());

	if(_direction == Arts::acPlay)
	{
		sb.createModule("Synth_AMAN_INJECT","inject");
		sb.createModule("Synth_BUS_UPLINK","uplink");

		sb.setStringValue("inject.source",source);
		sb.connect("inject.left","uplink.left");
		sb.connect("inject.right","uplink.right");
		sb.connect("inject.busname","uplink.busname");
	}
	else if(_direction == Arts::acRecord)
	{
		sb.createModule("Synth_AMAN_CAPTURE","capture");
		sb.createModule("Synth_BUS_DOWNLINK","downlink");

		sb.setStringValue("capture.source",source);
		sb.setFloatValue("downlink.clients",1.0);

		sb.connect("downlink.left","capture.left");
		sb.connect("downlink.right","capture.right");
		sb.connect("capture.busname","downlink.busname");
	}
	else
	{
		assert(false);
	}

	Arts::ArtsServerSeq preferredServers;
	structureID = sb.execute(preferredServers);

	if(!sb.okay())
	{
		fprintf(stderr,"error in SBuilder: %s\n",sb.error());
	}

	assert(sb.okay());
}

// Interface to AudioManager

void AudioUplink_impl::setBus(const char *bus)
{
	_bus = bus;
	if(theClient) theClient->setBus(bus); // propagate to client

	gotBus = true;
	play();
}

const char *AudioUplink_impl::bus()
{
	return _bus.c_str();
}

void AudioUplink_impl::setStatus(Arts::AudioClientStatus newstatus)
{
	_status = newstatus;
}

Arts::AudioClientStatus AudioUplink_impl::status()
{
	return _status;
}

// Interface to AMAN_INJECT:

PipeBuffer *AudioUplink_impl::pipeBuffer()
{
	return &buffer;
}

void AudioUplink_impl::clientCreated(AudioClient *client)
{
	assert(!theClient);

	theClient = client;
	if(gotBus) theClient->setBus(_bus.c_str());

	assert(theClient);
}

void AudioUplink_impl::clientDestroyed(AudioClient *client)
{
	assert(theClient);
	assert(theClient == client);
	theClient = 0;

	assert(gotClose);
	delete this;
	return;
}
			 
void AudioUplink_impl::needMoreData()
{
	assert(theClient);
	assert(playing);

	if(gotClose)
	{
		// shutdown here
		theClient->terminate();

		// this will cause clientDestroyed to be called as soon as theClient
		// is gone...
	}
	else
	{
		_underruns++;
	}
}

void AudioUplink_impl::haveMoreData()
{
	assert(theClient);
	assert(playing);

	if(gotClose)
	{
		// shutdown here
		theClient->terminate();

		// this will cause clientDestroyed to be called as soon as theClient
		// is gone...
	}
	else
	{
		artsdebug(" * AudioUplink_impl::haveMoreData called\n");
		artsdebug(" * buffer.size() = %d writeRunning = %d\n",buffer.size(),writeRunning);
		if(buffer.size() > BUFSIZE && !writeRunning)
		{
			writeRunning = true;
			writeIOEvent->start();
		}
	}
}

//---------------------------- INJECT MODULE ---------------------------

class Synth_AMAN_INJECT :public SynthModule, AudioClient
{
	// inputs:
	enum { PROP_SOURCE };

	// out:
	enum { LEFT,RIGHT,PROP_BUSNAME };

	AudioUplink_impl *au;
	PipeBuffer *sourceBuffer;
	unsigned char *buffer;
	unsigned long blen;
	bool terminating;

	string _bus;

public:
	void setBus(const char *bus) {
		_bus = bus;
		setStringProperty(PROP_BUSNAME,bus);
	}
	void terminate();
	void Initialize();
	void DeInitialize();
	void Calculate() { }
	void CalculateBlock(unsigned long samples);
	string getParams() { return("_source;left,right,_busname"); }
	static void *Creator() { return new Synth_AMAN_INJECT; }
};

ModuleClient MC_Synth_AMAN_INJECT(SynthModule::get_MS,"Synth_AMAN_INJECT",Synth_AMAN_INJECT::Creator);

void Synth_AMAN_INJECT::Initialize()
{
	long bufferID = atol(getStringProperty(PROP_SOURCE));
	haveCalculateBlock = true;

	au = AudioUplink_impl::find(bufferID);
	assert(au);

	au->clientCreated(this);
	sourceBuffer = au->pipeBuffer();

	terminating = false;
	buffer = 0;
	blen = 0;

	artsdebug("audiomanager inject module started\n");
}

void Synth_AMAN_INJECT::DeInitialize()
{
	au->clientDestroyed(this);
}

void Synth_AMAN_INJECT::terminate()
{
	assert(!terminating);
	terminating = true;

	Synthesizer->kill(mID);
}

void Synth_AMAN_INJECT::CalculateBlock(unsigned long samples)
{
	float *left=out[LEFT], *right = out[RIGHT];
	unsigned long i;

	unsigned long samplesFromBuffer = 0;
	unsigned long samplesInBuffer = sourceBuffer->size()/4;

	if(samplesInBuffer != 0)
	{
		// try to satisfy request
		samplesFromBuffer = samples;

		// but go back to a smaller size if the buffer doesn't contain enough
		if(samplesInBuffer < samplesFromBuffer)
			samplesFromBuffer = samplesInBuffer;

		if(blen < samplesFromBuffer * 4)
		{
			buffer = (unsigned char *)realloc(buffer,samplesFromBuffer * 4);
			blen = samplesFromBuffer * 4;
		}

		unsigned long bytesRead;
		bytesRead = sourceBuffer->read(samplesFromBuffer * 4,buffer);
		
		// we did ensure this before by calling size()
		assert(bytesRead == samplesFromBuffer * 4);

		convert_stereo_i16le_2float(samplesFromBuffer,buffer,left,right);
		left += samplesFromBuffer;
		right += samplesFromBuffer;
	}

	unsigned long remaining = samples - samplesFromBuffer;

	if(remaining)
	{
		// we didn't satisfy the request completely, this can have two causes
		//
		//  - the stream ends here
		//  - buffer underrun
		//

		for(i=0;i<remaining;i++) *left++ = *right++ = 0.0; // fill with zeros

		if(!terminating) au->needMoreData();
	}
}

// --------------------------------------------------------------------------

class Synth_AMAN_CAPTURE :public SynthModule, AudioClient
{
	// inputs:
	enum { LEFT,RIGHT, PROP_SOURCE };

	// out:
	enum { PROP_BUSNAME };

	AudioUplink_impl *au;
	PipeBuffer *destBuffer;
	unsigned char *buffer;
	unsigned long blen;
	bool terminating;

	string _bus;

public:
	void setBus(const char *bus) {
		_bus = bus;
		setStringProperty(PROP_BUSNAME,bus);
	}
	void terminate();
	void Initialize();
	void DeInitialize();
	void Calculate() { }
	void CalculateBlock(unsigned long samples);
	string getParams() { return("left,right,_source;_busname"); }
	static void *Creator() { return new Synth_AMAN_CAPTURE; }
};

ModuleClient MC_Synth_AMAN_CAPTURE(SynthModule::get_MS,"Synth_AMAN_CAPTURE",Synth_AMAN_CAPTURE::Creator);

void Synth_AMAN_CAPTURE::Initialize()
{
	long bufferID = atol(getStringProperty(PROP_SOURCE));
	haveCalculateBlock = true;

	au = AudioUplink_impl::find(bufferID);
	assert(au);

	au->clientCreated(this);
	destBuffer = au->pipeBuffer();

	terminating = false;
	buffer = 0;
	blen = 0;

	artsdebug("audiomanager capture module started\n");
}

void Synth_AMAN_CAPTURE::DeInitialize()
{
	au->clientDestroyed(this);
}

void Synth_AMAN_CAPTURE::terminate()
{
	assert(!terminating);
	terminating = true;

	Synthesizer->kill(mID);
}

void Synth_AMAN_CAPTURE::CalculateBlock(unsigned long samples)
{
	// well, if we are terminating, we don't need to capture things anymore,
	// as they will be thrown away anyway...
	if(terminating) return;

	float *left=in[LEFT], *right = in[RIGHT];

	if(blen < samples * 4)
	{
		buffer = (unsigned char *)realloc(buffer,samples * 4);
		blen = samples * 4;
	}

	convert_stereo_2float_i16le(samples,left,right,buffer);
	destBuffer->write(samples * 4,buffer);
	if(buffer[0] != 0) printf("%d\n",buffer[0]);

	au->haveMoreData();
}
