    /*

    Copyright (C) 1998,1999 Stefan Westerfeld
                            stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "config.h"
#include "synthmodule.h"
#include "synth_impl.h"
#include "utils.h"
#include "cache.h"
#include "convert.h"
#include <sys/stat.h>
#include <math.h>

class CachedAkaiSample : public CachedObject
{
protected:
	struct stat oldstat;
	string filename;

	CachedAkaiSample(Cache *cache, string filename);
	~CachedAkaiSample();

public:
	long bufferedSamples;
	float *flBuffer;
	long samplingRate;

	int midiRoot;

	static CachedAkaiSample *load(Cache *cache, string filename);
	bool isValid();
	int memoryUsage();
};

CachedAkaiSample *CachedAkaiSample::load(Cache *cache, string filename)
{
	CachedAkaiSample *sample;

	sample =
		(CachedAkaiSample *)cache->get(string("CachedAkaiSample:")+filename);

	if(!sample)
		sample = new CachedAkaiSample(cache,filename);

	return(sample);
}

bool CachedAkaiSample::isValid()
{
	struct stat newstat;

	lstat(filename.c_str(),&newstat);
	return(newstat.st_mtime == oldstat.st_mtime);
}

int CachedAkaiSample::memoryUsage()
{
	return(bufferedSamples * sizeof(float));
}

CachedAkaiSample::CachedAkaiSample(Cache *cache, string filename) : CachedObject(cache)
{
	this->filename = filename;
	setKey(string("CachedAkaiSample:")+filename);

	lstat(filename.c_str(),&oldstat);

	FILE *infile = fopen(filename.c_str(),"r");

	if(!infile)
	{
		printf("AKAI-Sample: fatal: can't load file %s\n",filename.c_str());
		flBuffer = 0;
		bufferedSamples = 0;
		return;
	}
	char type = fgetc(infile);   // type
	if(type != 3)
	{
		printf("AKAI-Sample: warning: expected type = 3, got %d\n",type);
	}

	// tag, should contain 0 or 1 if the samplingRate is 22050 or 44100 Hz,
	// but there is a complete value for that later
	fgetc(infile);

	midiRoot = fgetc(infile);

	fseek(infile,0x1a,SEEK_SET);

	unsigned long nsw = 0; 	// number of sample words
	nsw = fgetc(infile);
	nsw += fgetc(infile)*256;
	nsw += fgetc(infile)*256*256;
	nsw += fgetc(infile)*256*256*256;

	fseek(infile,0x8a,SEEK_SET);
	
	samplingRate = fgetc(infile);
	samplingRate += 256*fgetc(infile);

	unsigned char *buffer = (unsigned char *) malloc(nsw*2);
	assert(buffer);

	// start of sample
	fseek(infile,0x94,SEEK_SET);

	printf("loading akaisample %s\n",filename.c_str());
	printf("  sampling rate: %ld\n",samplingRate);
	printf("  length: %ld\n",nsw*2);

	fread(buffer,1,nsw*2,infile);
	fclose(infile);

	bufferedSamples = nsw;

	/*
	 * Convert the data into floats here to save time while playing later.
	 */
	flBuffer = (float *)malloc(bufferedSamples*sizeof(float));
	convert_mono_16le_float(nsw,buffer,flBuffer);
	free(buffer);
	/*
	unsigned long position;

	for(position=0;position<nsw*2;position+=2)
	{
		long sample;

		sample = (((buffer[position + 1]+128)&0xff) << 8) + buffer[position];
		flBuffer[position/2] = (float)(sample-32767)/32768.0;
	}
	*/
}

CachedAkaiSample::~CachedAkaiSample()
{
	free(flBuffer);
}

class Synth_PLAY_AKAI :public SynthModule {
protected:
	CachedAkaiSample *cachedakaisample;

	float recfrequency;
	float flpos;
	float *flBuffer;
	long bufferedSamples, position;

	// inputs:
	enum { FREQUENCY,PROP_FILENAME };

	// outputs:
	enum { OUT, DONE };

public:
	void Initialize();
	void DeInitialize();
	void Calculate() {};
	void CalculateBlock(unsigned long samples);
	string getParams() { return("frequency,_filename;out,done"); }
	static void *Creator() { return new Synth_PLAY_AKAI; }
};

ModuleClient MC_Synth_PLAY_AKAI(SynthModule::get_MS,"Synth_PLAY_AKAI",Synth_PLAY_AKAI::Creator);

void Synth_PLAY_AKAI::CalculateBlock(unsigned long samples)
{
	float step = *in[FREQUENCY]/recfrequency;
	float *outvalue = out[OUT], *done=out[DONE];
	double int_pos;

	unsigned long i;

	// close to end of sample?
	if(position + (step * samples) + 10 < bufferedSamples)
	{
		// no, then no need to check again and again
		for(i=0;i<samples;i++)
		{
			position = (long)flpos;

			float error = modf(flpos,&int_pos);
		
			*outvalue++ = (1.0-error)*flBuffer[position]
			  		+      error*flBuffer[position+1];

			*done++ = 0.0;
			flpos += step;
		}
	}
	else
	{
		for(i=0;i<samples;i++)
		{
			position = (long)flpos;
	
			float error = modf(flpos,&int_pos);
		
			if(position + 1 < bufferedSamples)
				// need two samples to do interpolation
			{
				*outvalue++ = (1.0-error)*flBuffer[position]
			  			+      error*flBuffer[position+1];

				*done++ = 0.0;
			}
			else
			{
				*outvalue++ = 0.0;
				*done++ = 1.0;	// ready, kill me ;)
			}
			flpos += step;
		}
	}
}

void Synth_PLAY_AKAI::DeInitialize()
{
	cachedakaisample->decRef();
}

void Synth_PLAY_AKAI::Initialize()
{
	cachedakaisample = CachedAkaiSample::load(Synthesizer->getCache(),
									getStringProperty(PROP_FILENAME));

	// may take some speed to access cachedakaisample every time
	bufferedSamples = cachedakaisample->bufferedSamples;	
	flBuffer = cachedakaisample->flBuffer;

	recfrequency = 261.63 * pow(2,(float)cachedakaisample->midiRoot/12)/32.0;

	recfrequency *= (float)samplingRate/(float)cachedakaisample->samplingRate;

	flpos = 0;
	haveCalculateBlock = true;
}

class Synth_PLAY_AKAIS :public SynthModule {
protected:
	CachedAkaiSample *leftsample, *rightsample;

	float recfrequency;
	float flpos;
	float *leftBuffer;
	float *rightBuffer;
	long bufferedSamples, position;

	// inputs:
	enum { FREQUENCY, PROP_LEFTFILE, PROP_RIGHTFILE };

	// outputs:
	enum { LEFT, RIGHT, DONE };

public:
	void Initialize();
	void DeInitialize();
	void Calculate() {};
	void CalculateBlock(unsigned long samples);
	string getParams() { return("frequency,_leftfile,_rightfile;left,right,done"); }
	static void *Creator() { return new Synth_PLAY_AKAIS; }
};

ModuleClient MC_Synth_PLAY_AKAIS(SynthModule::get_MS,"Synth_PLAY_AKAIS",Synth_PLAY_AKAIS::Creator);

void Synth_PLAY_AKAIS::CalculateBlock(unsigned long samples)
{
	float step = *in[FREQUENCY]/recfrequency;
	float *outleft = out[LEFT], *outright = out[RIGHT], *done=out[DONE];
	double int_pos;

	unsigned long i;

	// close to end of sample?
	if(position + (step * samples) + 10 < bufferedSamples)
	{
		// no, then no need to check again and again
		for(i=0;i<samples;i++)
		{
			position = (long)flpos;

			float error = modf(flpos,&int_pos);
		
			*outleft++ = ((1.0-error)*leftBuffer[position]
			  		+      error*leftBuffer[position+1]);

			*outright++ = ((1.0-error)*rightBuffer[position]
			  		 +      error*rightBuffer[position+1]);

			*done++ = 0.0;

			flpos += step;
		}
	}
	else
	{
		for(i=0;i<samples;i++)
		{
			position = (long)flpos;
	
			float error = modf(flpos,&int_pos);
		
			if(position + 1 < bufferedSamples)
				// need two samples to do interpolation
			{
				*outleft++ = (1.0-error)*leftBuffer[position]
			  			+      error*leftBuffer[position+1];

				*outright++ = (1.0-error)*rightBuffer[position]
			  		 	+      error*rightBuffer[position+1];

				*done++ = 0.0;
			}
			else
			{
				*outleft++ = 0.0;
				*outright++ = 0.0;
				*done++ = 1.0;	// ready, kill me ;)
			}
			flpos += step;
		}
	}
}

void Synth_PLAY_AKAIS::DeInitialize()
{
	leftsample->decRef();
	rightsample->decRef();
}

void Synth_PLAY_AKAIS::Initialize()
{
	leftsample = CachedAkaiSample::load(Synthesizer->getCache(),
									getStringProperty(PROP_LEFTFILE));
	rightsample = CachedAkaiSample::load(Synthesizer->getCache(),
									getStringProperty(PROP_RIGHTFILE));

	bufferedSamples = leftsample->bufferedSamples;
	if(bufferedSamples > rightsample->bufferedSamples)
		bufferedSamples = rightsample->bufferedSamples;

	leftBuffer = leftsample->flBuffer;
	rightBuffer = rightsample->flBuffer;

	// they have to be recorded at the same rate
	recfrequency = 261.63 * pow(2,(float)leftsample->midiRoot/12)/32.0;
	recfrequency *= (float)samplingRate/(float)leftsample->samplingRate;

	flpos = 0;
	haveCalculateBlock = true;
}
