    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "arts.h"
#include <unistd.h>
#include <list>

void usage(char *prog)
{
	fprintf(stderr,"\n");
	fprintf(stderr,"Usage: %s [ -f <filename> ] [ -p ] [ -e ] [ -t ] [ -s ]"
                                                             " [ -i ]\n",prog);
	fprintf(stderr,"       [ -C <cachesize> ] [ -R <ringbuffersize> ]"
                         " [ -S <scheduling cycle size> ]");
    fprintf(stderr,"       [ -D <debug> ]\n\n");
	fprintf(stderr,"   -f   the .arts-file to load\n");
	fprintf(stderr,"   -p   publish the specified file on the server\n");
	fprintf(stderr,"   -e   execute the specified file on the server\n");
	fprintf(stderr,"   -t   free synthserver after usage when possible\n");
	fprintf(stderr,"   -T   force the termination of the synthserver\n");
	fprintf(stderr,"        (use this only as last resort, its a bad hack)\n");
	fprintf(stderr,"   -i   retrieve status information of the server\n");
	fprintf(stderr,"   -m   list all available modules\n");
	fprintf(stderr,"   -s   show (realtime) settings\n");
	fprintf(stderr,"   -c   show connected clients\n");
	fprintf(stderr,"   -d <clientID>:<destination>  set client destination\n");

	fprintf(stderr,"   -R   set ringbuffer size im samples, the argument MUST be a 2^n value\n");
	fprintf(stderr,"   -S   set scheduling cycle size in samples\n");
	fprintf(stderr,"   -F <fragments>:<size>  set the OSS fragment settings\n");
	fprintf(stderr,"        fragments should be some small integer like 7\n");
	fprintf(stderr,"        and size some 2^n value (e.g. 256)\n");
	fprintf(stderr,"        the overall driver buffersize (=latency) in bytes will be the product\n");
	fprintf(stderr,"   -C   set cache size in KB\n");
	fprintf(stderr,"   -D   set debugging (1 to enable, 0 to disableable)\n");
	exit(1);
}

#ifdef COMMON_BINARY
int artsshell_main(int argc, char **argv)
#else
int main(int argc, char **argv)
#endif
{
	CORBA::ORB_var orb = CORBA::ORB_init( argc, argv, "mico-local-orb" );
	CORBA::BOA_var boa = orb->BOA_init( argc, argv, "mico-local-boa" );

	CORBA::Object_var obj = orb->bind ("IDL:Arts/Synthesizer:1.0",
												"inet:localhost:8888");
	Arts::Synthesizer_var Synthesizer = Arts::Synthesizer::_narrow (obj);

	if(CORBA::is_nil(Synthesizer))
	{
		fprintf(stderr,"%s will only work when the Arts Server is running\n",
					argv[0]);
		exit(1);
	}
	char filename[255];
	bool publish = false, execute = false,fileok = false,terminate = false;
	bool showrt = false, changert = false, info = false, showclients = false;
	bool forceTermination = false, listModules = false;
	int optch;
	Arts::RealtimeConfig rtc = Synthesizer->RTConfig();

	list<char *> clientSetup;

	while((optch = getopt(argc,argv,"mcd:tTf:piesS:C:R:D:F:")) > 0)
	{
		switch(optch)
		{
			case 'f': strncpy(filename,optarg,255);
					  fileok = true;
				break;
			case 'p': publish = true;
				break;
			case 'e': execute = true;
				break;
			case 't': terminate = true;
				break;
			case 'T': forceTermination = true;
				break;
			case 'i': info = true;
				break;
			case 's': showrt = true;
				break;
			case 'c': showclients = true;
				break;
			case 'd': clientSetup.push_back(strdup(optarg));
					  showclients = true;
				break;
			case 'm': listModules = true;
				break;
			case 'C': rtc.CacheSizeKB = atol(optarg);
					  changert = true;
				break;
			case 'R': rtc.RingBufferSize = atol(optarg);
					  changert = true;
				break;
			case 'S': rtc.Cycles = atol(optarg);
					  changert = true;
				break;
			case 'F': {
						char *d = strdup(optarg), *fcount = 0, *fsize = 0;
						fcount = strtok(d,":");
						if(fcount) fsize = strtok(0,":\n");
						if(fcount && fsize)
						{
							rtc.FragmentCount = atol(fcount);
							rtc.FragmentSize = atol(fsize);
					  		changert = true;
						}
						free(d);
					  }
				break;
			case 'D': rtc.Debug = atol(optarg);
					  changert = true;
				break;
			default: usage(argv[0]);
				break;
		}
	}

	if(forceTermination)
	{
		Synthesizer->forceTermination();
	}

	if(changert) showrt = true;

	if(!fileok && !showrt && !info && !showclients && !listModules)
		usage(argv[0]);

	if(listModules)
	{
		Arts::ModuleBroker_var ModuleBroker = Synthesizer->moduleBroker();
		assert(ModuleBroker);

		Arts::StringSeq_var Modules = ModuleBroker->publishedModules();
		
		unsigned long l;
		for(l=0;l<Modules->length();l++)
		{
			printf("%s\n",(const char *)(*Modules)[l]);
		}
	}

	if(changert)
	{
		Synthesizer->RTConfig(rtc);
		printf("Realtime settings updated.\n");
		rtc = Synthesizer->RTConfig();
	}

	if(showclients)
	{
		Arts::AudioManager_var am = Synthesizer->audioManager();

		while(clientSetup.size())
		{
			char *s = *clientSetup.begin();
			clientSetup.pop_front();
	
			char *idstr = strtok(s,":");
			if(idstr)
			{
				char *dest = strtok(NULL,":\n");
				long id = atol(idstr);

				am->configureClient(id,dest);
			}
		}

/*
 * sample output:
 *
 * 1    laber.mp3                     mp3       audio01        running *
 */
 			printf(
  "ID   Description/Title             Type      Destination    Status\n"
  "-----------------------------------------------------------------------\n");

		Arts::AudioClientInfoSeq_var acs = am->clients();

		unsigned long c;
		for(c=0;c<acs->length();c++)
		{
			CORBA::String_var description = (*acs)[c].description;
			CORBA::String_var type = (*acs)[c].type;
			CORBA::String_var destination = (*acs)[c].destination;
			char *status;

			switch((*acs)[c].status)
			{
				case Arts::acInitializing: status = "init";
									break;
				case Arts::acRunning: status = "running";
									break;
				default: status = "<unknown>";
			}

			printf("%-5ld%-30s%-10s%-15s%-10s\n",(*acs)[c].ID,
				(const char *)description,(const char *)type,
				(const char *)destination, status);
		}
	}

	if(showrt)
	{
		char debugstatus[10] = "disabled";
		if(rtc.Debug) strcpy(debugstatus,"enabled");
		printf("\n");
		printf("    Scheduling cycle size: %ld samples (delay %2.2f ms)\n",rtc.Cycles,(float)rtc.Cycles/44.1);
		printf("                Fragments: %ld fragments with %ld bytes"
	             " (delay %2.2f ms)\n",rtc.FragmentCount,rtc.FragmentSize,
				 (float)(rtc.FragmentCount*rtc.FragmentSize*1000)/44100.0/4.0);
		printf("         Ring buffer size: %ld samples\n",rtc.RingBufferSize);
		printf("Cache size (for WAVs etc): %ld KB\n",rtc.CacheSizeKB);
		printf("    Debugging information: %s\n",debugstatus);
		printf("\n");
		printf("To get useful results, the delay of the scheduling cycle should be smaller\n");
		printf("than the delay of the fragments. Rule: The delay of one or two fragments\n");
		printf("should probably correspond the delay of the scheduling cycle.\n");
		printf("All delay values calucated above are for 44kHz, stereo, 16bit.\n");
		printf("\n");
	}

	if(info)
	{
		Arts::Status status = Synthesizer->getStatus();
		printf("\n");
		printf("   Active modules: %ld\n",status.module_count);
		printf("        CPU usage: %2.2f%%\n",status.cpu_usage*100);
		printf("\n");
		Arts::ResourceInfoSeq *resources = Synthesizer->Resources();
		unsigned long i;
		for(i=0;i<resources->length();i++)
		{
			Arts::ResourceInfo &ri = (*resources)[i];
			printf("%10ld %s\n",ri.Used,(const char *)ri.Description);
		}
		if(i>0) printf("\n");
	}

	if(!fileok) exit(0);

    FILE *infile = fopen(filename,"r");
	if(!infile)
	{
		fprintf(stderr,"%s: can't open file '%s'!\n",argv[0],filename);
		exit(EXIT_FAILURE);
	}

    Arts::StringSeq_var strseq = new Arts::StringSeq;
 
    char line[1024];
    unsigned long i = 0;
 
    while(fgets(line,1024,infile))
    {
        // cut eventual CR and/or LFs at the end of the line
        while(strlen(line) && line[strlen(line)-1] < 14)
            line[strlen(line)-1] = 0;
 
        strseq->length(i+1);
        (*strseq)[i++] = CORBA::string_dup(line);
    }
    fclose(infile);

	// this is ugly, since the synthesizer may terminate if not referenced
	// so this option should probably make room for a more well behaved one?

	if(terminate)
		Synthesizer->incRef();

	Arts::StructureDesc_var StructureDesc = Synthesizer->createStructureDesc(); 
	StructureDesc->incRef();
    StructureDesc->loadFromList(strseq);

	if(!StructureDesc->valid())
	{
		fprintf(stderr,
"The structure couldn't be loaded correctly. Usually this means that some\n"
"of the modules the structure uses aren't available in this version of aRts.\n"
);
		execute = publish = false;
	}

	if(publish)
		Synthesizer->publishStructureDesc(StructureDesc);

	if(execute)
	{
		Arts::ArtsServerSeq preferredservers;	// oh well, don't care where
												// our stuff gets executed
	    // just in case synthesis has been halted before,
   		// restart it and hope we'll have enough computing power now
    	Synthesizer->Reset();
    	long execID = Synthesizer->createStructure(StructureDesc,
				preferredservers);
    	assert(execID);
		printf(" <press return to end execution> \n\n"); 
		getchar();
    	Synthesizer->freeStructure(execID);
	}
	StructureDesc->decRef();

	if(terminate)
		Synthesizer->decRef();
}
