	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "main.h"
#include "arts.h"
#include "guimodule.h"
#include "kcorbaApp.h"
#include "structureport.h"
#include "artsorb.h"
#include "menumaker.h"
#include "session.h"
#include "dirmanager.h"

#include <list>

#include <kmsgbox.h>
#include <kapp.h>
#include <kstdaccel.h>
#include <ktopwidget.h>
#include <kfiledialog.h>
#include <ksimpleconfig.h>
#include <qlistbox.h>
#include <qscrollview.h>
#include <qtablevw.h>
#include <qpainter.h>
#include <qdrawutl.h>
#include <qwidget.h>
#include <qlayout.h>
#include <qframe.h>
#include <qpixmap.h>
#include <qmsgbox.h>
#include <string.h>
#include <unistd.h>
#include "module.h"
#include "autorouter.h"
#include "portpropdlg.h"
#include "portposdlg.h"
#include "execdlg.h"
#include "namedlg.h"
#include "utils.h"

/*************************************/
class ArtsBuilderApp :public KCorbaApplication
{
protected:
	ModuleView *moduleview;
public:
	ArtsBuilderApp( int &argc, char** argv );
	void start();
	void end();
} *theArtsBuilderApp;

Arts::Synthesizer_ptr theSynthesizer;

/*************************************/

Structure *ModuleWidget::theStructure()
{
	return structure;
}

void ModuleWidget::addModule ( Arts::ModuleInfo *minfo )
{
	if(activeTool) delete activeTool;

	activeTool = new CreateModuleTool(this,minfo);
}

void ModuleWidget::addPort ( Arts::PortType type )
{
	if(activeTool) delete activeTool;

	activeTool = new CreatePortTool(this, type);
}

void ModuleWidget::insertPort( Arts::PortType type, int x, int y )
{
	StructurePort *port = structure->createStructurePort(type);
	unselectAll();
	port->move(x,y);
	port->setSelected(true);
	port->show();
}

void ModuleWidget::leaveTool(CreateTool *tool)
{
	assert(tool == activeTool);
	delete tool;
	activeTool = 0;
}

ModulePort *ModuleWidget::findPort(Module *m, int xoffset, int direction)
{
	return m->findPort(xoffset,direction);
}

bool ModuleWidget::insertModule( Module *newModule )
{
	if(hasSpace(newModule,newModule->x(),newModule->y(),true))
	{
		newModule->show();
		reRoute();
		return true;
	}
	return false;
}

void ModuleWidget::mousePressEvent( QMouseEvent *e )
{
	if(activeTool)
	{
		activeTool->mousePressEvent(e);
		return;
	}
	if ( e->button() == LeftButton )
	{
		movestart = e->pos();
		mouseButtonDown = true;

		int x = findCol(e->x());
		int y = findRow(e->y());

		int cellx = 0, celly=0;

		colXPos(x,&cellx);
		rowYPos(y,&celly);

		cellx = e->x() - cellx;
		celly = e->y() - celly;

		//printf("clicked at %d,%d\n",cellx,celly);

		StructureComponent *c = structure->componentAt(x,y,false);
		ModulePort *port = NULL;

		if(c && (port = c->portAt(x-(c->x()),cellx,celly)))
		{
			if(port->selected)
			{
				PortPropDlg *ppd = new PortPropDlg(0,port);
				ppd->exec();
				port->selected = false;
				selectedports--;
			}
			else
			{
				port->selected = true;
				selectedports++;
			}
			if(selectedports == 2)
			{
				if(lastport->direction == port->direction)
				{
					port->selected = false;
					selectedports--;
   					QMessageBox::message( i18n("Arts Warning"),
					i18n("You can only connect an IN-port with an OUT-port, \n"
							"not two ports with the same direction") );

				}
				else
				{
					if(lastport->direction == ModulePort::in)
						connectPorts(port,lastport);
					else
						connectPorts(lastport,port);
				}
			}
			else
			{
				lastport = port;
			}
/*
				redrawModule(m);
*/
			reRoute();
			return;
		}

		beginUpdate();
		// select operations:
	
		// maintain selected group when
		// - clicking into a module that was already selected
		// - pressing the shift button
	
		bool group = (c && (c->selected() || (e->state() & ShiftButton)));
	
		// unselect all before moving, when not grouped
		if(!group) unselectAll();
	
		if(c) c->setSelected(true);

		endUpdate();
	}
}

void ModuleWidget::mouseMoveEvent( QMouseEvent *e )
{
	if(activeTool)
	{
		activeTool->mouseMoveEvent(e);
		return;
	}
	if(!mouseButtonDown) return;

	int dx = (e->x() - movestart.x())/cellsize;
	int dy = (e->y() - movestart.y())/cellsize;

	if(dx == 0 && dy == 0) return;

	list<StructureComponent *>::iterator i;
	list<StructureComponent *> *ComponentList = structure->getComponentList();

	for(i = ComponentList->begin();i != ComponentList->end();i++)
	{
		StructureComponent *c = *i;
		if(c->selected() && !hasSpace(c,c->x()+dx,c->y()+dy,true)) return;
	}

	beginUpdate();
	for(i = ComponentList->begin();i != ComponentList->end();i++)
	{
		StructureComponent *c = *i;
		if(c->selected())
			c->move(c->x()+dx,c->y()+dy);
	}
	endUpdate();
	
	movestart.setX(movestart.x()+dx*cellsize);
	movestart.setY(movestart.y()+dy*cellsize);
	reRoute();
}

bool ModuleWidget::hasSpace(StructureComponent *c,int destx,int desty,
											bool ignore_selected)
{
	int ddx,ddy;

	if(destx < 0) return(false);
	if(desty < 0) return(false);
	if(destx+c->width() > numCols()) return(false);
	if(desty+c->height() > numRows()) return(false);

	for (ddx = 0; ddx < c->width(); ddx++)
	{
		for (ddy = 0; ddy < c->height(); ddy++)
		{
			if(structure->componentAt(destx+ddx,desty+ddy,ignore_selected))
				return(false);
		}
	}

	return(true);
}

void ModuleWidget::mouseReleaseEvent( QMouseEvent *e )
{
	if(activeTool)
	{
		activeTool->mouseReleaseEvent(e);
		return;
	}
	mouseButtonDown = false;
}

void ModuleWidget::paintCellBackground(QPainter *p, int y, int x)
{
	QColor bgcolor;

	if((y & 1) == 1)
		bgcolor = QColor(168,168,168);
	else
		bgcolor = QColor(146,168,146);

	p->fillRect(0,0,cellsize,cellsize,QBrush(bgcolor));

	p->setPen(bgcolor.dark(115));
	p->drawLine(0,0,0,cellsize-1);
	p->drawLine(0,0,cellsize-1,0);

	if(x == (numCols()-1))
		p->drawLine(cellsize-1,0,cellsize-1,cellsize-1);
	if(y == (numRows()-1))
		p->drawLine(0,cellsize-1,cellsize-1,cellsize-1);
}

void ModuleWidget::unselectAll()
{
	setSelectAll(false);
}

void ModuleWidget::setSelectAll(bool newstate)
{
	list<StructureComponent *>::iterator module;
	
	for(module = structure->getComponentList()->begin();
        module != structure->getComponentList()->end();module++)
	{
		(*module)->setSelected(newstate);
	}
}

void ModuleWidget::beginUpdate()
{
	inUpdate = true;
}

void ModuleWidget::endUpdate()
{
	inUpdate = false;

	list<QRect>::iterator i;

	for (i = UpdateList.begin(); i != UpdateList.end(); i++)
	{
		redrawCells(*i);
	}

	UpdateList.erase(UpdateList.begin(),UpdateList.end());
}

void ModuleWidget::redrawModule(Module *m)
{
	QRect r = QRect(m->x(),m->y(),m->width(),m->height());

	if(!inUpdate)
	{
		redrawCells(r);
	}
	else
	{
		UpdateList.push_back(r);
	}
}

void ModuleWidget::redrawRect(int x, int y, int width, int height)
{
	QRect r = QRect(x,y,width,height);

	if(!inUpdate)
	{
		redrawCells(r);
	}
	else
	{
		UpdateList.push_back(r);
	}
}

void ModuleWidget::redrawCells(QRect &r)
{
	int x,y;

	for(x = r.left(); x<r.width()+r.left(); x++)
	{
		for(y = r.top(); y<r.height()+r.top(); y++)
		{
			updateCell(y,x,false);
		}
	}
}


void ModuleWidget::reRoute()
{
// clear everything
	autorouter->clear();

// add structure components (external ports/modules) to the router, so that
// cables won't be drawn over them

	list<StructureComponent *>::iterator c;
	list<ModulePort *> portlist;

	for(c = structure->getComponentList()->begin();
		c != structure->getComponentList()->end();c++)
	{
		StructureComponent *sc = *c;
		autorouter->set(sc->x()*2,sc->y()*2,
						(sc->x()+sc->width())*2-1,
						(sc->y()+sc->height())*2-1, AutoRouter::solid);

		sc->dumpPorts(portlist);
	}

	list<ModulePort *>::iterator pi;

// build a map with all input ports to find corresponding ports of connections

	map<long, ModulePort *> portmap;

	for(pi = portlist.begin();pi != portlist.end(); pi++)
	{
		ModulePort *port = *pi;

		if(port->direction == ModulePort::in) portmap[port->pdID] = port;
	}

// add connections to the router

	for(pi = portlist.begin();pi != portlist.end();pi++)
	{
		ModulePort *p = *pi;

		if(p->direction == ModulePort::out && p->PortDesc->isConnected())
		{
			ModulePort *src = p, *dest;
			long route_owner = 0;
			unsigned long c;

			Arts::PortDescSeq_var conn = p->PortDesc->Connections();
	
			for(c=0;c<conn->length();c++)
			{
				dest = portmap[(*conn)[c]->ID()];
				if(dest)		// otherwise something bad has happend?
				{
/*
						printf("autoroute add connection port %s.%s to %s.%s\n",
							src->owner->type.data(),src->description.data(),
							dest->owner->type.data(),dest->description.data());
*/

					int x1 = (src->owner->x()+src->drawsegment)*2;
					int y1 = src->owner->y()*2+1;

					int x2 = (dest->owner->x()+dest->drawsegment)*2;
					int y2 = dest->owner->y()*2;

					route_owner = autorouter->connect(x1,y1,x2,y2,route_owner);
				}
			}
		}
	}

	autorouter->sync();
}

void ModuleWidget::redrawAll()
{
// redraw everything
	QRect updaterect(0,0,cols,rows);
	redrawCells(updaterect);
}

void ModuleWidget::paintConnection(QPainter *p, int x, int y, int arx, int ary)
{
	long linetype = autorouter->get(arx,ary);
	long ud_owner = -1, lr_owner = -1, lr_break = 0, ud_break = 0;

	autorouter->getowners(arx,ary,ud_owner,lr_owner);

	p->setPen(QColor(255,255,255));

/*
	if(linetype == AutoRouter::none)
	{
		p->drawPoint(x+cellsize/4,y+cellsize/4);
	}
	if(linetype & AutoRouter::solid)
	{
		QBrush whitefill(QColor(255,255,255));

		p->fillRect(x+cellsize/6,y+cellsize/6,cellsize/6,cellsize/6,whitefill);
	}
*/
	x += cellsize/4;
	y += cellsize/4;

	// both used?
	if(ud_owner != -1 && lr_owner != -1)
	{
		// and not of the same owner?
		if(ud_owner != lr_owner)
		{
			// then we'll have to paint one of them broken
			if(ud_owner > lr_owner)
				lr_break = cellsize/8;
			else
				ud_break = cellsize/8;
		}
	}

	if(linetype & AutoRouter::left)
		p->drawLine(x-cellsize/4,y,x-lr_break,y);
	if(linetype & AutoRouter::right)
		p->drawLine(x+cellsize/4,y,x+lr_break,y);
	if(linetype & AutoRouter::up)
		p->drawLine(x,y-cellsize/4,x,y-ud_break);
	if(linetype & AutoRouter::down)
		p->drawLine(x,y+cellsize/4,x,y+ud_break);
}

void ModuleWidget::paintConnections(QPainter *p, int y, int x)
{
	int dx,dy;
	for(dx = 0; dx < 2; dx++)
	{
		for(dy = 0; dy < 2; dy++)
		{
			paintConnection(p,(cellsize*dx)/2,(cellsize*dy)/2,x*2+dx,y*2+dy);
		}
	}
}

void ModuleWidget::paintCell(QPainter *p, int y, int x)
{
	if(theArtsBuilderApp->eventStackDepth() > 1)
	{
		// FIXME: set some redraw flag or something like that
		return;
	}

	list<StructureComponent *>::iterator c;
	for(c = structure->getComponentList()->begin();
		c != structure->getComponentList()->end();c++)
	{
		StructureComponent *mwc = *c;
		if(y == mwc->y() && mwc->visible())
		{
			int xoffset = x-mwc->x();

			if(xoffset >= 0 && xoffset < mwc->width())
			{
				if(mwc->drawNeedsBackground(xoffset))
					 paintCellBackground(p,y,x);

				mwc->drawSegment(p,cellsize,xoffset);
				paintConnections(p,y,x);
				return;
			}
		}
	}

	paintCellBackground(p,y,x);
	paintConnections(p,y,x);
}

bool ModuleWidget::connectPorts(ModulePort *src, ModulePort *dest)
{
	bool success = src->PortDesc->connectTo(dest->PortDesc);

	selectedports = 0;
	src->selected = false;
	dest->selected = false;

/** TODO
	if(src->owner) redrawModule(src->owner);
	if(dest->owner) redrawModule(dest->owner);
*/
	return success;
}

// ---------------------------------------------------------------------------
// public part of modulewidget
// ---------------------------------------------------------------------------

void ModuleWidget::setZoom(int zoom)
{
	cellsize = (int)(50.0 * (float)zoom/100);

	setCellHeight(cellsize);
	setCellWidth(cellsize);
	updateTableSize();
	resize(cellsize*cols,cellsize*rows);
}

void ModuleWidget::selectAll()
{
	setSelectAll(true);
}

void ModuleWidget::reInit()
{
	selectedports = 0;

	reRoute();
	redrawAll();
}

void ModuleWidget::delModule()
{
	if(!structure->countSelected()) return;

	if (QMessageBox::warning(0,kapp->getCaption(), i18n("Delete selected module(s), port(s) and connections (no undo possible)?"), QMessageBox::Ok,
			QMessageBox::Cancel) != QMessageBox::Cancel)
	{
		structure->deleteSelected();
		selectedports = 0;
		reRoute();
		redrawAll();
	}
}

void ModuleWidget::autoRedrawRouter()
{
	if(autorouter->needRedraw()) redrawAll();
}

ModuleWidget::ModuleWidget(Structure *structure, QWidget *parent=0, const char *name=0, WFlags f=0)
		:QTableView(parent,name,f)
{
	this->ModuleList = structure->getModuleList();
	this->structure = structure;
	activeTool = 0;
	mouseButtonDown = false;

	cols = 24;
	rows = 32;

	setNumCols(cols);
	setNumRows(rows);
	setTableFlags(Tbl_autoScrollBars);
	setZoom(100);

	setBackgroundMode(NoBackground);

	selectedports = 0;
	autorouter = new AutoRouter(cols*2,rows*2);

    QTimer *timer = new QTimer( this );
	connect( timer, SIGNAL(timeout()),
		this, SLOT(autoRedrawRouter()) );
	timer->start( 100, FALSE );                 // 100 ms reoccuring check
};

ModuleWidget::~ModuleWidget()
{
	delete autorouter;
}

ModuleView::ModuleView(const char *name, Arts::Synthesizer *Synthesizer) : KTopLevelWidget(name)
{
	KStdAccel keys(kapp->getConfig());
 
	this->Synthesizer = Synthesizer;

	structure = new Structure(Synthesizer);
	//ModuleList = structure->getModuleList();

	modulewidget = new ModuleWidget(structure, this, "mwidget");

	structure->setCanvas(modulewidget);

	m_file = new QPopupMenu;
	m_file_new = new QPopupMenu;
	m_edit = new QPopupMenu;
	m_view = new QPopupMenu;
	m_modules = new QPopupMenu;
	m_ports = new QPopupMenu;

/*
	m_modules_gui = new QPopupMenu;
	m_modules_synth = new QPopupMenu;
	m_modules_instruments = new QPopupMenu;
	m_modules_other = new QPopupMenu;
	*/

	m_file->insertItem(i18n("Open Session..."), this , SLOT(openSession()));
    m_file->insertSeparator();
	m_file->insertItem(i18n("&New"), m_file_new);
	m_file->insertItem(i18n("&Open..."), this , SLOT(open()), keys.open());
	m_file->insertItem(i18n("&Save..."), this , SLOT(save()), keys.save());
	m_file->insertItem(i18n("&Retrieve from server..."), this, SLOT(retrieve()));
    m_file->insertSeparator();
	m_file->insertItem(i18n("&Execute Structure..."), this, SLOT(execute()));
	m_file->insertItem(i18n("&Rename Structure..."), this, SLOT(rename()));
	m_file->insertItem(i18n("&Publish Structure!"), this, SLOT(publish()),
														CTRL+Key_P);
    m_file->insertSeparator();
	m_file->insertItem(i18n("&Quit"), this , SLOT(hide()), keys.quit());

	m_edit->insertItem(i18n("&Delete"), modulewidget, SLOT(delModule()), Key_Delete);
    m_edit->insertSeparator();
    m_edit->insertItem(i18n("Select &all"), modulewidget, SLOT(selectAll()),
														CTRL+Key_A);
 

	m_view->insertItem(i18n("200%"), 200);
	m_view->insertItem(i18n("150%"), 150);
	m_view->insertItem(i18n("100%"), 100);
	m_view->insertItem(i18n("50%"), 50);

	m_ports->insertItem(i18n("Create IN audio signal"), 1);
	m_ports->insertItem(i18n("Create OUT audio signal"), 2);
	m_ports->insertItem(i18n("Create IN string property"), 3);
	m_ports->insertItem(i18n("Create IN audio property"), 4);
    m_ports->insertSeparator();
	m_ports->insertItem(i18n("Change positions/names"), this, SLOT(changePortPositions())); //,0,-1);

	ModuleBroker = Synthesizer->moduleBroker();
	assert(ModuleBroker);

//---- publish my widgets on the server ----

	GUIServer=new GUIServer_impl(ModuleBroker,Synthesizer);
    printf("GUIServer:\n%s\n",ArtsOrb->object_to_string(GUIServer));
	GUIServer->incRef();

//---- trigger autoloading of all structures that are present in my dirs ----
	
	string common = (const char *)KApplication::kde_datadir();
	common += "/artsbuilder";
	if(chdir(common.c_str()) == 0)
		ModuleBroker->addPublishingPath(common.c_str());

	// just make sure that the mapsDir exists
	(void)DirManager::mapDir();

	if(chdir(DirManager::structureDir()) == 0)	// retry
	{
		ModuleBroker->addPublishingPath(DirManager::structureDir());
		Synthesizer->addArtsDirectory(DirManager::baseDir());

	}

//----

	mbroker_updateCount = 0;
	execDlg = 0;

    QString about = i18n("Arts Builder ");
	about += VERSION;
    about += i18n("\n\n(C) 1998-1999 Stefan Westerfeld <stefan@twc.de>\n\n"
	       "Arts synthesizer designer.\n\nArts Homepage: http://linux.twc.de/arts");
    QPopupMenu *m_help = kapp->getHelpMenu(true, about.data());
		       
	menubar = new KMenuBar (this, "menubar");
	menubar->insertItem (i18n("&File"), m_file);
	menubar->insertItem (i18n("&Edit"), m_edit);
	menubar->insertItem (i18n("&View"), m_view);
	modules_menu_item = menubar->insertItem (i18n("&Modules"), m_modules);
	menubar->insertItem (i18n("&Ports"), m_ports);
    menubar->insertSeparator();
    menubar->insertItem( i18n("&Help"), m_help);

	setMenu (menubar);      

	setView(modulewidget);

	menumaker = new MenuMaker(m_modules);
	menumaker->addCategory("&Gui","Gui_");
	menumaker->addCategory("&Synthesis","Synth_");
	menumaker->addCategory("&Synthesis/&Arithmetic + Mixing","Synth_ADD$");
	menumaker->addCategory("&Synthesis/&Arithmetic + Mixing","Synth_AUTOPANNER$");
	menumaker->addCategory("&Synthesis/&Arithmetic + Mixing","Synth_MUL$");
	menumaker->addCategory("&Synthesis/&Arithmetic + Mixing","Synth_XFADE$");
	menumaker->addCategory("&Synthesis/&Busses","Synth_BUS_");
	menumaker->addCategory("&Synthesis/&Delays","Synth_DELAY$");
	menumaker->addCategory("&Synthesis/&Delays","Synth_CDELAY$");
	menumaker->addCategory("&Synthesis/&Envelopes","Synth_PSCALE$");
	menumaker->addCategory("&Synthesis/&Envelopes","Synth_ENVELOPE_");
	menumaker->addCategory("&Synthesis/Effe&cts","Synth_FX_");
	menumaker->addCategory("&Synthesis/&Filters","Synth_ATAN_SATURATE$");
	menumaker->addCategory("&Synthesis/&Filters","Synth_MOOG_VCF");
	menumaker->addCategory("&Synthesis/&Filters","Synth_SHELVE_CUTOFF$");
	menumaker->addCategory("&Synthesis/&Filters","Synth_RC$");
	menumaker->addCategory("&Synthesis/&Filters","Synth_STD_EQUALIZER$");
	menumaker->addCategory("&Synthesis/&Midi + Sequencing","Synth_MIDI");
	menumaker->addCategory("&Synthesis/&Midi + Sequencing","Interface_MIDI");
	menumaker->addCategory("&Synthesis/&Midi + Sequencing","Synth_SEQUENCE$");
	menumaker->addCategory("&Synthesis/&Midi + Sequencing","Synth_STRUCT_KILL$");
	menumaker->addCategory("&Synthesis/Sam&ples ","Synth_PLAY_");
	menumaker->addCategory("&Synthesis/&Sound IO","Synth_PLAY$");
	menumaker->addCategory("&Synthesis/&Sound IO","Synth_FULL_DUPLEX_");
	menumaker->addCategory("&Synthesis/&Sound IO","Synth_FILEPLAY");
	menumaker->addCategory("&Synthesis/&Tests","Synth_NIL$");
	menumaker->addCategory("&Synthesis/&Tests","Synth_DEBUG$");
	menumaker->addCategory("&Synthesis/&Tests","Synth_MIDI_DEBUG$");
	menumaker->addCategory("&Synthesis/&Oscillation & Modulation","Synth_FREQUENCY$");
	menumaker->addCategory("&Synthesis/&Oscillation & Modulation","Synth_FM_SOURCE$");
	menumaker->addCategory("&Synthesis/&WaveForms","Synth_WAVE_");
	menumaker->addCategory("&Synthesis/&Internal","Synth_PARAM_");
	menumaker->addCategory("&Synthesis/&Internal","Synth_AMAN_");

	menumaker->addCategory("&Examples","example_");
	menumaker->addCategory("&Instruments","instrument_");
	menumaker->addCategory("&Mixer-Elements","mixer_element_");
	menumaker->addCategory("&Templates","template_");
	menumaker->addCategory("&Other","*");

/*
	m_modules->insertItem(i18n("&Gui"), m_modules_gui);
	m_modules->insertItem(i18n("&Synthesis"), m_modules_synth);
	m_modules->insertItem(i18n("&Instruments"), m_modules_instruments);
	m_modules->insertItem(i18n("&Other"), m_modules_other);
 */

	connect(menubar,SIGNAL(highlighted(int)),this,SLOT(activateMenu(int)));
	connect(m_view,SIGNAL(activated(int)),modulewidget,SLOT(setZoom(int)));
	connect(m_ports,SIGNAL(activated(int)),this,SLOT(addPort(int)));
	connect(m_file_new,SIGNAL(activated(int)),this,SLOT(fileNew(int)));

	connect(menumaker,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	//connect(m_modules,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	/*
	connect(m_modules_synth,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	connect(m_modules_gui,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	connect(m_modules_instruments,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	connect(m_modules_other,SIGNAL(activated(int)),this,SLOT(addModule(int)));
	*/
	connect(theArtsBuilderApp,SIGNAL(lastWindowClosed()), this , SLOT(quit()));

	// update the modules menu once for the start
	activateMenu(modules_menu_item);
}

void ModuleView::activateMenu(int item)
{
	if(item == modules_menu_item)
	{
		long updateCount = ModuleBroker->updateCount();

		// if the contents of the ModukeBroker changed, update our "modules"-Menu
		if(updateCount != mbroker_updateCount)
		{
			mbroker_updateCount = updateCount;
			//---- query all available objects ----
			Arts::StringSeq_var Modules = ModuleBroker->publishedModules();
			assert(Modules);

			menumaker->clear();
			m_file_new->clear();

			unsigned long i;
			for(i=0;i<Modules->length();i++)
			{
				const char *name = (*Modules)[i];
				menumaker->addItem(name,i);
			
				if(strncmp(name,"template_",strlen("template_")) == 0)
				{
					char *xname = strdup(&name[strlen("template_")]);
					int x;
					for(x=0;xname[x] != 0; x++)
						if(xname[x] == '_') xname[x] = ' ';

					m_file_new->insertItem(xname, i);
				}
			}
		}
	}
}

void ModuleView::quit()
{
	if(execDlg) return;
	printf(">> ModuleView::quit() called\n");
	delete structure;
	GUIServer->decRef();
	printf("calling theArtsBuilderApp->quit();\n");
	theArtsBuilderApp->quit();
	printf("<< leaving ModuleView::quit()\n");
}

ModuleView::~ModuleView()
{
}

void ModuleView::clear()
{
	structure->clear();
}

void ModuleView::publish()
{
	checkName();
	structure->publish();
	QMessageBox::information(0,kapp->getCaption(),
		QString(i18n("The structure has been published as: '"))
		+QString(structure->name())+QString(i18n("' on the server.")),
	QMessageBox::Ok);
}

QString ModuleView::getOpenFilename(const char *pattern)
{
	printf(">>>>> getOpenFilename\n");
	QString filename = KFileDialog::getOpenFileName(0,pattern,this);
	printf(">>>>> opendlg closed; esd = %d\n",theArtsBuilderApp->eventStackDepth());
	if(!filename.isEmpty())
	{
		printf("open... %s\n",filename.data());

		// check that the file is ok:

		FILE *infile = fopen(filename,"r");

		if(infile)
		{
			fclose(infile);
			return(filename);
		}
	}
	return QString("");
}

void ModuleView::open()
{
	chdir(DirManager::structureDir());
	QString filename = getOpenFilename("*.arts");

	if(!filename.isEmpty())
	{
		structure->load(filename);
		modulewidget->reInit();
		if(!structure->valid())
		{
   			QMessageBox::message( i18n("Arts Warning"),
			i18n("The structure couldn't be loaded correctly. Maybe some of\n"
				 "the modules used in the file aren't available in this\n"
				 "version of aRts."));
		}
	}
}

void ModuleView::openSession()
{
	chdir(DirManager::sessionDir());

	QString filename = getOpenFilename("*.arts-session");

	if(!filename.isEmpty())
	{
		Arts::ArtsServerSeq preferredservers;

		preferredservers.length(1);
		preferredservers[0] = Arts::ArtsServer::_duplicate(GUIServer);

		Session *session = new Session(Synthesizer);
		session->loadSession(filename);

		assert(!execDlg);
		execDlg = new ExecDlg(0,Synthesizer,session,GUIServer);
		assert(execDlg);

		// this will create the widgets that will eventually get into the
		// execdlg
		session->startExecute(preferredservers);

		execDlg->start();
		execDlg->show();

		connect(execDlg,SIGNAL( ready() ), this, SLOT( endexecute() ));

		hide();
	}
}

void ModuleView::save()
{
	checkName();
	chdir(DirManager::structureDir());
    QString filename;
	KFileDialog *dlg = new KFileDialog(0,"*.arts",this,0,true,false);
	string defaultname = string(structure->name())+string(".arts");

	dlg->setSelection(defaultname.c_str());
	dlg->setCaption(i18n("Open"));
			  
    if (dlg->exec() == QDialog::Accepted)
    filename = dlg->selectedFile();
					   
    delete dlg;                                                                 
	// QString filename = KFileDialog::getSaveFileName(0,"*.arts",this);
	// filename.detach();

	if(!filename.isEmpty())
	{
		printf("save... %s\n",filename.data());

		// clear the file first:

		FILE *outfile = fopen(filename,"w");

		if(!outfile) return;
		fclose(outfile);

		structure->save(filename);
	}
}

void ModuleView::checkName()
{
	if(strncmp(structure->name(),"template_",strlen("template_")) == 0)
		rename();
}

void ModuleView::rename()
{
	NameDlg *nd = new NameDlg(0,structure->name(),i18n("Rename Structure"));
	assert(nd);

	if(nd->exec())
	{
		printf("rename OK...\n");
		structure->rename(nd->name());
	}
	delete nd;
}

void ModuleView::retrieve()
{
	RetrieveDlg *rd = new RetrieveDlg(0,Synthesizer);
	assert(rd);

	if(rd->exec())
	{
		const char *result = rd->result();
		if(result)
		{
			structure->retrieve(result);
			modulewidget->reInit();
		}
	}
	delete rd;
}

void ModuleView::execute()
{
	Arts::ArtsServerSeq preferredservers;

	preferredservers.length(1);
	preferredservers[0] = Arts::ArtsServer::_duplicate(GUIServer);

	assert(!execDlg);
	execDlg = new ExecDlg(0,Synthesizer,structure,GUIServer);
	assert(execDlg);

	// this will create the widgets that will eventually get into the
	// execdlg
	structure->startExecute(preferredservers);

	execDlg->start();
	execDlg->show();

	connect(execDlg,SIGNAL( ready() ), this, SLOT( endexecute() ));

	hide();
}

void ModuleView::endexecute()
{
	show();
	assert(execDlg);
	delete execDlg;
	// will be done by the execDlg itself now
	//structure->stopExecute();

	execDlg = 0;
}

void ModuleView::fileNew(int what)
{
	const char *name = menumaker->findID(what);
	assert(name);
	structure->retrieve(name);
	modulewidget->reInit();
}

void ModuleView::addPort(int port)
{
	Arts::PortType pt;

	printf("selected (port=%d)\n", port);
	switch(port)
	{
		case 1:		// data that goes into the structure
			pt.Direction = Arts::output;
			pt.DataType = Arts::audio_data;
			pt.ConnType = Arts::stream;
			break;
		case 2:		// data that goes out of the structure
			pt.Direction = Arts::input;
			pt.DataType = Arts::audio_data;
			pt.ConnType = Arts::stream;
			break;
		case 3:		// data that goes into the structure
			pt.Direction = Arts::output;
			pt.DataType = Arts::string_data;
			pt.ConnType = Arts::property;
			break;
		case 4:		// data that goes into the structure
			pt.Direction = Arts::output;
			pt.DataType = Arts::audio_data;
			pt.ConnType = Arts::property;
			break;
		default:	// forget it, no add intended
			return;
	}
	modulewidget->addPort(pt);
}

void ModuleView::changePortPositions()
{
	PortPosDlg *ppd = new PortPosDlg(this,structure);
	ppd->exec();
	// XXX: delete ppd?
}

void ModuleView::addModule(int module)
{
	const char *name = menumaker->findID(module);
	assert(name);

	printf("selected (%s) (module=%d)\n", name,module);

	Arts::ModuleBroker_var ModuleBroker = Synthesizer->moduleBroker();
	Arts::ModuleInfo_var minfo = ModuleBroker->lookupModule(name);


	if(minfo)
	{
		modulewidget->addModule(minfo);

/*
		Module *m = structure->createModule(minfo);
		modulewidget->addModule(m);
*/
	}
}

ArtsBuilderApp::ArtsBuilderApp( int &argc, char** argv ) : 
  KCorbaApplication( argc, argv, "artsbuilder" )
{
}

void ArtsBuilderApp::start()
{
	moduleview = new ModuleView("main", theSynthesizer);

	moduleview->resize(680,500);
	moduleview->setCaption("Arts Builder - Module View");
	moduleview->show();
	Arts::ModuleBroker_var ModuleBroker = theSynthesizer->moduleBroker();
	assert(ModuleBroker);

	//setMainWidget(moduleview);
	setTopWidget(moduleview);
}

void ArtsBuilderApp::end()
{
	delete moduleview;

	printf(">>>>>decref now\n");
	fflush(stdout);
	theSynthesizer->decRef();
	printf("<<<<<decref ok\n");
	fflush(stdout);
}

#ifdef COMMON_BINARY
int artsbuilder_main(int argc, char **argv)
#else
int main(int argc, char **argv)
#endif
{
	const char *serveriiop = "inet:localhost:8888";

	ArtsBuilderApp *Application = new ArtsBuilderApp(argc,argv);

// ---- connect to synthesizer
	CORBA::ORB_ptr orb = Application->orb();

	ArtsOrb = orb;		// for artsobject implementation

	CORBA::Object_var obj =
		orb->bind ("IDL:Arts/Synthesizer:1.0", serveriiop);
	if (CORBA::is_nil (obj)) {
		if(QMessageBox::information(0,Application->getCaption(),
		i18n("Synth server unavailable.\nShould I try to start it (or quit)?"),
		QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes)
		{
			system("kvt -e artsserver &");
			sleep(3);

			obj = orb->bind ("IDL:Arts/Synthesizer:1.0", serveriiop);
			while (CORBA::is_nil (obj))
			{
				if (QMessageBox::warning(0,Application->getCaption(),
					"Synthesizer still not found...", QMessageBox::Retry,
						QMessageBox::Cancel) == QMessageBox::Cancel)
				{
					exit(1);
				}
				obj = orb->bind ("IDL:Arts/Synthesizer:1.0", serveriiop);
			}
		}
		else
		{
			exit(1);
		}
	}

	Arts::Synthesizer_var Synthesizer = Arts::Synthesizer::_narrow (obj);

// ---- now Synthesizer is a valid CORBA Synthesizer object ;)
	Synthesizer->incRef();

	theSynthesizer = Synthesizer;		// for ArtsBuilderApp
	theArtsBuilderApp = Application;	// for QPainter Eventstack check

	Application->exec();

	fprintf(stderr,"================== leaving event loop ================\n");

	//Synthesizer->decRef();
}
