// KreateCD - CD recording software for the K desktop environment
//
// 1999 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "FileTree.h"
#include "IsoFile.h"

#include <qobject.h>
#include <qlistview.h>
#include <kfileinfo.h>
#include <kdir.h>
#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>

#if QT_VERSION >= 200
#include <kglobal.h>
#endif

extern KLocale *locale;

FileTreeItem::FileTreeItem(ISOFile *isofile,FileTreeItem *parent):QListViewItem(parent)
 {
  parentItem=parent;
  setupISOFile(isofile);
 }

FileTreeItem::FileTreeItem(ISOFile *isofile,FileTree *parent):QListViewItem(parent)
 {
  parentItem=0;
  setupISOFile(isofile);
 }


void FileTreeItem::setupISOFile(ISOFile *isofile)
 {
  QPixmap pmap;
  fileObject=0;
  dirObject=0;
  dummyItem=0;

  fileList=new FileTreeItems();
  isoFile=isofile;
  isDirExpanded=false;
  isDir=false;

  if (isoFile->type()==ISOFile::ISO_RealDir)
   {
    fileObject=new KFileInfo(isoFile->reference()->data(),"");
    setupDirObject();
   }

  switch (isoFile->type())
   {
    case ISOFile::ISO_ISODir:
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_folder_red");
#else
      pmap=Icon("kcd_folder_red.xpm");
#endif
      isDir=true;
      break;
    case ISOFile::ISO_RealDir:
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_folder_blue");
#else
      pmap=Icon("kcd_folder_blue.xpm");
#endif
      isDir=true;
      break;
    case ISOFile::ISO_RealFile:
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_unknown_blue");
#else
      pmap=Icon("kcd_unknown_blue.xpm");
#endif
      break;
   }

  setText(0,isoFile->name()->data());
  setPixmap(0,pmap);

  if (isDir)
   {
    dummyItem=new FileTreeItem(this);
   }
 }



FileTreeItem::FileTreeItem(FileTreeItem *parent):QListViewItem(parent)
 {
  QPixmap pmap;
  fileObject=0;
  dirObject=0;
  dummyItem=0;

  fileList=new FileTreeItems();
  isoFile=0;
  isDirExpanded=false;
  isDir=false;
  parentItem=parent;

#if QT_VERSION >= 200
  pmap=BarIcon("kcd_folder");
#else
  pmap=Icon("kcd_folder.xpm");
#endif
  setText(0,".");
  setPixmap(0,pmap);
 }



FileTreeItem::FileTreeItem(KFileInfo *fileinfo,FileTree *parent):QListViewItem(parent)
 {
  parentItem=0;
  setupKFileInfo(fileinfo);
 }

FileTreeItem::FileTreeItem(KFileInfo *fileinfo,FileTreeItem *parent):QListViewItem(parent)
 {
  parentItem=parent;
  setupKFileInfo(fileinfo);
 }


void FileTreeItem::setupKFileInfo(KFileInfo *fileinfo)
 {
  QPixmap pmap;

  fileObject=new KFileInfo(*fileinfo);
  fileList=new FileTreeItems();
  isoFile=0;
  dummyItem=0;
  isDirExpanded=false;
  isDir=false;

  setupDirObject();

  if (fileObject->isDir())
   {
    if (fileObject->isReadable())
     {
      isDir=true;
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_folder");
#else
      pmap=Icon("kcd_folder.xpm");
#endif
     }
     else
     {
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_lockedfolder");
#else
      pmap=Icon("kcd_lockedfolder.xpm");
#endif
     }
   }
   else
   {
    if (fileObject->isReadable())
     {
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_unknown");
#else
      pmap=Icon("kcd_unknown.xpm");
#endif
     }
     else
     {
#if QT_VERSION >= 200
      pmap=BarIcon("kcd_locked");
#else
      pmap=Icon("kcd_locked.xpm");
#endif
     }
   }

  if (strcmp("",fileObject->fileName())==0)
   {
    this->setText(0,fileObject->absURL());
   }
   else
   {
    this->setText(0,fileObject->fileName());
   }
  this->setPixmap(0,pmap);

  if (isDir)
   {
    dummyItem=new FileTreeItem(this);
   }
 }




FileTreeItem::~FileTreeItem(void)
 {
  if (fileObject!=0) delete(fileObject);
  if (dirObject!=0) delete (dirObject);
  delete(fileList);
 }

void FileTreeItem::setupDirObject(void)
 {
  if ( (fileObject->isDir()) && (fileObject->isReadable()) )
   {
    QString filename;
    getFilename(&filename,fileObject);
    dirObject=new KDir(filename);
    connect(dirObject,SIGNAL(newFilesArrived(const KFileInfoList *)),
            this,SLOT(newFiles(const KFileInfoList *)));
    connect(dirObject,SIGNAL(finished()),
            this,SLOT(dropDummy()));
   }
   else
   {
    dirObject=0;
   }
 }


void FileTreeItem::setOpen(bool doopen)
 {
  if ( (doopen) && (parentItem!=0) )
   {
    parentItem->setOpen(doopen);
   }
  doopen?expandTree():collapseTree();
  QListViewItem::setOpen(doopen);
 }

void FileTreeItem::expandTree(void)
 {
  if (!isDir) return;
  if (isDirExpanded==true) return;
  if (isoFile!=0)
   {
    if (isoFile->type()==ISOFile::ISO_ISODir)
     {
      ISOFile *walker=0;
      while ( (walker=isoFile->getChildren(walker))!=0)
       {
        addISOObject(walker,false,true);
       }
      dropDummy();
      isDirExpanded=true;
      return;
     }
   }
  if (dirObject==0) return;
  newFiles(dirObject->entryInfoList());
  if (dirObject->isFinished()) dropDummy();
  isDirExpanded=true;
 }

void FileTreeItem::dropDummy(void)
 {
  if (dummyItem)
   {
    delete dummyItem;
    dummyItem=0;
   }
 }

void FileTreeItem::collapseTree(void)
 {
  FileTreeItem *item;

  if (!isDir) return;
  if (!fileList->isEmpty())
   {
    collapseSubTree();
   }

  while ( (item=fileList->first())!=0)
   {
    fileList->remove(item);
    delete item;
   }
  if (!dummyItem)
   {
    dummyItem=new FileTreeItem(this);
   }
  isDirExpanded=false;
 }

void FileTreeItem::expandSubTree(void)
 {
  FileTreeItem *item;
  item=fileList->first();
  while (item!=0)
   {
    item->expandTree();
    if (item->isOpen()) item->expandSubTree();
    item=fileList->next();
   }
 }

void FileTreeItem::collapseSubTree(void)
 {
  FileTreeItem *item;
  item=fileList->first();
  while (item!=0)
   {
    item->collapseTree();
    item=fileList->next();
   }
 }



void FileTreeItem::newFiles(const KFileInfoList *infos)
 {
  KFileInfoList *infolist;
  KFileInfo *info;
  FileTreeItem *item;

  infolist=new KFileInfoList(*infos);
  info=infolist->first();
  while (info!=0)
   {
    if ( (strcmp("..",info->fileName())==0) ||
         (strcmp(".",info->fileName())==0) )
     {
      info=infolist->next();
      continue;
     }
    item=new FileTreeItem(info,this);
    addItemSorted(item);
    info=infolist->next();
   }
 }


void FileTreeItem::getFilename(QString *fname,KFileInfo *finfo)
 {
  *fname=finfo->absURL();
  if (strcmp(finfo->absURL().data(),"//")==0)
   {
    *fname="/";
    *fname+=finfo->fileName();
   }
   else
   {
    QString tmp;
    *fname=finfo->absURL();
    tmp=fname->right(2);
    while (tmp=fname->right(2),strcmp(tmp.data(),"//")==0) {
      fname->truncate(fname->length()-1);
     }
    *fname+=finfo->fileName();
   }
 }

bool FileTreeItem::fileName(QString *filename)
 {
  if (fileObject==0) return (false);
  getFilename(filename,fileObject);
  return(true);
 }

bool FileTreeItem::addISOObject(ISOFile *isoobj,bool addit,bool forceitem)
 {
  FileTreeItem *item=0;
  if (isoFile==0) return(false);
  if ( (addit) && (isoFile->findChildren(isoobj->name())!=0)) return(false);
  if (isoFile->type()!=ISOFile::ISO_ISODir) return(false);
  if ( (this->isOpen()) || forceitem)
   {
    item=new FileTreeItem(isoobj,this);
    addItemSorted(item);
   }
  if (addit) isoFile->addObject(isoobj);
  if ( ((this->isOpen()) || forceitem) && (item->isOpen())) item->expandTree();
  return(true);
 }

void FileTreeItem::addItemSorted(FileTreeItem *item)
 {
  FileTreeItem *child,*ochild;
  const char *text1;
  int idx=0;
  bool preflag=false;

  child=fileList->first();
  if (child==0)
   {
    fileList->append(item);
    return;
   }
  text1=item->text(0);
  while (child!=0)
   {
    ochild=child;
    if (strcmp(text1,child->text(0))<0)
     {
      preflag=true;
      break;
     }
    child=fileList->next();
    ++idx;
   }
  fileList->insert(idx,item);
 }


bool FileTreeItem::deleteObject(void)
 {
  if (isoFile!=0)
   {
    if (isoFile->type()==ISOFile::ISO_ISODir)
     {
      while (!this->fileList->isEmpty())
       {
        FileTreeItem *temp;
        temp=this->fileList->first();
        if (!temp->deleteObject()) return(false);
        delete(temp);
       }
     }
   }

  if (parentItem!=0)
   {
    if (fileObject!=0) delete(fileObject);
    fileObject=0;
    if (dirObject!=0) delete(dirObject);
    dirObject=0;
    if (isoFile!=0) delete(isoFile);
    isoFile=0;
    parentItem->fileList->removeRef(this);
   }
   else
   {
    return(false);
   }
  return(true);
 }


// main tree widget

FileTree::FileTree(QWidget *parent,const char *name)
         :QListView(parent,name)
 {
  this->addColumn(locale->translate("Filename"));
  currentItem=0;
  connect(this,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(changeSelect(QListViewItem *)));
 }

FileTree::~FileTree(void)
 {
 }

void FileTree::changeSelect(QListViewItem *sitem)
 {
  currentItem=(FileTreeItem *) sitem;
 }

void FileTree::addDir(KFileInfo *finfo)
 {
  FileTreeItem *item;

  item=new FileTreeItem(finfo,this);
  item->setOpen(true);
 }

void FileTree::addDir(ISOFile *isofile)
 {
  FileTreeItem *item;
  item=new FileTreeItem(isofile,this);
  item->setOpen(true);
  setSelected(item,true);
 }


bool FileTree::getSelected(QString *fname)
 {
  if (currentItem==0) return(false);

  return(currentItem->fileName(fname));
 }

bool FileTree::deleteObject(void)
 {
  FileTreeItem *ci;

  if (currentItem==0) return(false);
  ci=currentItem;
  currentItem=0;
  if (!ci->deleteObject()) return(false);
  delete(ci);
  return(true);
 }

bool FileTree::addRealObject(QString *fname)
 {
  QString path;
  QString file;
  QString complete;

  KFileInfo *kfi;
  ISOFile::ISOType itype;
  int index;
  ISOFile *ifil;

  path=fname->data();
  file=fname->data();
  if (currentItem==0) return(false);

  index=fname->findRev('/');
  if (index==-1) return(false);
  path.remove(index+1,file.length()-(index+1));
  file.remove(0,index+1);
  if (strcmp(path.right(2).data(),"//")==0)
   {
    path.truncate(path.length()-1);
   }
  if (strcmp(path.right(1).data(),"/")!=0)
   {
    path+="/";
   }
  kfi=new KFileInfo(path,file);
  if (kfi==0) return(false);

  if (!kfi->isReadable())
   {
    delete(kfi);
    return(false);
   }

  if (kfi->isDir())
   {
    itype=ISOFile::ISO_RealDir;
   }
   else
   {
    itype=ISOFile::ISO_RealFile;
   }


  complete=path.data();
  complete+=file.data();

  ifil=new ISOFile(itype,&file,&complete);

  currentItem->setOpen(true);
  if (!currentItem->addISOObject(ifil))
   {
    delete(ifil);
    delete(kfi);
    return(false);
   }
  delete(kfi);
  return(true);
 }

bool FileTree::createISODir(const char *dirname)
 {
  ISOFile *isofil;
  QString qs;

  if (currentItem==0) return(false);
  qs=dirname;

  isofil=new ISOFile(ISOFile::ISO_ISODir,&qs);
  currentItem->setOpen(true);
  if (!currentItem->addISOObject(isofil))
   {
    delete(isofil);
    return(false);
   }
  return(true);
 }

void FileTree::resizeEvent(QResizeEvent *rsiz)
 {
  QListView::resizeEvent(rsiz);
  setColumnWidth(0,width());
 }
