// KreateCD - CD recording software for the K desktop environment
//
// 1999 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "ConfPath.h"

#include <qstring.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qlineedit.h>

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include <kurl.h>

#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>


extern KLocale *locale;

ConfPath::ConfPath( QWidget *parent, const char *name):QWidget(parent,name)
 {
  KConfig *config;
  QVBoxLayout *qwinlay;
  QHBoxLayout *buttonlayout;
  QGridLayout *qgrid;
  QPushButton *parabut,*cdrbut,*mkisobut,*tmpbut;

  QRect DefaultRect(100,50,500,300);
  config=kapp->getConfig();

  qwinlay=new QVBoxLayout(this);
  qgrid=new QGridLayout(9,6);
  qwinlay->addStretch(1);
  qwinlay->addLayout(qgrid,10);
  qwinlay->addStretch(1);

  setCaption(locale->translate("KreateCD - path configuration"));

  qgrid->setColStretch(0,3);
  qgrid->setColStretch(3,3);
  qgrid->setColStretch(5,3);
  qgrid->setColStretch(1,10);
  qgrid->setColStretch(2,20);
  qgrid->setColStretch(4,0);
  qgrid->setRowStretch(0,1);
  qgrid->setRowStretch(2,1);
  qgrid->setRowStretch(4,1);
  qgrid->setRowStretch(6,1);
  qgrid->setRowStretch(8,1);


  ParanoiaLabel=new QLabel(locale->translate("Path : cdparanoia"),this);
  ParanoiaLabel->setMinimumSize(ParanoiaLabel->sizeHint());
  ParanoiaPath=new QLineEdit(this,"paranoia path");
  ParanoiaPath->setMaxLength(39);
  ParanoiaPath->setMinimumSize(ParanoiaPath->sizeHint());
  qgrid->addWidget(ParanoiaLabel,1,1,AlignCenter);
  qgrid->addWidget(ParanoiaPath,1,2);

  CdrecordLabel=new QLabel(locale->translate("Path : cdrecord"),this);
  CdrecordLabel->setMinimumSize(CdrecordLabel->sizeHint());
  CdrecordPath=new QLineEdit(this,"cdrecord path");
  CdrecordPath->setMaxLength(39);
  CdrecordPath->setMinimumSize(CdrecordPath->sizeHint());
  qgrid->addWidget(CdrecordLabel,3,1,AlignCenter);
  qgrid->addWidget(CdrecordPath,3,2);

  IsofsLabel=new QLabel(locale->translate("Path : mkisofs"),this);
  IsofsLabel->setMinimumSize(IsofsLabel->sizeHint());
  IsofsPath=new QLineEdit(this,"mkisofs path");
  IsofsPath->setMaxLength(39);
  IsofsPath->setMinimumSize(IsofsPath->sizeHint());
  qgrid->addWidget(IsofsLabel,5,1,AlignCenter);
  qgrid->addWidget(IsofsPath,5,2);

  TemporaryLabel=new QLabel(locale->translate("Temporary directory"),this);
  TemporaryLabel->setMinimumSize(TemporaryLabel->sizeHint());
  TemporaryPath=new QLineEdit(this,"temporary path");
  TemporaryPath->setMaxLength(39);
  TemporaryPath->setMinimumSize(TemporaryPath->sizeHint());
  qgrid->addWidget(TemporaryLabel,7,1,AlignCenter);
  qgrid->addWidget(TemporaryPath,7,2);

  parabut=new QPushButton("...",this);
  parabut->setFixedSize(parabut->sizeHint());
  qgrid->addWidget(parabut,1,4,AlignCenter);
  cdrbut=new QPushButton("...",this);
  cdrbut->setFixedSize(parabut->sizeHint());
  qgrid->addWidget(cdrbut,3,4,AlignCenter);
  mkisobut=new QPushButton("...",this);
  mkisobut->setFixedSize(parabut->sizeHint());
  qgrid->addWidget(mkisobut,5,4,AlignCenter);
  tmpbut=new QPushButton("...",this);
  tmpbut->setFixedSize(parabut->sizeHint());
  qgrid->addWidget(tmpbut,7,4,AlignCenter);


  buttonlayout=new QHBoxLayout();
  qwinlay->addLayout(buttonlayout,1);

  okbutton=new QPushButton(locale->translate("OK"),this);
  okbutton->setFixedSize(okbutton->sizeHint());
  buttonlayout->addStretch();
  buttonlayout->addWidget(okbutton,1,AlignCenter);
  buttonlayout->addStretch();
  applybutton=new QPushButton(locale->translate("Apply"),this);
  applybutton->setFixedSize(applybutton->sizeHint());
  buttonlayout->addWidget(applybutton,1,AlignCenter);
  buttonlayout->addStretch();
  cancelbutton=new QPushButton(locale->translate("Cancel"),this);
  cancelbutton->setFixedSize(cancelbutton->sizeHint());
  buttonlayout->addWidget(cancelbutton,1,AlignCenter);
  buttonlayout->addStretch();

  connect(okbutton,SIGNAL(clicked()),this,SLOT(okButton()));
  connect(applybutton,SIGNAL(clicked()),this,SLOT(applySettings()));
  connect(cancelbutton,SIGNAL(clicked()),this,SLOT(cancelButton()));
  connect(parabut,SIGNAL(clicked()),this,SLOT(changeParanoia()));
  connect(cdrbut,SIGNAL(clicked()),this,SLOT(changeCdrecord()));
  connect(mkisobut,SIGNAL(clicked()),this,SLOT(changeMkisofs()));
  connect(tmpbut,SIGNAL(clicked()),this,SLOT(changeTemp()));

   {
    QString tstring;
    config->setGroup("Path");
    tstring=config->readEntry("PathCdparanoia","cdparanoia");
    ParanoiaPath->setText(tstring.data());
    tstring=config->readEntry("PathCdrecord","cdrecord");
    CdrecordPath->setText(tstring.data());    
    tstring=config->readEntry("PathMkisofs","mkisofs");
    IsofsPath->setText(tstring.data());
    tstring=config->readEntry("PathTemporary","/tmp/");
    TemporaryPath->setText(tstring.data());
   }
  

  config->setGroup("Geometry");
  this->setGeometry(config->readRectEntry("PathWindowGeo",&DefaultRect));
  qwinlay->activate();
 }


ConfPath::~ConfPath(void)
 {
  KConfig *config;
  QRect SaveRect(this->pos(),this->size());
  config=kapp->getConfig();
  config->setGroup("Geometry");
  config->writeEntry("PathWindowGeo",SaveRect);
 }

void ConfPath::applySettings(void)
 {
  KConfig *config;
  config=kapp->getConfig();
  config->setGroup("Path");
  config->writeEntry("PathCdparanoia",ParanoiaPath->text());
  config->writeEntry("PathCdrecord",CdrecordPath->text());
  config->writeEntry("PathMkisofs",IsofsPath->text());
  config->writeEntry("PathTemporary",TemporaryPath->text());

 }

void ConfPath::hide()
 {
  QWidget::hide();
  emit closed();

 }

void ConfPath::okButton(void)
 {
  applySettings();
  hide();
 }

void ConfPath::cancelButton(void)
 {
  hide();
 }


int ConfPath::searchPath(const char *program,char *path)
 {
  char *pathenv;
  char thispath[512];
  char *sptr,*cptr;
  struct stat statbuf;

  pathenv=getenv("PATH");
  
  sptr=pathenv;
  cptr=sptr;
  do
   {
    if ( ( *cptr==':') || ( *cptr=='\0') )
     {
      strncpy(thispath,sptr,cptr-sptr);
      thispath[cptr-sptr]='/';
      thispath[cptr-sptr+1]=0;
      strcat(thispath,program); 
      if ( stat(thispath,&statbuf)==0)
       {
        if (path==0) return(1);
        strcpy(path,thispath);
        return(1);
       }
      sptr=cptr+1;
     }
    cptr++;
   } while (*(cptr-1)!=0);

  //check for /opt/schily/bin (standard install location,often not in PATH)

  strcpy(thispath,"/opt/schily/bin/");
  strcat(thispath,program);
  if (stat(thispath,&statbuf)==0)
   {
    if (path==0) return(1);
    strcpy(path,thispath);
    return(1);
   }
  return(0);
 }

int ConfPath::verifyPrograms(void)
 {
  int retv=1;
  const char *prog;
  QString progqs;
  KConfig *config;
  struct stat statbuf;
  char temppath[256];

  config=kapp->getConfig();
  config->setGroup("Path");
  progqs=config->readEntry("PathCdrecord","");
  prog=progqs.data();
  if ( (prog[0]==0) || (stat(prog,&statbuf)!=0) )
   {
    if (searchPath("cdrecord",temppath)==0)
     {
      retv=0;
     }
     else
     {
      config->writeEntry("PathCdrecord",temppath);
     }    
   }

  config=kapp->getConfig();
  progqs=config->readEntry("PathCdparanoia","");
  prog=progqs.data();
  if ( (prog[0]==0) || (stat(prog,&statbuf)!=0) )
   {
    if (searchPath("cdparanoia",temppath)==0)
     {
      retv=0;
     }
     else
     {
      config->writeEntry("PathCdparanoia",temppath);
     }    
   }

  config=kapp->getConfig();
  progqs=config->readEntry("PathMkisofs","");
  prog=progqs.data();
  if ( (prog[0]==0) || (stat(prog,&statbuf)!=0) )
   {
    if (searchPath("mkisofs",temppath)==0)
     {
      retv=0;
     }
     else
     {
      config->writeEntry("PathMkisofs",temppath);
     }    
   }

  progqs=config->readEntry("PathTemporary","");
  prog=progqs.data();
  if ( (prog[0]==0))
   {
    config->writeEntry("PathTemporary","/tmp");
   }

  return(retv);
 }


void ConfPath::changeParanoia(void)
 {
  QString openname;
#if QT_VERSION >= 200
  openname=KFileDialog::getOpenFileName(ParanoiaPath->text(),"*",this,locale->translate("Select location of cdparanoia"));
  KURL::decode(openname);
#else
  openname=KFileDialog::getOpenFileName(ParanoiaPath->text(),"*",this,locale->translate("Select location of cdparanoia"));
  KURL::decodeURL(openname);
#endif
  if (openname.isEmpty()) return;
  ParanoiaPath->setText(openname);
 }

void ConfPath::changeCdrecord(void)
 {
  QString openname;
#if QT_VERSION >= 200
  openname=KFileDialog::getOpenFileName(CdrecordPath->text(),"*",this,locale->translate("Select location of cdrecord"));
  KURL::decode(openname);
#else
  openname=KFileDialog::getOpenFileName(CdrecordPath->text(),"*",this,locale->translate("Select location of cdrecord"));
  KURL::decodeURL(openname);
#endif
  if (openname.isEmpty()) return;
  CdrecordPath->setText(openname);
 }

void ConfPath::changeMkisofs(void)
 {
  QString openname;
#if QT_VERSION >= 200
  openname=KFileDialog::getOpenFileName(IsofsPath->text(),"*",this,locale->translate("Select location of mkisofs"));
  KURL::decode(openname);
#else
  openname=KFileDialog::getOpenFileName(IsofsPath->text(),"*",this,locale->translate("Select location of mkisofs"));
  KURL::decodeURL(openname);
#endif
  if (openname.isEmpty()) return;
  IsofsPath->setText(openname);
 }

void ConfPath::changeTemp(void)
 {
  QString openname;
#if QT_VERSION >= 200
  openname=KFileDialog::getOpenFileName(TemporaryPath->text(),"*",this,locale->translate("Select temporary directory"));
  KURL::decode(openname);
#else
  openname=KFileDialog::getOpenFileName(TemporaryPath->text(),"*",this,locale->translate("Select temporary directory"));
  KURL::decodeURL(openname);
#endif
  if (openname.isEmpty()) return;
  TemporaryPath->setText(openname);
 }
