/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <SFNode.h>
#include <Children.h>
#include <TopLevel.h>
#include <SFEvent.h>
#include <SFRenderInfo.h>

/*************************************************************************/
void TopLevel::prepare (SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered TopLevel::prepare (" << DEF << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif 

  /* prepare the children */
  children.prepare(min, max);

#ifdef DEBUG_PREPARE
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone TopLevel\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void TopLevel::render  (SFRenderInfo &SFRI) 
{    

#ifdef DEBUG_RENDER
  cout << "\nEntered TopLevel::render (" << DEF << ")\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif 

  /* render the children */
  children.render(SFRI);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone TopLevel\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void TopLevel::parse(SFRenderInfo &SFRI, char *header, istream &InFile)
{

#ifdef DEBUG_PARSE
    cout << "\nEntered TopLevel::operator >> \n";
#endif

  /* verify that the first line in the file is correct */
  char Header[16];
  for (int i = 0; i < 16; i++) InFile >> Header[i];
  if (strncmp(Header, "#VRML V2.0 utf8", 15) != 0) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (TopLevel::operator >>)\n";
    cerr << "\tIncorrect vrml header (" << Header << ")\n";
    exit(2);
  }

  /* continue to read children until the end of file is reached. */
  while (! InFile.eof()) children.parse(header, InFile);
  
#ifdef DEBUG_PARSE
  cout << "\nTopLevel: Starting to Resolve ROUTES: looking for to/from events\n";
#endif

  /* create the list of event routes */
  SFString l = SFEventLabel;
  eventQueue() = new Children;
  push(*(eventQueue()), l);

#ifdef DEBUG_PARSE
    cout << "\nTopLevel: Event Queue created\n";
#endif

  /* visit each event and resolve the pointers */
  for (Children *C = eventQueue(); C != (Children *) 0; C = C->next()) {

    /* do not process if bogus */
    if (C->data() != (SFNode *) 0) {

      /* get the event */
      SFEvent *e = (SFEvent *) C->data();

#ifdef DEBUG_PARSE
      cout << "\tAdding Event: FROM (" << e->nodeFromName();
      cout << ") TO: (" << e->nodeToName();
      cout << ")\n";
#endif
      
      /* find the two nodes for routine */
      e->from() = findDEF(e->nodeFromName());
      e->to()   = findDEF(e->nodeToName());
      
      /* check */
      if (e->from() == (SFNode *) 0) {
	cerr << "\nError:\n";
	cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
	cerr << "\tCould not locate (" << e->nodeFromName();
	cerr << ") from route.\n";
	exit(0);      
      } else if (e->to() == (SFNode *) 0) {
	cerr << "\nError:\n";
	cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
	cerr << "\tCould not locate (" << e->nodeFromName();
	cerr << ") to route.\n";
	exit(0);      
      }
      
      /* add the event to the generator */
      e->from()->addEvent(e);
    }
  }

  /* resolve the use defs */
  ResolveUSEDEF();

  /* make sure everything is prepared */
  prepare(SFRI.min(), SFRI.max());

}
/*************************************************************************/

/*************************************************************************/
void TopLevel::ResolveUSEDEF()
{

#ifdef DEBUG_PARSE
  cout << "\nEntered TopLevel::ResolveUSEDEF\n";
#endif
  
  /* continue to resolve unsatisfied references until there are none */
  SFNode *Unsat = (SFNode *) 0;
  do {
    
    /* go through and resolve any unsatistified USE statements */
    Unsat = findUSE();
    
    /* if one was found find the corresponding def */
    if (Unsat != (SFNode *) 0) {    
      SFString *str = Unsat->use();
      if ((str != (SFString *) 0) && (str->length() > 0)) {
#ifdef DEBUG_PARSE
      cout << "\tTopLevel::ResolveUSEDEF(" << *str << ")\n";
#endif      
      SFNode *Sat = findDEF(*str);
      if (Sat != (SFNode *) 0) {
	  Unsat->markUseFound();
	  Unsat->copy(Sat);	
	} else {
	  cerr << "\nError:\n";
	  cerr << "\tOccurred in TopLevel::ResolveUSEDEF\n";
	  cerr << "\tUnsatisfied USE for (" << Unsat->nodeType() << ")\n";
	  exit(0);
	}
      } else {
	Unsat->markUseFound();
	Unsat = (SFNode *) 0;
      }
    }
  } while (Unsat != (SFNode *) 0);

#ifdef DEBUG_PARSE
  cout << "\tExiting TopLevel::ResolveUSEDEF\n";
#endif

}
/*************************************************************************/
