/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Material.h>

/*************************************************************************/
void Material::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Material::eventIn (" << DEF << ")\n";
#endif 
  
  /* Determine what action to take based on incoming event */
  if        ((eventType == "diffuseColor") || (eventType == "set_diffuseColor")) {
    SFColor   *n = (SFColor *) eventData;
    diffuseColor = *n;
    
  } else if ((eventType == "ambientIntensity")||(eventType == "set_ambientIntensity")) {
    SFFloat       *n = (SFFloat *) eventData;
    ambientIntensity = *n;

  } else if ((eventType == "specularColor") || (eventType == "set_specularColor")) {
    SFColor    *n = (SFColor *) eventData;
    specularColor = *n;

  } else if ((eventType == "emissiveColor") || (eventType == "set_emissiveColor")) {
    SFColor    *n = (SFColor *) eventData;
    emissiveColor = *n;

  } else if ((eventType == "shininess") || (eventType == "set_shininess")) {
    SFFloat *n = (SFFloat *) eventData;
    shininess  = *n;

  } else if ((eventType == "transparency") || (eventType == "set_transparency")) {
    SFFloat   *n = (SFFloat *) eventData;
    transparency = *n;

  }

}
/*************************************************************************/

/*************************************************************************/
void Material::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Material::render (" << DEF << ")\n";
#endif 

  /* set up the properties */
  GLfloat mat_diffuse[]   = { 0.8, 0.8, 0.8, 1.0 };
  GLfloat mat_specular[]  = { 0.0, 0.0, 0.0, 1.0 };
  GLfloat mat_shininess[] = { 0.2 };
  GLfloat mat_emission[]  = { 0.0, 0.0, 0.0, 0.0 };
  GLfloat mat_ambient[]   = { 0.2, 0.2, 0.2, 1.0 };
     
  /* set the diffuse color if present */
  mat_diffuse[0] = diffuseColor[0];
  mat_diffuse[1] = diffuseColor[1];
  mat_diffuse[2] = diffuseColor[2];

  /* set the ambient color if present */
  mat_ambient[0] = ambientIntensity * diffuseColor[0];
  mat_ambient[1] = ambientIntensity * diffuseColor[1];
  mat_ambient[2] = ambientIntensity * diffuseColor[2];

  /* set the specular color if present */
  mat_specular[0] = specularColor[0];
  mat_specular[1] = specularColor[1];
  mat_specular[2] = specularColor[2];

  /* set the emissive color if present */
  mat_emission[0] = emissiveColor[0];
  mat_emission[1] = emissiveColor[1];
  mat_emission[2] = emissiveColor[2];

  /* set the shininess color if present */
  mat_shininess[0] = shininess;

  /* set the parameters with opengl */
  glMaterialfv(GL_FRONT, GL_AMBIENT,   mat_ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE,   mat_diffuse); 
  glMaterialfv(GL_FRONT, GL_SPECULAR,  mat_specular);
  glMaterialfv(GL_FRONT, GL_EMISSION,  mat_emission);  
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);  

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif


}
/*************************************************************************/

/*************************************************************************/
void Material::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE    
    cout << "\tMaterial: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "diffuseColor") {
      diffuseColor.parse(header, InFile);

    } else if (Token == "ambientIntensity") {
      ambientIntensity.parse(header, InFile);

    } else if (Token == "specularColor") {
      specularColor.parse(header, InFile);

    } else if (Token == "emissiveColor") {
      emissiveColor.parse(header, InFile);

    } else if (Token == "shininess") {
      shininess.parse(header, InFile);

    } else if (Token == "transparency") {
      transparency.parse(header, InFile);

    } else if (Token == "Material") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/


