/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* C headers */
#include <values.h>

/* local headers */
#include <SFToken.h>
#include <Group.h>

/*************************************************************************/
void Group::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Group::prepare (" << DEF << ")\n";
#endif 

  /* first we need local max and mins */
  SFVec3f localMin;
  SFVec3f localMax;

  /* zero them out */
  localMin[0] =  MAXFLOAT;
  localMin[1] =  MAXFLOAT;
  localMin[2] =  MAXFLOAT;
  localMax[0] = -MAXFLOAT;
  localMax[1] = -MAXFLOAT;
  localMax[2] = -MAXFLOAT;

  /* prepare the children */
  children.prepare(localMin, localMax);

  /* update the local bounding box */
  for (int i = 0; i < 3; i++) {

    /* do the current box */
    bboxCenter[i] = (localMin[i] + localMax[i]) / 2.0;
    bboxSize[i]   = (localMax[i] - localMin[i]);

    /* update the scene box */
    if (localMin[i] < min[i]) min[i] = localMin[i];
    if (localMax[i] > max[i]) max[i] = localMax[i];
  }

  /* the boxes are valid */
  bboxCenter.isValid() = true;
  bboxSize.isValid() = true;

}
/*************************************************************************/

/*************************************************************************/
void Group::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Group::eventsIn (" << DEF << ")\n";
#endif 

  /* Determine what action to take based on incoming event */
  if (eventType == "addChildren") {
    children.add(eventData);

  } else if (eventType == "removeChildren") {
    children.remove(eventData);

  }
}
/*************************************************************************/

/*************************************************************************/
void Group::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Group::render (" << DEF << ")\n";
#endif 

  /* render the children */
  children.render(SFRI);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void Group::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tGroup: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "bboxSize") {
      bboxSize.parse(header, InFile);

    } else if (Token == "bboxCenter") {
      bboxCenter.parse(header, InFile);

    } else if (Token == "children") {
      children.parse(header, InFile);

    } else if (Token == "Group") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;

    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the node token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/


