/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** world.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "world.h"

world::world(const char *n) :
	base(0,n)
{
	numtype = NUM_WORLD;

	objectlist.setErrorValue(0);
	referencelist.setErrorValue(0);
	viewlist.setErrorValue(0);

	tlib = new TextureLibrary();

	ambient_light = Vector3(5,5,5);
	assumed_gamma = 2.2;
	background_color = Vector3(0,0,0);
}

world::world(world *wc) :
	base(0,wc)
{
	int	t;

	numtype = NUM_WORLD;

	objectlist.setErrorValue(0);
	objectlist.empty();
	referencelist.setErrorValue(0);
	referencelist.empty();
	viewlist.setErrorValue(0);
	viewlist.empty();

	for(t = 0;t < wc->objectlist.length();t++)
	{
		wc->objectlist[t]->copy(this);
	}

	for(t = 0;t < wc->referencelist.length();t++)
	{
		wc->referencelist[t]->copy(this);
	}

	for(t = 0;t < wc->viewlist.length();t++)
	{
		viewlist.append(wc->viewlist[t]);
	}


	// hier kopie erstellen !!!
	tlib = wc->tlib;

	ambient_light = wc->ambient_light;
	assumed_gamma = wc->assumed_gamma;
	background_color = wc->background_color;
}

world::~world()
{
	int	i;

	for(i = 0;i < objectlist.length();i++) 
	{ 
		delete objectlist[i]; 
	} 
	for(i = 0;i < referencelist.length();i++) 
	{ 
		delete referencelist[i]; 
	}
	objectlist.empty(); 
	referencelist.empty(); 
	viewlist.empty(); 

	if(tlib) delete tlib;
} 

base	*world::copy()
{
	return new world(this);
}

TextureLibrary	*world::getTextureLibrary()
{
	return tlib;

}

int	world::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	ref	*r;
	int	t;

	if(viewlist.length() > 0)
	{
		viewlist[0]->exportPOV(fp,tab,anim);
	}
	
	printTab(fp,tab);
	fprintf(fp,"// Global settings\n");
	printTab(fp,tab);
	fprintf(fp,"global_settings\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"ambient_light rgb<%g,%g,%g>\n",ambient_light[0],ambient_light[1],ambient_light[2]);
	printTab(fp,tab + tabsize);
	fprintf(fp,"assumed_gamma %g\n",assumed_gamma);
	printTab(fp,tab);
	fprintf(fp,"}\n\n\n");

	printTab(fp,tab);
	fprintf(fp,"// Background\n");
	printTab(fp,tab);
	fprintf(fp,"background\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"color rgb<%g,%g,%g>\n",background_color[0],background_color[1],background_color[2]);
	printTab(fp,tab);
	fprintf(fp,"}\n\n\n");

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name); 
	printTab(fp,tab); 
	fprintf(fp,"// Objecttype = world\n\n\n"); 

	for(t = 0;t < referencelist.length();t++)
	{
		r = referencelist[t];
		fprintf(fp,"#declare %s =\n",r->getName());
		r->exportPOV(fp,tab + tabsize,tabsize,anim);
		fprintf(fp,"\n");
	}

	for(t = 0;t < objectlist.length();t++)
	{
		objectlist[t]->exportPOV(fp,tab,tabsize,anim);
	}
	
	return 0;
}

void	world::dumpNames(int tab,int tabsize)
{
	int		i;

	printTab(stdout,tab);
	printf("world: %s\n",name);

	for(i = 0;i < viewlist.length();i++)
	{
		printf("view %s\n",viewlist[i]->getName());
	}
 
	for(i = 0;i < referencelist.length();i++)
	{
		referencelist[i]->dumpNames(tab + tabsize,tabsize);
	}

	for(i = 0;i < objectlist.length();i++)
	{ 
		objectlist[i]->dumpNames(tab + tabsize,tabsize); 
	} 
}

void	world::dumpNames(int tabsize)
{
	dumpNames(0,tabsize);
}

int	world::save(media *m)
{
	int	i;
        chunk	data;

	if(!m) return -1;

	setMedia(m);

	writeChunk("WRLD");
	
	writeNameChunk(name);

	{
		data.setMedia(m);
	
		data.writeChunk("DATA");
	
		data.writeVector(ambient_light);
		data.writeDouble(assumed_gamma);
		data.writeVector(background_color);

		data.writeChunkLen();
	}
	
	anim::save(m);

	tlib->save(m);

	printf("Save %i views\n",viewlist.length());
	for(i = 0;i < viewlist.length();i++)
	{
		viewlist[i]->save(m);
	}
	for(i = 0;i < referencelist.length();i++)
	{
		referencelist[i]->save(m);
	}
	for(i = 0;i < objectlist.length();i++)
	{
		objectlist[i]->save(m);
	}

	writeChunkLen();

	return 0;
}

int	world::load(media *m,int l)
{
	int	pos = m->tell(),len;
	base	*b;
	char	chunk[4];

	readChunk(chunk,len);

	if(strncmp(chunk,"DATA",4) == 0)
	{
		ambient_light = readVector();
		assumed_gamma = readDouble();
		background_color = readVector();
	}
	else
	{
		rejectChunk();
	}

	anim::load(m,l - (m->tell() - pos));

	readChunk(chunk,len);

	if(strncmp(chunk,"TXTL",4) == 0)
	{
		tlib->load(m,len);
	}
	else
	{
		rejectChunk();
	}

	do
	{
		readChunk(chunk,len);
		if(strncmp(chunk,"VIEW",4) == 0)
		{
			view	*v = new view(this,"View");

			v->setMedia(m);
			v->load(m,len);

			v->setWorld(this);
		}
		else
		{
			rejectChunk();
		}
	}
	while((m->tell() < pos + l) && (strncmp(chunk,"VIEW",4) == 0));

	while(m->tell() < pos + l)
	{
		b = parse(m);
		if(b)
		{
			b->addToParent(this);
			b->setParent(this);
		}
	}

	return 0;
}

int	world::addToParent(base *parent)
{
	if(!parent) return -2;
	return parent->addChild(this);
}

int	world::removeFromParent()
{
	if(!parent) return -2;
	return parent->removeChild(this);
}

int	world::addChild(body *child)
{
	if(!child) return -1;
	return objectlist += child;
}

int	world::addChild(blobobj *child)
{
	if(!child) return -1;
	return objectlist += child;
}

int	world::addChild(csgobj *child)
{
	if(!child) return -1;
	return objectlist += child;
}

int	world::addChild(nonsolid *child)
{
	if(!child) return -1;
	return objectlist += child;
}

int	world::addChild(refptr *child)
{
	if(!child) return -1;
	return objectlist += child;
}

int	world::addChild(ref *child)
{
	if(!child) return -1;
	return referencelist += child;
}

int	world::removeChild(base *child)
{
	if(!child) return -1;
	if(child->getType() == NUM_REF)
	{
		if(referencelist.find((ref*)child) < 0) return -2;
		referencelist.deleteCurrent();
		return 0;
	}
	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::removeChild(body *child)
{
	if(!child) return -1;

	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::removeChild(refptr *child)
{
	if(!child) return -1;

	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::removeChild(ref *child)
{
	if(!child) return -1;
	if(referencelist.find((ref*)child) < 0) return -2;
	referencelist.deleteCurrent();
	return 0;
}

int	world::removeChild(csgobj *child)
{
	if(!child) return -1;

	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::removeChild(blobobj *child)
{
	if(!child) return -1;

	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::removeChild(nonsolid *child)
{
	if(!child) return -1;

	if(objectlist.find((body*)child) < 0) return -2;
	objectlist.deleteCurrent();

	return 0;
}

int	world::existsName(const char *n)
{
	body	*d;
	ref	*r;
	view	*v;
	int	i;

	if(n == 0)
		return -1;

	if(name != 0 && strcmp(name,n) == 0)
		return !0;

	for(i = 0;i < viewlist.length();i++)
	{
		v = viewlist[i];
		if(v->existsName(n))
			return !0;
	}

	for(i = 0;i < objectlist.length();i++)
	{
		d = objectlist[i];
		if(d->existsName(n))
			return !0;
	}

	for(i = 0;i < referencelist.length();i++)
	{
		r = referencelist[i];
		if(r->existsName(n))
			return !0;
	}

	return 0;
}

base	*world::searchName(const char *n)
{
	body	*d;
	ref	*r;
	base	*b;
	view	*v;
	int	i;

	if(n == 0)
		return 0;

	if(name != 0 && strcmp(name,n) == 0)
		return this;

	for(i = 0;i < viewlist.length();i++)
	{
		v = viewlist[i];
		if((b = v->searchName(n)) != 0)
			return b;
	}

	for(i = 0;i < objectlist.length();i++)
	{
		d = objectlist[i];
		if((b = d->searchName(n)) != 0)
			return b;
	}

	for(i = 0;i < referencelist.length();i++)
	{
		r = referencelist[i];
		if((b = r->searchName(n)))
			return b;
	}

	return 0;
}

int	world::draw(view *v,Matrix44 m,int anim)
{
	body	*d;
	ref	*r;
	view	*vp;
	int	t;

	if(v == 0) return -1;

	for(t = 0;t < viewlist.length();t++)
	{
		vp = viewlist[t];
		vp->draw(v,m,anim);
	}
	for(t = 0;t < objectlist.length();t++)
	{
		d = objectlist[t];
		d->draw(v,m,anim);
	}

	for(t = 0;t < referencelist.length();t++)
	{
		r = referencelist[t];
		r->draw(v,m,anim);
	}

	return 0;
}

int	world::calculate(int)
{
	return 0;
}

Vector3	world::getAmbientLight()
{
	return ambient_light;
}

void	world::setAmbientLight(double r,double g,double b)
{
	ambient_light[0] = r;
	ambient_light[1] = g;
	ambient_light[2] = b;
}

void	world::setAmbientLight(Vector3 v)
{
	ambient_light = v;
}

void	world::setAmbientLight(double v)
{
	ambient_light[0] = v;
	ambient_light[1] = v;
	ambient_light[2] = v;
}

Vector3	world::getBackgroundColor()
{
	return background_color;
}

void	world::setBackgroundColor(double r,double g,double b)
{
	background_color[0] = r;
	background_color[1] = g;
	background_color[2] = b;
}

void	world::setBackgroundColor(Vector3 v)
{
	background_color = v;
}

int	world::addView(view *v)
{
	int	i;
	
	if(v == 0)
		return -1;
	if(v->getName() == 0)
		return -2;
	if(viewlist.find(v) >= 0)
		return -3;
        for(i = 0;i < viewlist.length();i++)
        {
        	if(viewlist[i]->getName()
        	 && strcmp(viewlist[i]->getName(),v->getName()) == 0)
        	 	return -4;
        }

	viewlist.append(v);

	return 0;
}

view	*world::getView(char *name)
{
	int		t;

	for(t = 0;t < viewlist.length();t++)
	{
		if(strcmp(viewlist[t]->getName(),name) == 0)
		{
			return viewlist[t];
		}
	}

	return 0;
}

view	*world::getView(int n)
{
	return viewlist[n];
}

int	world::removeView(view *v)
{
	int	pos;

	pos = viewlist.find(v);

	if(pos < 0)
		return pos;

	return viewlist.deleteAt(pos);
}

int	world::getNumViews()
{
	return viewlist.length();
}


