// vi:shiftwidth=4:tabstop=4:
/****************************************************************************
**
** Implementation of QwSpriteField and associated classes
**
** Author  : Warwick Allison (warwick@cs.uq.edu.au)
** Created : 19960330
**
** Copyright (C) 1996 by Warwick Allison.
**
*****************************************************************************/


#include "QwSpriteField.h"
#include "QwPublicList.cpp"
#include "QwCluster.h"
#include <qpainter.h>
#include <qimage.h>
#include <qwidget.h>
#include <qfile.h>

typedef QwSpriteFieldGraphic* GraphicPtr;
typedef QwPublicList<GraphicPtr> GraphicList;
template class QwPublicList<GraphicPtr>;
template void sort(QwPublicList<QwSpriteFieldGraphic *> *&, int (*)(QwSpriteFieldGraphic *const &, QwSpriteFieldGraphic *const &), bool);
template QwPublicList<QwSpriteFieldGraphic *> remove(QwSpriteFieldGraphic *, QwPublicList<QwSpriteFieldGraphic *> *&);

static int compareZ(const GraphicPtr& a, const GraphicPtr& b)
{
	// Order same-z objects by identity.
	if (a->z()==b->z())
		return (long)a - (long)b;

	return a->z() - b->z();
}


static int compareGraphics(const void* a, const void* b)
{
	const QwSpriteFieldGraphic* aa=*(QwSpriteFieldGraphic**)a;
	const QwSpriteFieldGraphic* bb=*(QwSpriteFieldGraphic**)b;

	// Order same-z objects by identity.
	if (aa->z()==bb->z())
		return (long)aa > (long)bb;

	return aa->z() > bb->z();
}


class QwChunkRec {
public:
	QwChunkRec() : head(0), changed(true) { }
	// Other code assumes lists are not deleted.  Assignment is also
	// done on ChunkRecs.  So don't add that sort of thing here.

	GraphicList* head;
	bool changed;
};




/*!
\class QwSpriteField QwSpriteField.h
\brief A QwSpriteField is a 2D graphic area upon which QwSpriteFieldGraphic objects are drawn.

The QwSpriteField and related classes (primarily QwSpriteFieldView and
QwSprite, but also the other sprite abstractions)
provide for multiple concurrent views of a 2D area
containing moving graphical objects.  

The QwSpriteField is also an indexing mechanism to the sprites it contains,
providing 2D-area-based iteration and pixelwise collision detection.

Most of the methods in QwSpriteField are mainly used by the existing
QwSpriteFieldGraphic classes, so will only be interesting if you intend
to add a new type of QwSpriteFieldGraphic class.  The methods of primary
interest to the typical user are, in approximate
order of decreasing usefulness:

<ul>
      <li> \link QwSpriteField QwSpriteField \endlink (int w, int h, int chunksize=16, int maxclusters=20) 
      <li> void \link update update \endlink () 
      <li> int \link width width \endlink () const 
      <li> int \link height height \endlink () const 
      <li> Pix \link topAt topAt \endlink (int x, int y) 
      <li> Pix \link lookIn lookIn \endlink (int x, int y, int w, int h) 
      <li> QwSpriteFieldGraphic* \link at at \endlink (Pix) const 
      <li> bool \link exact exact \endlink (Pix) const 
      <li> void \link next next \endlink (Pix&) 
      <li> void \link end end \endlink (Pix&) 
      <li> void \link protectFromChange protectFromChange \endlink (Pix) 
      <li> virtual void \link drawBackground drawBackground \endlink (QPainter&, const QRect& area) 
      <li> virtual void \link drawForeground drawForeground \endlink (QPainter&, const QRect& area) 
</ul>

This class provides for very high redraw efficiency.  The properties
of the mechanism are:

<ul>
	<li>There is no size or number limitation on sprites (except memory).
	<li>Sprites are fairly light-weight (small in memory size).
	<li>VirtualSprites are very light-weight (small in memory size).
	<li>Collision detection is efficient.
	<li>Finding sprites in an area is efficient.
	<li>Only moving sprites are redrawn.
	<li>The number of unmoving sprites on the field has little effect.
	<li>Redraws can be batched (see \link update update\endlink)
	<li>Only visible areas (those in the visible part of a QwSpriteFieldView) are redrawn.
</ul>

For example, it is quite feasible to have \e thousands of sprites used to
create background scenery, and even animate pieces of that scenery occasionally.
It is less feasible to have views where the entire scene is in continous
motion, such as a continously scrolling background.  You will be suprised
how little animation is required to make a scene look `alive'.

The various sprite abstractions are:

<ul>
	<dt><b>QwSprite</b>
		<dd>A sprite with a position given by int coordinates.
	<dt><b>QwRealSprite</b>
		<dd>A sprite with a position given by double coordinates.
	<dt><b>QwPositionedSprite&lt;COORD&gt;</b>
		<dd>A sprite with a position given by COORD coordinates.  This
			is a template class.
	<dt><b>QwVirtualSprite</b>
		<dd>A sprite with a position determined dynamically.
</ul>

QwVirtualSprite is the base class, from which is derived QwPositionedSprite,
and from that is derived the QwSprite and QwRealSprite classes.  This hierarchy
provides you with multiple points at which to adapt to your requirements.

For the purpose of simplified description, we use the term sprite to refer
to all these and derived classes.
*/

/*!
Returns the chunk at a chunk position.
*/
QwChunkRec& QwSpriteField::chunk(int i, int j)
{ return chunks[i+chwidth*j]; }

/*!
Returns the chunk at a pixel position.
*/
QwChunkRec& QwSpriteField::chunkContaining(int x, int y)
{ return chunk(x/chunksize,y/chunksize); }

/*!
Returns the sorted list of chunks at the given chunk.  You should
not need this method - it is used internally for collision detection.
\sa QwSpriteField::topAt(int,int)
*/
void* QwSpriteField::listAtChunkTopFirst(int i, int j)
{
	if (i>=0 && i<chwidth && j>=0 && j<chheight) {
		::sort(chunk(i,j).head,compareZ,true);
		return (void*)chunk(i,j).head;
	} else {
		return 0;
	}
}

/*!
\fn void QwSpriteField::setPositionPrecision(int downshifts)

The precision of QwSpriteFieldGraphic positions (and collisions - see sprite) can
be set.  A positive precision means world co-ordinates will be
bit-shifted that many places higher.  So, if the precision is 1,
world co-ordinate (50,100) is the pixel co-ordinate (25,50); for
precision 3, (50,100) is pixel (6,12).  Negative positions are
also allowed, and they imply that world-coordinates are \e lower
resolution than pixel co-ordinates. IF ANYONE FINDS A USE FOR NEGATIVES,
LET ME KNOW, OR I MIGHT REMOVE IT.

By default, pixel and world coordinates are the same.

*/

/*!
\fn int QwSpriteField::positionPrecision()
Returns the position precision for all coordinates in the QwSpriteField.
\sa setPositionPrecision(int)
*/

/*!
\fn int QwSpriteField::world_to_x(int i)
Convert a coordinate from world to pixel coordinates.
\sa setPositionPrecision(int)
*/

/*!
\fn int QwSpriteField::x_to_world(int i)
Convert a coordinate from pixel to world coordinates.
\sa setPositionPrecision(int)
*/

/*!
Constructs a QwSpriteField with size \c w wide and \c h high, measured
in world coordinates (by default, world coordinates equal pixel coordinates).

The values \c maxclusters and \c chunksize effect the grouping heuristics
used when redrawing the QwSpriteField.

   \c chunksize is the size of square chunk used to break up the
     QwSpriteField into area to be considered for redrawing.  It
     should be about the average size of graphics in the QwSpriteField.
     Chunks too small increase the amount of calculation required
     when drawing.  Chunks too large increase the amount of drawing
     that is needed.

   \c maxclusters is the number of rectangular groups of chunks that
     will be separately drawn.  If the QwSpriteField has a large number
     of small, dispersed graphics, this should be about that number.
     If the QwSpriteField is fully active, use a small number (say, 1).
     The more clusters the slower the redraw, but also the bigger
     clusters are the slower the redraw, so a balance is needed.
*/
QwSpriteField::QwSpriteField(int w, int h, int chunksze, int maxclust) :
	awidth(w),aheight(h),
	chunksize(chunksze),
	maxclusters(maxclust),
	chwidth((w+chunksize-1)/chunksize),
	chheight((h+chunksize-1)/chunksize),
	chunks(new QwChunkRec[chwidth*chheight])
{
	QwSpriteFieldGraphic::setCurrentSpriteField(this);
}

/*!
Destruct the field.  Does nothing.  Does \e not destroy QwSpriteFieldGraphic objects in
the field.
*/
QwSpriteField::~QwSpriteField()
{
}

/*!
Change the size of the QwSpriteField.
*/
void QwSpriteField::resize(int w, int h)
{
	QList<QwSpriteFieldGraphic> hidden;
	for (QwSpriteFieldGraphic* graphic=graphicList.first(); graphic != 0; graphic=graphicList.next()) {
		if (graphic->visible()) {
			graphic->hide();
			hidden.append(graphic);
		}
	}

	int nchwidth=(w+chunksize-1)/chunksize;
	int nchheight=(h+chunksize-1)/chunksize;

	QwChunkRec* newchunks = new QwChunkRec[nchwidth*nchheight];

	// Commit the new values.
	//
	awidth=w;
	aheight=h;
	chwidth=nchwidth;
	chheight=nchheight;
	delete chunks;
	chunks=newchunks;

	for (QwSpriteFieldGraphic* graphic=hidden.first(); graphic != 0; graphic=hidden.next()) {
		graphic->show();
	}
}

/*!
\fn int QwSpriteField::width() const
Returns the width of the sprite field, in world coordinates.
*/

/*!
\fn int QwSpriteField::height() const
Returns the height of the sprite field, in world coordinates.
*/

/*!
\fn int QwSpriteField::chunkSize() const
Returns the chunk size of the sprite field as set at construction.
\sa QwSpriteField::QwSpriteField(...)
*/

/*!
\fn bool QwSpriteField::sameChunk(int x1, int y1, int x2, int y2) const
Tells if the points (x1,y1) and (x2,y2) are within the same chunk.
*/

/*!
This method is called for all updates of the QwSpriteField.  It renders
any background graphics.  The default is to simply clear it in the
background colour of the QWidget.

Note that you should not put \e dynamic graphics on this background.
If the graphics for a region change, call forceRedraw(const QRect&).
Such a change should be done only very occasionally, as the redraw
is inefficient.  See the main notes for this class for a discussion
of redraw efficiency.

\sa forceRedraw(QRect&)
*/
void QwSpriteField::drawBackground(QPainter& painter, const QRect& area)
{
	painter.eraseRect(area);
}

/*!
This method is called for all updates of the QwSpriteField.  It renders
any foreground graphics.  In general, it is more efficient to use additional
QwSpriteFieldGraphic objects on the QwSpriteField rather than adding rendering
at this point, as this method is <em>always</em> called, whereas
QwSpriteFieldGraphic objects are only redrawn if they are in the area of change.

The same warnings regarding change apply to this method
as given in drawBackground(QPainter&, const QRect&).

The default is to do nothing.

\sa forceRedraw(QRect&)
*/
void QwSpriteField::drawForeground(QPainter&, const QRect&)
{
}

/*!
Forces the given area to be redrawn.  This is useful when the foreground
or background are changed.
*/
void QwSpriteField::forceRedraw(const QRect& area)
{
	drawArea(area, false, 0);
}

/*!
(internal)
This method adds an element to the list of QwSpriteFieldGraphic objects
in this QwSpriteField.  The QwSpriteFieldGraphic class calls this, so you
should not need it.
*/
void QwSpriteField::addGraphic(QwSpriteFieldGraphic* graphic)
{
	graphicList.append(graphic);
}

/*!
(internal)
This method removes an element from the list of QwSpriteFieldGraphic objects
in this QwSpriteField.  The QwSpriteFieldGraphic class calls this, so you
should not need it.
*/
void QwSpriteField::removeGraphic(QwSpriteFieldGraphic* graphic)
{
	graphicList.removeRef(graphic);
}

/*!
(internal)
This method adds an element to the list of QwSpriteFieldView objects
viewing this QwSpriteField.  The QwSpriteFieldView class calls this, so you
should not need it.
*/
void QwSpriteField::addView(QwSpriteFieldView* view)
{
	viewList.append(view);
}

/*!
(internal)
This method removes an element from the list of QwSpriteFieldView objects
viewing this QwSpriteField.  The QwSpriteFieldView class calls this, so you
should not need it.
*/
void QwSpriteField::removeView(QwSpriteFieldView* view)
{
	viewList.removeRef(view);
}

/*!
This method causes all QwSpriteFieldView objects currently viewing this
QwSpriteField to be refreshed in all areas of the QwSpriteField which have
changed since the last call to this method.

In a continuously animated QwSpriteField, this method should be called at
a regular rate.  The best way to accomplish this is to use the QTimer
class of the Qt toolkit, or the startTimer method of QObject.  It would
not be useful to call this method excessively (eg. whenever a sprite
is moved) as the efficiency of the system is derived from a clustering
and bulk-update mechanism.
*/
void QwSpriteField::update()
{
	QwClusterizer clusterizer(viewList.count());

	for (QwSpriteFieldView* view=viewList.first(); view != 0; view=viewList.next()) {
		QRect area=view->viewArea();
		if (area.width()>0 && area.height()>0) {
			clusterizer.add(area);
		}
	}

	for (int i=0; i<clusterizer.clusters(); i++) {
		drawArea(clusterizer[i],true,0);
	}
}

/*!
(internal)
This method causes a specific area of a QwSpriteField to be completely
redrawn in a given QwSpriteFieldView.  This is called by QwSpriteFieldView
to effect repaint events.
*/
void QwSpriteField::updateInView(QwSpriteFieldView* view, const QRect& area)
{
	drawArea(area,false,view);
}


/*!
(private)
Redraw a given area of the QwSpriteField.

If only_changes then only changes to the area are redrawn.

If one_view then only one view is updated, otherwise all are.
*/
void QwSpriteField::drawArea(const QRect& inarea, bool only_changes, QwSpriteFieldView* one_view)
{
	QRect area=inarea.intersect(QRect(0,0,width(),height()));

	if (only_changes) {
		QwClusterizer clusters(maxclusters);

		int mx=(area.x()+area.width()+chunksize)/chunksize;
		int my=(area.y()+area.height()+chunksize)/chunksize;
		if (mx>chwidth) mx=chwidth;
		if (my>chheight) my=chheight;

		for (int x=area.x()/chunksize; x<mx; x++) {
			for (int y=area.y()/chunksize; y<my; y++) {
				QwChunkRec& ch=chunk(x,y);
				if (ch.changed) {
					clusters.add(x,y);
					ch.changed=false;
				}
			}
		}

		for (int i=0; i<clusters.clusters(); i++) {
			QRect elarea=clusters[i];
			elarea.setRect(
				elarea.left()*chunksize,
				elarea.top()*chunksize,
				elarea.width()*chunksize,
				elarea.height()*chunksize
			);
			drawArea(elarea,false,one_view);
		}
	} else {
		QPixmap offscr(world_to_x(area.width()),world_to_x(area.height()));

		QPainter painter;

		painter.begin(&offscr);

		painter.setViewport(-area.x(),-area.y(),area.width(),area.height());
		drawBackground(painter,area);

		// NON-REENTRANT.  Make this a member of QwSpriteField, or always reallocate
		// it in order to make this function reentrant.
		static int max=16;
		static QwSpriteFieldGraphic* *allvisible=new QwSpriteFieldGraphic*[max];

		int count=0;
		int lx=area.x()/chunksize;
		int ly=area.y()/chunksize;
		int mx=(area.x()+area.width()+chunksize)/chunksize;
		int my=(area.y()+area.height()+chunksize)/chunksize;
		if (mx>chwidth) mx=chwidth;
		if (my>chheight) my=chheight;

		for (int x=lx; x<mx; x++) {
			for (int y=ly; y<my; y++) {
				GraphicList* cursor;

				// Only reset change if all views updating, and
				// wholy within area. (conservative:  ignore entire boundary)
				//
				// Disable this to help debugging.
				//
				if (!one_view) {
					if (x>lx && x<mx-1 && y>ly && y<my-1)
						chunk(x,y).changed=false;
				}

				for (cursor=chunk(x,y).head; cursor; cursor=cursor->next) {
					if (count == max) {
						max*=2;
						QwSpriteFieldGraphic* *newallvisible=new QwSpriteFieldGraphic*[max];
						for (int i=0; i<count; i++) {
							newallvisible[i]=allvisible[i];
						}
						allvisible=newallvisible;
					}
					allvisible[count++]=cursor->element;
				}
			}
		}
		qsort(allvisible, count, sizeof(*allvisible), compareGraphics);

		QwSpriteFieldGraphic* prev=0;
		int drawn=0;
		for (int i=0; i<count; i++) {
			QwSpriteFieldGraphic *g=allvisible[i];
			if (g!=prev) {
				g->draw(painter);
				prev=g;
				drawn++;
			}
		}

		drawForeground(painter,area);

		painter.end();

		QPainter onview;

		if (one_view) {
			onview.begin(one_view);
			onview.drawPixmap(area.topLeft(), offscr);
			onview.end();
		} else {
			for (QWidget* view=viewList.first(); view != 0; view=viewList.next()) {
				onview.begin(view);
				onview.drawPixmap(area.topLeft(), offscr);
				onview.end();
			}
		}
	}
}

/*!
This method to informs the QwSpriteField that a given chunk is
`dirty' and needs to be redrawn in the next Update.

(x,y) is a chunk location.

The sprite classes call this.  Any new derived class
of QwSpriteFieldGraphic must do so too.  SetChangedChunkContaining can be used
instead.
*/
void QwSpriteField::setChangedChunk(int x, int y)
{
	if (x>=0 && x<chwidth && y>=0 && y<chheight) {
		QwChunkRec& ch=chunk(x,y);
		ch.changed=true;
	}
}

/*!
This method to informs the QwSpriteField that the chunk containging
a given pixel is `dirty' and needs to be redrawn in the next Update.

(x,y) is a pixel location.

The sprite classes call this.  Any new derived class
of QwSpriteFieldGraphic must do so too. SetChangedChunk can be used instead.
*/
void QwSpriteField::setChangedChunkContaining(int x, int y)
{
	if (x>=0 && x<width() && y>=0 && y<height()) {
		QwChunkRec& chunk=chunkContaining(x,y);
		chunk.changed=true;
	}
}

/*!
This method adds a QwSpriteFieldGraphic to the list of those which need to
be drawn if the given chunk is redrawn.  Like SetChangedChunk
and SetChangedChunkContaining, this method marks the chunk as `dirty'.
*/
void QwSpriteField::addGraphicToChunk(QwSpriteFieldGraphic* g, int x, int y)
{
	if (x>=0 && x<chwidth && y>=0 && y<chheight) {
		QwChunkRec& ch=chunk(x,y);
		ch.head=new GraphicList(g,ch.head);
		ch.changed=true;
	}
}

/*!
This method removes a QwSpriteFieldGraphic from the list of those which need to
be drawn if the given chunk is redrawn.  Like SetChangedChunk
and SetChangedChunkContaining, this method marks the chunk as `dirty'.
*/
void QwSpriteField::removeGraphicFromChunk(QwSpriteFieldGraphic* g, int x, int y)
{
	if (x>=0 && x<chwidth && y>=0 && y<chheight) {
		QwChunkRec& ch=chunk(x,y);
		delete remove(g,ch.head);
		ch.changed=true;
	}
}


/*!
This method adds a QwSpriteFieldGraphic to the list of those which need to
be drawn if the chunk containging the given pixel is redrawn.
Like SetChangedChunk and SetChangedChunkContaining, this method
marks the chunk as `dirty'.
*/
void QwSpriteField::addGraphicToChunkContaining(QwSpriteFieldGraphic* g, int x, int y)
{
	if (x>=0 && x<width() && y>=0 && y<height()) {
		QwChunkRec& ch=chunkContaining(x,y);
		ch.head=new GraphicList(g,ch.head);
		ch.changed=true;
	}
}

/*!
This method removes a QwSpriteFieldGraphic from the list of those which need to
be drawn if the chunk containging the given pixel is redrawn.
Like SetChangedChunk and SetChangedChunkContaining, this method
marks the chunk as `dirty'.
*/
void QwSpriteField::removeGraphicFromChunkContaining(QwSpriteFieldGraphic* g, int x, int y)
{
	if (x>=0 && x<width() && y>=0 && y<height()) {
		QwChunkRec& ch=chunkContaining(x,y);
		delete remove(g,ch.head);
		ch.changed=true;
	}
}

/*
(internal)
A QwSpriteFieldIterator is used for the actual implementation of
the Pix iterator.  It stores all the information needed to
iterate over the various queries.

\sa QwSpriteField::topAt(int,int)
*/
class QwSpriteFieldIterator {
private:
	QwSpriteFieldIterator(int ax, int ay, int aw, int ah, QwSpriteField* spritefield, QImage* coll_mask) :
		x(ax),y(ay),
		w(aw),h(ah),
		collision_mask(coll_mask),
		i(x/spritefield->chunkSize()),
		j(y/spritefield->chunkSize()),
		x1(i), y1(j),
		x2((x+w-1)/spritefield->chunkSize()),
		y2((y+h-1)/spritefield->chunkSize()),
		list((GraphicList*)spritefield->listAtChunkTopFirst(i,j)),
		is_copy(false)
	{
	}

public:
	static QwSpriteFieldIterator* make(int ax, int ay, int aw, int ah, QwSpriteField* spritefield, QImage* coll_mask)
	{
		QwSpriteFieldIterator* result=new QwSpriteFieldIterator(ax,ay,aw,ah,
			spritefield, coll_mask);

		if (result->empty()) {
			result=result->next(spritefield);
		}

		return result;
	}

	static QwSpriteFieldIterator* make(int sx, int sy, QwSpritePixmap* im, QwSpriteField* spritefield)
	{
		return make(
			QwVirtualSprite::col_to_world(QwVirtualSprite::world_to_col(sx)-im->colhotx),
			QwVirtualSprite::col_to_world(QwVirtualSprite::world_to_col(sy)-im->colhoty),
			QwVirtualSprite::col_to_world(im->colw),
			QwVirtualSprite::col_to_world(im->colh),
			spritefield, im->collision_mask);
	}

	~QwSpriteFieldIterator()
	{
		if (is_copy) delete list;
	}

	QwSpriteFieldIterator* next(QwSpriteField* spritefield)
	{
		if (!empty()) {
			nextElement();
			if (!empty()) {
				return this;
			}
		}

		while (empty()) {
			i++;

			if (i <= x2) {
				newList((GraphicList*)spritefield->listAtChunkTopFirst(i,j));
			} else {
				i=x1; j++;
				if (j <= y2) {
					newList((GraphicList*)spritefield->listAtChunkTopFirst(i,j));
				} else {
					delete this;
					return 0;
				}
			}
		}

		return this;
	}

	QwSpriteFieldGraphic* element()
	{
		return list->element;
	}

	void protectFromChange()
	{
		if (!is_copy) {
			is_copy=true;
			list=list->copy();
		}
	}

	const int x, y, w, h;
	QImage* collision_mask;

private:
	void newList(GraphicList* nl)
	{
		if (is_copy) {
			if (list) delete list;
			list=nl->copy();
		} else {
			list=nl;
		}
	}

	void nextElement()
	{
		if (is_copy) {
			GraphicList* n=list->next;
			list->next=0;
			delete list;
			list=n;
		} else {
			list=list->next;
		}
	}

	bool empty()
	{
		return !list;
	}

	int i, j;
	const int x1, y1, x2, y2;

	GraphicList* list;
	bool is_copy; // indicates that list is a copy, not spritefield list.
};

/*!
QwSpriteFieldGraphic objects at a <em>point</em> can be traversed from top
to bottom using:

\code
    for (Pix p=topAt(x,y); p; next(p)) {
        QwSpriteFieldGraphic* the_graphic = at(p);
        if (you_are_interested_in_collisions_with(the_graphic) && exact(p)) {
			// Collision!
			...
		}
    }
\endcode

This traverses <em>at least</em> all the graphics at pixel position (x,y).

The QwSpriteFieldGraphic returned by at(Pix) is very approximate, but is found
very efficiently.  It should be used as a first-cut to ignore
QwSpriteFieldGraphic objects you are not interested in.

exact(Pix) should be used to further
examine the QwSpriteFieldGraphic, as depicted above.

During the traversal, the QwSpriteField must not be modified unless
protectFromChange(Pix) is first called on the iterator.  This is for
efficiency reasons (protecting the list of hits requires extra work).

\warning Currently, the collision model is not quite correct at this
	level of abstraction, because it cannot check collision exactness
	beyond the rectangular level.  This will change in futures versions.
	For now, you should perhaps use the QwVirtualSprite::neighbourhood(...)
	methods instead.

\sa QwSpriteFieldGraphic::rtti()
*/
Pix QwSpriteField::topAt(int x, int y)
{
	QwSpriteFieldIterator* iterator=QwSpriteFieldIterator::make(x,y,1,1, this, 0);
	return (Pix)iterator;
}

/*!
QwSpriteFieldGraphic objects in an <em>area</em> can be traversed using:

\code
   for (Pix p=lookIn(x,y,w,h); p; next(p)) {
        QwSpriteFieldGraphic* the_graphic = at(p);
        ...
    }
\endcode

This traverses \e at \e least all the graphics in the given rectangle
\e at \e least once.

\sa topAt(int,int) at(Pix)
*/
Pix QwSpriteField::lookIn(int x, int y, int w, int h)
{
	QwSpriteFieldIterator* iterator=QwSpriteFieldIterator::make(x,y,w,h, this, 0);
	return (Pix)iterator;
}

/*!
Look to the next QwSpriteFieldGraphic in the traversal.

\sa topAt(int,int) lookIn(int,int,int,int)
*/
void QwSpriteField::next(Pix& p)
{
	QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;

	iterator=iterator->next(this);

	p=(Pix)iterator;
}

/*!
This should be called if you want to terminate a traversal without
looking at all results.  It frees memory used for the iteration.
*/
void QwSpriteField::end(Pix& p)
{
	if (p) {
		delete (QwSpriteFieldIterator*)p;
		p=0;
	}
}

/*!
Returns the QwSpriteFieldGraphic for a given point in a traversal.

\sa topAt(int,int) lookIn(int,int,int,int)
*/
QwSpriteFieldGraphic* QwSpriteField::at(Pix p) const
{
	if (p) {
		QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;
		return iterator->element();
	} else {
		return 0;
	}
}

/*!
Test to see if the QwSpriteFieldGraphic which at(Pix) indicates is
in a traversal truly \e is in that traversal.

The QwSpriteFieldGraphic returned by at(Pix) is very approximate, but is found
very efficiently.  It should be used as a first-cut to ignore
QwSpriteFieldGraphic objects you are not interested in.
*/
bool QwSpriteField::exact(Pix p) const
{
	QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;
	QRect area(iterator->x,iterator->y,iterator->w,iterator->h);
	return iterator->element()->at(area);
}

/*!
During the traversal, the QwSpriteField must not be modified.
If you wish to modify it, call this method on the
Pix.  This will allow the QwSpriteField to be subsequently
modified without damaging the list of hits.
*/
void QwSpriteField::protectFromChange(Pix p)
{
	if (p) {
		QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;
		iterator->protectFromChange();
	}
}


unsigned int QwSpriteField::posprec=0;

/*!
\class QwImageSpriteField QwSpriteField.h
\brief A QwSpriteField with a background image.

Although it is not very useful to provide a complex background drawing
method for a QwSpriteField (since redraw efficiency is achieved by sprites,
not static background graphics), if your application requires only a
simple repeating background image, this class will suffice.

This class is also intended as documentation-by-example for the
QwSpriteField::drawBackground method.
*/

/*!
Construct an QwImageSpriteField which uses the given image file
as the background image.  Any Qt-supported image format may be used.
*/
QwImageSpriteField::QwImageSpriteField(const char* filename, int w, int h, int chunksize, int maxclusters) :
	QwSpriteField(w, h, maxclusters, chunksize)
{
	if (!image.load(filename)) {
		fprintf(stderr,"QwImageSpriteField::QwImageSpriteField - Failed to read %s\n",filename);
		exit(1);
	}
}

/*!
Does nothing.
*/
QwImageSpriteField::~QwImageSpriteField()
{
}

/*!
Reimplements QwSpriteField::drawBackground to draw the image
as the background.
*/
void QwImageSpriteField::drawBackground(QPainter& painter, const QRect& area)
{
	for (int x=area.x()/image.width();
		x<(area.x()+area.width()+image.width()-1)/image.width(); x++)
	{
		for (int y=area.y()/image.height();
			y<(area.y()+area.height()+image.height()-1)/image.height(); y++)
		{
			painter.drawPixmap(x*image.width(), y*image.height(),image);
		}
	}
}


/*!
\class QwSpriteFieldGraphic QwSpriteField.h
\brief An abstract graphic object on a QwSpriteField.

This class will primarily be of interest to those wanting to
add graphic objects other than the sprite object already defined.

In most graphic rendering systems, graphic objects are considered to
have a bounding \e rectangle, and redraw optimization is based on this
simplification.  This simplistic view is not used by QwSpriteField.  A
QwSpriteField considers itself to contain numerous graphic objects, each
of which covers certain `chunks' in the QwSpriteField.  For graphic
objects with a rectangular bias, this has only a minor effect on
redraw efficiency (although still a major effect on collision
detection and other area indexing).  But for other shapes, such as
lines, the tighter area-bound made possible by chunks can provide
improvements.

Whenever a QwSpriteFieldGraphic moves, it must add and remove itself
from the chunks of the QwSpriteField upon which it moves.  If the
QwSpriteFieldGraphic is much smaller than the chunk size of the
QwSpriteField, it will usually be in one, sometimes 2, and occasionally
3 or 4 chunks.  If the QwSpriteFieldGraphic is larger than the
chunk size of the QwSpriteField, it will span a number of chunks.
Clearly there is a trade-off between tight bounds and excessive
numbers of chunks a QwSpriteFieldGraphic will have to add and remove
itself from.

Note that a QwSpriteFieldGraphic may be `on' a QwSpriteField even if it's
coordinates place it far off the edge of the area of the QwSpriteField.
*/ 

/*!
Construct a QwSpriteFieldGraphic on the current spritefield.

\sa setCurrentSpriteField(QwSpriteField*) setSpriteField(QwSpriteField*)
*/
QwSpriteFieldGraphic::QwSpriteFieldGraphic() :
	spritefield(current_spritefield),
	vis(true)
{
	if (spritefield) spritefield->addGraphic(this);
}

/*!
Destruct a QwSpriteFieldGraphic.  It is removed from its spritefield.
*/
QwSpriteFieldGraphic::~QwSpriteFieldGraphic()
{
	if (spritefield) spritefield->removeGraphic(this);
}

/*!
\fn int QwSpriteFieldGraphic::z() const

This abstract method should return the z of the graphic,
which is used for visual order:  higher z sprites obscure
lower-z ones.
*/

/*!
\fn void QwSpriteFieldGraphic::draw(QPainter&)

This abstract method should draw the the graphic on the given QPainter.
*/

/*!
QwSpriteFieldGraphics may be created on a `current' QwSpriteField, which must be
set prior to any graphic creation.  QwSpriteField objects set this,
so the most recently created QwSpriteField will get new QwSpriteFieldGraphics.

This notion of `currency' makes the most common case easy to use - that
of a single instance of QwSpriteField in the application - rather than
having to pass around a pointer to a QwSpriteField.
*/
void QwSpriteFieldGraphic::setCurrentSpriteField(QwSpriteField* pf)
{
	current_spritefield=pf;
}

/*!
Set the QwSpriteField upon which the QwSpriteFieldGraphic is to be drawn.
Initially this will be the current spritefield.

\sa setCurrentSpriteField(QwSpriteField*)
*/
void QwSpriteFieldGraphic::setSpriteField(QwSpriteField* pf)
{
	bool v=visible();
	visible(false);
	if (spritefield) spritefield->removeGraphic(this);
	spritefield=pf;
	if (spritefield) spritefield->addGraphic(this);
	visible(v);
}

/*!
Alias for visible(true).
*/
void QwSpriteFieldGraphic::show()
{
	visible(true);
}

/*!
Alias for visible(false).
*/
void QwSpriteFieldGraphic::hide()
{
	visible(false);
}

/*!
Calls makeVisible(bool) if necessary to set the
visibility of the QwSpriteFieldGraphic to the given state.
*/
void QwSpriteFieldGraphic::visible(bool yes)
{
	if (vis!=yes) {
		if (yes) {
			vis=yes;
			makeVisible(yes);
		} else {
			makeVisible(yes);
			vis=yes;
		}
	}
}

/*!
Returns true if the QwSpriteFieldGraphic is visible.  This does <em>not</em>
mean the QwSpriteFieldGraphic is currently in a view, merely that if a view
was showing the area where the QwSpriteFieldGraphic is, and the graphic
was not obscured by graphics at a higher z, it would be visible.
*/
bool QwSpriteFieldGraphic::visible() const
{
	return vis;
}

QwSpriteField* QwSpriteFieldGraphic::current_spritefield=0;

/*!
\fn bool QwSpriteFieldGraphic::at(int x, int y) const
Should return true if the graphic includes the given pixel position.
*/

/*!
\fn bool QwSpriteFieldGraphic::at(const QRect& rect) const
true if the graphic intersects with the given area.  This need not be
perfectly accurate.
*/

/*!
\fn int QwSpriteFieldGraphic::world_to_x(int i)

Same as QwSpriteField::world_to_x(int i)
*/
/*!
\fn int QwSpriteFieldGraphic::x_to_world(int i)

Same as QwSpriteField::x_to_world(int i)
*/

/*!
QwSpriteFieldGraphic derivatives can give pixel-accuracy for collision
detection by overriding this method.  By default, it simply returns
at(const QRect&) on the same area.
*/
bool QwSpriteFieldGraphic::at(const QImage*, const QRect& yourarea) const
{
	return at(yourarea);
}

/*!
The default implementation of makeVisible(bool) does nothing.

Note: derived classes should not be in any chunks if they are not visible,
so during this method, they should remove themselves if they are becoming
invisible, and add themselves if they are becoming visible.
*/
void QwSpriteFieldGraphic::makeVisible(bool)
{
}

/*!
\class QwSpritePixmap QwSpriteField.h
\brief A QwSpritePixmap is a sprite frame image

Note that QwSpritePixmap should be considered an internal class at
the current time.  This will change (as will the interface) once
alpha and description support for QImage is clarified.

\sa QwSpritePixmapSequence QwVirtualSprite QwSprite
*/

/*!
Construct a QwSpritePixmap from two image files.

The QwSpritePixmap is a masked QPixmap used internally by
sprite classes.

The dataname file must be a PPM file of the form:

\code
P6
# HOTSPOT x y
...
\endcode

That is, it must have an additional comment which gives the (x,y)
coordinate of the `hotspot' of the image.

The `hotspot' position defines the origin pixel in the image
For example, if the hotspot is (10,5), it will be displayed
drawn 10 pixels to the left of and 5 pixels above the actual
(x,y) coordinate of the sprite.

The maskname can be any monochrome image format, such as PBM.
No special comments in the file are needed or recognized.
*/
QwSpritePixmap::QwSpritePixmap(const char* dataname, const char* maskname) :
	hotx(0),hoty(0),
	collision_mask(0),
	colhotx(0),colhoty(0)
{
	{
		QFile file(dataname);
		if (file.open(IO_ReadOnly)) {
			char line[128];
			file.readLine(line,128); // Skip "P6"/"P3" line
			file.readLine(line,128);

			while (line[0]=='#') {
				// Comment line - see if it has additional parameters
				if (0==strncmp(line,"# HOTSPOT ",10)) {
					sscanf(line+10,"%d %d",&hotx,&hoty);
					colhotx=hotx;
					colhoty=hoty;
				}
				file.readLine(line,128);
			}
		}
	}

	if (!load(dataname)) {
		fprintf(stderr,"QwSpritePixmap::QwSpritePixmap - Failed to read %s\n",dataname);
		exit(1);
	}

	QImageIO iio;
	iio.setFileName(maskname);
	if (!iio.read()) {
		fprintf(stderr,"QwSpritePixmap::QwSpritePixmap - Failed to read %s\n",maskname);
		exit(1);
	}
	collision_mask=new QImage(iio.image());
	mask.convertFromImage(*collision_mask);

	setMask(mask);

	colw=width();
	colh=height();
}

/*!
Deletes any collision mask.
*/
QwSpritePixmap::~QwSpritePixmap()
{
	delete collision_mask;
}

/*!
\class QwSpritePixmapSequence QwSpriteField.h
\brief A sequence of QwSpritePixmap
to have multiple frames for animation.

A sequence of QwSpritePixmap, for use in allowing sprite objects
to have multiple frames for animation.

QwSpritePixmaps for sprite objects are collected into
QwSpritePixmapSequences, which are
the set of images a sprite will use, indexed by the sprite's
frame.  This allows sprites to simply have animated forms.
*/

/*!
Construct a QwSpritePixmapSequence 

The framecount parameter sets the number of frames to be
loaded for this image.  The filenames should be printf-style
strings such as "foo%03d.ppm" or "animdir/%4d.pbm".  The actual
filenames are formed by sprintf-ing a string with each integer
from 0 to framecount-1, eg. foo000.ppm, foo001.ppm, foo002.ppm, etc.
*/
QwSpritePixmapSequence::QwSpritePixmapSequence(const char* datafilenamepattern,
		const char* maskfilenamepattern, int fc) :
	framecount(fc),
	img(new QwSpritePixmap*[fc])
{
	for (int i=0; i<framecount; i++) {
		char data[1024],mask[1024];
		sprintf(data,datafilenamepattern,i);
		sprintf(mask,maskfilenamepattern,i);
		img[i]=new QwSpritePixmap(data,mask);
	}
}

QwSpritePixmapSequence::~QwSpritePixmapSequence()
{
  int i;
  for (i=0;i<framecount;i++) delete img[i];
  delete img;
}

/*!
Constructor failure check.  Currently not implemented.
Exceptions would be a better solution.
*/
int QwSpritePixmapSequence::operator!()
{
	return 0;
}

/*!
\fn QwSpritePixmap* QwSpritePixmapSequence::image(int i) const
Returns the i-th pixmap in the sequence.
*/

/*!
\fn int QwSpritePixmapSequence::frameCount() const
Returns the length of the sequence.
*/

/*!
When testing sprite collision detection with QwSpriteField::exact(Pix), 
the default is to use the image mask of the sprite.  By using
readCollisionMasks(const char*), an alternate mask can be used.
Also, by using QwVirtualSprite::setPixelCollisionPrecision(int),
the resolution of the collision mask can be different to the
resolution of the pixel coordinates of the sprites.

The filename is a printf-style string, as per the constructor
for QwSpritePixmapSequence.

The image must be a PBM file, with a HOTSPOT comment as described
in QwSpritePixmap::QwSpritePixmap(const char*, const char*).

\sa QwVirtualSprite::setPixelCollisionPrecision(int).
*/
void QwSpritePixmapSequence::readCollisionMasks(const char* fname)
{
	for (int i=0; i<framecount; i++) {
		char filename[1024];
		sprintf(filename,fname,i);

		{
			QFile file(filename);
			if (file.open(IO_ReadOnly)) {
				char line[128];
				file.readLine(line,128); // Skip "P1"/"P4" line
				file.readLine(line,128);

				while (line[0]=='#') {
					// Comment line - see if it has additional parameters
					if (0==strncmp(line,"# HOTSPOT ",10)) {
						sscanf(line+10,"%d %d",&img[i]->colhotx,&img[i]->colhoty);
					}
					file.readLine(line,128);
				}
			}
		}
		delete img[i]->collision_mask;
		QImageIO iio;
		iio.setFileName(filename);
		if (!iio.read()) {
			fprintf(stderr,"QwSpritePixmapSequence::readCollisionMasks - Failed to read %s\n",filename);
			exit(1);
		}
		img[i]->collision_mask=new QImage(iio.image());
	}
}

/*!
\class QwVirtualSprite QwSpriteField.h
\brief A QwSpriteFieldGraphic which renders as a masked image.

VirtualSprites are most of the implementation of sprites, except they
do not define the x(), y(), z() and image() methods.
This allows for user-defined coordinates systems and animation methods.

\sa QwSprite
*/

unsigned int QwVirtualSprite::colprec=0;

/*!
Construct a QwVirtualSprite.  Derived classes should call addToChunks()
in their constructor once x(), y(), and image() are valid.
*/
QwVirtualSprite::QwVirtualSprite()
{
}

/*!
\fn QwSpritePixmap* QwVirtualSprite::image() const
This abstract method should return the pixmap to be drawn
for the sprite.
*/

/*!
\fn int QwVirtualSprite::x() const
This abstract method should return the horizontal location, measured
from the left edge of the QwSpriteField, at
which to draw the pixmap of the sprite.  Note that the hotspot of the
pixmap will be taken into account.
*/
/*!
\fn int QwVirtualSprite::y() const
This abstract method should return the vertical location, measured
from the top edge of the QwSpriteField, at
which to draw the pixmap of the sprite.  Note that the hotspot of the
pixmap will be taken into account.
*/

/*!
\fn int QwVirtualSprite::world_to_col(int i)
Convert a coordinate from world to collision coordinates.
\sa setPixelCollisionPrecision(int) QwSpriteField::setPositionPrecision
*/
/*!
\fn int QwVirtualSprite::col_to_world(int i)
Convert a coordinate from collision to world coordinates.
\sa setPixelCollisionPrecision(int) QwSpriteField::setPositionPrecision
*/

/*!
Adds/removes the sprite from the QwSpriteField chunks it covers, as
required by QwSpriteFieldGraphic.  This, addToChunks, and the removeFromChunks
method may be moved up to QwSpriteFieldGraphic.  Either way, this or
the other two would have to be virtual.
*/
void QwVirtualSprite::makeVisible(bool yes)
{
	if (yes) {
		addToChunks();
	} else {
		removeFromChunks();
	}
}

/*!
The absolute horizontal position of the QwVirtualSprite.  This is
the pixel position of the left edge of the image, as it takes into
account the hotspot.
*/
int QwVirtualSprite::absX() const
{
	return x()-image()->hotx;
}

/*!
The absolute vertical position of the QwVirtualSprite.  This is
the pixel position of the top edge of the image, as it takes into
account the hotspot.
*/
int QwVirtualSprite::absY() const
{
	return y()-image()->hoty;
}

/*!
The right edge of the sprite image.

\sa absX()
*/
int QwVirtualSprite::absX2() const
{
	return absX()+image()->width()-1;
}

/*!
The bottom edge of the sprite image.

\sa absY()
*/
int QwVirtualSprite::absY2() const
{
	return absY()+image()->height()-1;
}

/*!
The left edge, in collision-detection coordinates.

\sa QwVirtualSprite::setPixelCollisionPrecision(int).
*/
int QwVirtualSprite::absColX() const
{
	return world_to_col(x())-image()->colhotx;
}

/*!
The top edge, in collision-detection coordinates.

\sa QwVirtualSprite::setPixelCollisionPrecision(int).
*/
int QwVirtualSprite::absColY() const
{
	return world_to_col(y())-image()->colhoty;
}

/*!
The right edge, in collision-detection coordinates.

\sa QwVirtualSprite::setPixelCollisionPrecision(int).
*/
int QwVirtualSprite::absColX2() const
{
	return absColX()+image()->colw-1;
}

/*!
The bottom edge, in collision-detection coordinates.

\sa QwVirtualSprite::setPixelCollisionPrecision(int).
*/
int QwVirtualSprite::absColY2() const
{
	return absColY()+image()->colh-1;
}

/*!
Add the sprite to the chunks in its QwSpriteField which it overlaps.

This must be called as the values of x(), y(), and image() change
such that the QwVirtualSprite is removed from chunks it is in,
the values of x(), y(), and image() change, then it is added
back into the then covered chunks in the QwSpriteField.

The QwPositionedSprite and derived classes deals with this automatically,
as must other derived classes of QwVirtualSprite.
*/
void QwVirtualSprite::addToChunks()
{
	if (visible() && spritefield) {
		int chunksize=spritefield->chunkSize();
		for (int j=absY()/chunksize; j<=absY2()/chunksize; j++) {
			for (int i=absX()/chunksize; i<=absX2()/chunksize; i++) {
				spritefield->addGraphicToChunk(this,i,j);
			}
		}
	}
}

/*!
Remove the sprite to the chunks in its QwSpriteField which it overlaps.

\sa addToChunks()
*/
void QwVirtualSprite::removeFromChunks()
{
	if (visible() && spritefield) {
		int chunksize=spritefield->chunkSize();
		for (int j=absY()/chunksize; j<=absY2()/chunksize; j++) {
			for (int i=absX()/chunksize; i<=absX2()/chunksize; i++) {
				spritefield->removeGraphicFromChunk(this,i,j);
			}
		}
	}
}

/*!
Destruct the QwVirtualSprite.  Derived classes must remove the sprite
from any chunks, as this destructor cannot call the virtual methods
required to do so.
*/
QwVirtualSprite::~QwVirtualSprite()
{
}

/*!
The width of the sprite, in its current image.
*/
int QwVirtualSprite::width() const
{
	return image()->width();
}

/*!
The height of the sprite, in its current image.
*/
int QwVirtualSprite::height() const
{
	return image()->height();
}

/*!
The width of the sprite in collision-detection coordinates.
*/
int QwVirtualSprite::colWidth() const
{
	return image()->colw;
}

/*!
The height of the sprite in collision-detection coordinates.
*/
int QwVirtualSprite::colHeight() const
{
	return image()->colh;
}

/*!
(override)

Tests if the given pixel touches the sprite.  This uses pixel-accurate
detection, using the collision mask of the sprites current image (which
is by default the image mask).  This test is useful for example to test
when the user clicks a point with they mouse.  Note however, that
QwSpriteField::topAt(int x, int y) and associated methods give a more
efficient and flexible technique for this purpose.
*/
bool QwVirtualSprite::at(int px, int py) const
{
	px=world_to_col(px)-absColX();
	py=world_to_col(py)-absColY();

	if (px<0 || px>colWidth() || py<0 || py>colHeight())
		return false;

	QImage& img=*(image()->collision_mask);

	if (img.bitOrder() == QImage::LittleEndian) {
		return *(img.scanLine(py) + (px >> 3)) & (1 << (px & 7));
	} else {
		return *(img.scanLine(py) + (px >> 3)) & (1 << (7 -(px & 7)));
	}
}

/*!
\sa neighbourhood(int,int,QwSpritePixmap*)
*/
bool QwVirtualSprite::exact(Pix p) const
{
	QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;
	QRect area(iterator->x,iterator->y,iterator->w,iterator->h);
	return iterator->element()->at(area)
		&& (!iterator->collision_mask
		|| iterator->element()->at(iterator->collision_mask, area));
}

/*!
Used by the methods associated with
QwSpriteField::topAt(int x, int y) to test for a close hit.
\sa neighbourhood(int,int,QwSpritePixmap*)
*/
bool QwVirtualSprite::at(const QRect& rect) const
{
	QRect crect(world_to_col(rect.x()),world_to_col(rect.y()),
		world_to_col(rect.width()),world_to_col(rect.height()));
	QRect myarea(absColX(),absColY(),colWidth(),colHeight());
	return crect.intersects(myarea);
}

/*!
Used by the methods associated with
QwSpriteField::topAt(int x, int y) to test for an exact hit.
\sa neighbourhood(int,int,QwSpritePixmap*)
*/
bool QwVirtualSprite::at(const QImage* yourimage, const QRect& yourarea) const
{
	QRect cyourarea(world_to_col(yourarea.x()),world_to_col(yourarea.y()),
		world_to_col(yourarea.width()),world_to_col(yourarea.height()));

	QImage* myimage=image()->collision_mask;

	QRect ourarea(absColX(),absColY(),colWidth(),colHeight()); // myarea

	ourarea=ourarea.intersect(cyourarea);

	int yx=ourarea.x()-cyourarea.x();
	int yy=ourarea.y()-cyourarea.y();
	int mx=ourarea.x()-absColX();
	int my=ourarea.y()-absColY();
	int w=ourarea.width();
	int h=ourarea.height();

	// XXX 
	// XXX A non-linear search would typically be more
	// XXX efficient.  Optimal would be spiralling out
	// XXX from the center, but a simple vertical expansion
	// XXX from the centreline would suffice.
	// XXX
	// XXX My sister just had a baby 40 minutes ago, so
	// XXX I'm too brain-spun to implement it correctly!
	// XXX 
	//

	// Let's make an assumption.  That sprite masks don't have
	// bit orders for different machines!
	//
	// ASSERT(myimage->bitOrder()==yourimage->bitOrder());

	if (myimage->bitOrder() == QImage::LittleEndian) {
		for (int j=0; j<h; j++) {
			for (int i=0; i<w; i++) {
				if (*(yourimage->scanLine(yy+j) + (yx+i >> 3)) & (1 << ((yx+i) & 7))
				&& *(myimage->scanLine(my+j) + (mx+i >> 3)) & (1 << ((mx+i) & 7)))
				{
					return true;
				}
			}
		}
	} else {
		for (int j=0; j<h; j++) {
			for (int i=0; i<w; i++) {
				if (*(yourimage->scanLine(yy+j) + (yx+i >> 3)) & (1 << (7-((yx+i) & 7)))
				&& *(myimage->scanLine(my+j) + (mx+i >> 3)) & (1 << (7-((mx+i) & 7))))
				{
					return true;
				}
			}
		}
	}

	return false;
}

/*!
Creates an iterator which can traverse the area which the QwVirtualSprite
would cover if it had the given position and image.  This `would cover'
concept is useful as it allows you to check for a collision <em>before</em>
moving the sprite.

*/
Pix QwVirtualSprite::neighbourhood(int nx, int ny, QwSpritePixmap* img)
{
	if (!spritefield) return 0;

	QwSpriteFieldIterator* iterator=
		QwSpriteFieldIterator::make(nx,ny,img,spritefield);

	while (iterator && iterator->element()==this)
		iterator=iterator->next(spritefield);

	return (Pix)iterator;
}

/*!
Creates an iterator which traverses the QwSpriteFieldGraphic objects which
collide with this sprite at its current position.

\sa neighbourhood(int,int,QwSpritePixmap*)
*/
Pix QwVirtualSprite::neighbourhood() { return neighbourhood(x(),y(),image()); }

/*!
Creates an iterator which traverses the QwSpriteFieldGraphic objects which
would collide with this sprite if it was moved to the given position (yet
kept its current image).

\sa neighbourhood(int,int,QwSpritePixmap*)
*/
Pix QwVirtualSprite::neighbourhood(int nx, int ny) { return neighbourhood(nx,ny,image()); }

/*!
Traverse to the next QwSpriteFieldGraphic in a collision list.

\sa neighbourhood(int,int,QwSpritePixmap*)
*/
void QwVirtualSprite::next(Pix& p)
{
	QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;

	iterator=iterator->next(spritefield);

	while (iterator && iterator->element()==this)
		iterator=iterator->next(spritefield);

	p=(Pix)iterator;
}

/*!
Terminate a traversal early.

\sa neighbourhood(int,int,QwSpritePixmap*) QwSpriteField::end(Pix&)
*/
void QwVirtualSprite::end(Pix& p)
{
	if (p) {
		delete (QwSpriteFieldIterator*)p;
		p=0;
	}
}

/*!
Returns the QwSpriteFieldGraphic at the given traversal point.

\sa neighbourhood(int,int,QwSpritePixmap*)
*/
QwSpriteFieldGraphic* QwVirtualSprite::at(Pix p) const
{
	if (p) {
		QwSpriteFieldIterator* iterator=(QwSpriteFieldIterator*)p;
		return iterator->element();
	} else {
		return 0;
	}
}

/*!
Test if this sprite is touching the given QwSpriteFieldGraphic.
This is a convenient packaging of neighbourhood() and related methods.
*/
bool QwVirtualSprite::hitting(QwSpriteFieldGraphic& other) const
{
	return wouldHit(other,x(),y(),image());
}

/*!
Test if this sprite would (exactly) touch the given QwSpriteFieldGraphic.
This is a convenient packaging of neighbourhood() and related methods.
*/
bool QwVirtualSprite::wouldHit(QwSpriteFieldGraphic& other, int x, int y, QwSpritePixmap* img) const
{
	// Cast off const.  Safe, because we don't call non-const methods
	// of the QwVirtualSprite returnsed by at().
	//
	QwVirtualSprite* nconst_this=(QwVirtualSprite*)this;

	for (Pix p=nconst_this->neighbourhood(x,y,img); p; nconst_this->next(p)) {
		if (nconst_this->at(p)==&other) {
			if (nconst_this->exact(p)) {
				nconst_this->end(p);
				return true;
			}
		}
	}
	return false;
}




/*!
(override)

Draws the current image of the sprite at its current position,
as given by image() and x(), y().
*/
void QwVirtualSprite::draw(QPainter& painter)
{
	painter.drawPixmap(absX(),absY(),*image());
}

/*!
Using this method,
the resolution of the collision mask can be different to the
resolution of the pixel coordinates of the sprites.  One possible
use for this is to have differently scaled graphics being used by
different players in a multiplayer distributed game, yet for the
collision detection to be identical for each player.

\sa QwSpritePixmapSequence::readCollisionMasks(...)
*/
void QwVirtualSprite::setPixelCollisionPrecision(int downshifts)
{
	colprec=downshifts;
}


/*!
\class QwSpriteFieldView QwSpriteField.h
\brief A QWidget which views a QwSpriteField

A QWidget which views a QwSpriteField.
*/

/*!
Construct a QwSpriteFieldView which views the given QwSpriteField.  The
usual QWidget parameters may also be passed.
*/
QwSpriteFieldView::QwSpriteFieldView(QwSpriteField* v, QWidget* parent, const char* name, WFlags f) :
	QWidget(parent,name,f),
	viewing(v)
{
	view(v);
	if (v) {
		resize(v->width(),v->height());
	}
}

/*!
Deconstruct the QwSpriteFieldView
*/
QwSpriteFieldView::~QwSpriteFieldView()
{
	view(0);
}

/*!
Change the QwSpriteField which the QwSpriteFieldView is viewing.
*/
void QwSpriteFieldView::view(QwSpriteField* v)
{
	if (viewing) {
		viewing->removeView(this);
	}
	viewing=v;
	if (viewing) {
		viewing->addView(this);
	}
}

/*!
viewArea returns the area of `viewing' which this QwSpriteFieldView is
viewing.  The implementation in QwSpriteFieldView returns the area of
the widget, clipped by the area of the widget's parent (if any).
This should suffice for most cases.  The tighter this rectangle can
be defined, the more efficient QwSpriteField::Update will be.

Return zero-width area if nothing viewed.
*/
QRect QwSpriteFieldView::viewArea() const
{
	QWidget* parent=parentWidget();

	if (parent) {
		// translate parent rect to coord system of child (this)
		QRect clip=parent->rect();

		// clip child rect by parent rect
		clip=clip.intersect(geometry());

		// translate child rect to coord system of field
		clip.moveBy(-x(),-y());

		return clip;
	} else {
		// translate this to coord system of field
		return rect();
	}
}

/*!
(override)

Repaint the appropriate area of the QwSpriteField which this
QwSpriteFieldView is viewing.
*/
void QwSpriteFieldView::paintEvent(QPaintEvent *ev)
{
	if (viewing) {
		viewing->updateInView(this,ev->rect());
	}
}

/*!
Returns 0.

Although often frowned upon by purists, Run Time
Type Identification is very useful in this case, as it allows
a QwSpriteField to be an efficient indexed storage mechanism.

Make your derived classes return their own values for rtti(), and you
can distinguish between objects returned by QwSpriteField::at().  You should
use values greater than 1000 preferrably large a random number,
to allow for extensions to this class.

However, it is important not to overuse this facility, as
it damages extensibility.  For example, once you have identified
a base class of a QwSpriteFieldGraphic found by QwSpriteField::at(), cast it
to that type and call meaningful methods rather than acting
upon the object based on its rtti value.

For example:

\code
    if (field.at(p)->rtti() == MySprite::rtti()) {
        MySprite* s = (MySprite*)field.at(p);
        if (s->isDamagable()) s->loseHitPoints(1000);
        if (s->isHot()) myself->loseHitPoints(1000);
        ...
    }
\endcode
*/
int QwSpriteFieldGraphic::rtti() const { return 0; }

/*!
Returns 1.

\sa QwSpriteFieldGraphic::rtti()
*/
int QwVirtualSprite::rtti() const { return 1; }

