/* go_board.h
 *
 * Pieter Eendebak ( pte@ddsw.nl )
 *
 */

#ifndef GO_BOARD_H
#define GO_BOARD_H

#include <qobject.h>
#include <qsignal.h>
#include "ktimer.h"

/* for saving and loading */
#define FILE_PLAYER "Kgo player"

/* states for the go-board fields */
#define EMPTY 0
#define WHITE 1
#define BLACK 2
#define NEUTRAL 3
// will be phased out for NEUTRAL
#define NO_PLAYER 3
#define WHITE_ZONE 4
#define BLACK_ZONE 5

#define ILLEGAL -2
#define PASS -1
// used for marking 
#define MARK 1
#define MARKED 1
#define UN_MARKED 0

// reasons for illegal move
#define LEGAL 0
#define FIELD_NOT_EMPTY 4
#define SUICIDE 1
#define KO 2
#define OUTSIDE_BOARD 3
#define OPPONENTS_TURN 5

/* boardStateChange and play states*/
#define PLAYING 0
#define ENDED 3
#define COUNTING 6

// only signals
#define NEWGAME 1
#define RESUMED 4
#define PAUZED 5 

#define BOARD_ERROR -1
#define BOARD_SET 12
#define PLAYER_MADE_MOVE 10
#define PLAYER_PASSED 11

/* board size */
#define DEFAULT_X_SIZE 19
#define DEFAULT_Y_SIZE 19
#define MIN_SIZE 3
#define MAX_SIZE 32

/* determine opponent color */
#define OPPONENT(x)	((x==WHITE)?BLACK:WHITE) 

class KGoPlayer ;
class CountDialog ;
class KZoneBoard ;

/**
 * This class represents a Go board.
 * It offers methods for setting 2 players, @see KGoPlayer,
 * and can do basic operations such as counting pieces and
 * removing dead ones
 */
class KGoBoard : public QObject
{
	//friend CountDialog ;
	friend KZoneBoard ;

	Q_OBJECT

public:
	KGoBoard( int xWidth=DEFAULT_X_SIZE,
			int yWidth=DEFAULT_Y_SIZE );
	~KGoBoard();

	/**
	 * Start a new game (that means cleaning the board and 
	 * removing the current players).
	 */
	void newGame( int, int ) ;

	/**
	 * Call this after newGame() and setPlayer()
	 */
	void startGame();

	/**
	 * Pauze the current game
	 */
	void pauzeGame() ;

	/**
	 * Resume a pauzed game
	 */
	void resumeGame() ;

	/**
	 * End the current game. No moves can be made,
	 * the board cannot be changed.
	 */
	void endGame() ;

	/**
	 * This should be called after endGame().
	 * You can now remove dead pieces, allocate zones etc. 
	 */
	void startCounting() ;
	
	/**
	 * Load game data from a stream.
	 *
	 * @return A possible error code
	 *
	 * @see go.h
	 */
	int load( QDataStream & ) ;

	/**
	 * Save the current game to disk
	 */
	int save( QDataStream & ) ;

	/**
	 * @return true if this game can be saved
	 */
	bool saveAbleGame();

	int get( int, int ) const ;

//	/**
//	 * @return the last move made, usefull for checking on Ko
//	 */
//	void returnLastMove( &i, &j ) const ;

	/* resend a signal to player */
	void pushPlayer() ;

	/**
	 * The KGoBoard can have a variable size (assigned
	 * when creating the KGoBoard).
	 *
	 * @return x size of the board
	 */
	int getXWidth() const ;
	int getYWidth() const ;

	/**
	 * Calculate the score of the white player.
	 * The kGoBoard needs some help in calculating
	 * the final score, @see CountDialog
	 *
	 * @ return the score of the white player, or
	 *	-1 if the score cannot be determined
	 */
	int getWhiteScore() ;

	/**
	 * Calculate the score of the black player.
	 * The KGoBoard needs some help in calculating
	 * the final score, @see CountDialog
	 *
	 * @ return the score of the black player, or
	 *	-1 if the score cannot be determined
	 */
	int getBlackScore() ;

	/**
	 * Return the number of white pieces on the board
	 */
	int whitePieces() const ;
	/**
	 * Return the number of black pieces on the board
	 */
	int blackPieces() const ;
	/**
	 * Return the number of white pieces that have been captured
 	 */
	int whiteCaptured() const ;
	/**
	 * Return the number of black pieces pieces that have been captured
	 */
	int blackCaptured() const ; 

	/**
	 * Set the white player
	 */
	void setWhitePlayer( KGoPlayer *p ) ;
	/**
	 * Set the black player
	 */
	void setBlackPlayer( KGoPlayer *p ) ;

	/**
	 * @return The time white has used to far
	 */
	int getWhiteTime() ;
	/**
	 * @return The time black has used to far
	 */
	int getBlackTime() ;

	/**
	 * @return the color of the player to move
	 *
	 * Possible values : WHITE, BLACK, NO_PLAYER
	 */
	int playerToMove() ;

	/**
	 * @param p the color of the player to move
	 */
	void setPlayerToMove( int p ) ;

	void removeWhitePlayer() ;
	void removeBlackPlayer() ;

	/**
	 * @param i The illegal move
	 * @return Reason for an illegal move
	 */
	static QString getIllegalMoveReason( int i ) ; 

	/**
	 * @return string version of a move 
	 */
	static QString translateMove(int,int) ;

	/**
	 * @return string version of a player's color
	 */
	static QString translatePlayer(int) ;

	static void createArray( unsigned char** &p, int w, int h ) ;
	static void deleteArray( unsigned char** &p, int w, int h ) ;

signals:
	void boardStateChange( int ) ;
	void boardSetup() ;

	/**
	 * Emitted whenever a player makes an illegal move.
	 * The signal is connected to both players.
	 *
	 * @param r Reason for the illegal move
	 * @param p Player that made the illegal move
	 */
	void illegalMove( int r, int p ) ;

	/**
	 * Emitted whenever a player makes a legal move
	 */
	void playerMoved( int, int, int ) ;
	/**
	 * Obsolete.
	 * @see #playerMoved
	 */
	void blackMoved( int, int ) ;
	/**
	 * Obsolete.
	 * @see #playerMoved
	 */
	void whiteMoved( int, int ) ;

protected:
	/**
	 * Check if a point is in the board
	 *
	 * @return true if point is in the current board
	 */
	bool inRange( int x, int y ) const ;

	int getWhiteFields() ;
	int getBlackFields() ;

private slots:
	/**
	 * This slot is used for receiving moves from the white player.
	 */
	void whiteMove( int, int ) ;
	/**
	 * This slot is used for receiving moves from the black player.
	 */
	void blackMove( int, int ) ; 
private:
	void initData( int xS, int yS ) ;
	// for debugging
	void printBoard() ;

	void stopTimers() ;
	void resumeTimers() ;
	void resetTimers() ;

	void player_moved( int x, int y, int player ) ;
	//
	void emit_boardStateChange(int) ;
	void tryMove (int, int , int player ) ;
	void makeMove (int, int , int player ) ;
	void rejectMove( int, int, int player, int r ) ;

	// internal board 
	void set( int, int, int, bool notify=TRUE ) ;
	int legalMove( int, int, int player ) ;
	bool suicide( int, int, int player ) ;
	int countLib( int, int, int player ) ;
	void count( int, int, int player ) ;
	void eval( int player ) ;
	void examBoard( int player ) ;

	KGoPlayer *white ;
	KGoPlayer *black ;

	KTimer *white_timer ;
	KTimer *black_timer ;
	bool pauze ;

	// data
	/* size of the current field */
	int xWidth, yWidth ;

	// the board
	unsigned char **p ;
	/* working matrix for marking */
	unsigned char **ma ;
	/* liberty of current color */
	unsigned char **l ;

	/* current stone liberty */
	int lib ;

	/* game state */
	int play;
	/* pass indicator */
	int pass;

        /* location of white stone captured */
	int wh_cap_x, wh_cap_y ;
        /* location of black stone captured */
	int bl_cap_x, bl_cap_y ;

	/* number of captured stones */
	int white_cap, black_cap ;

	int player_to_move ;
} ;

#endif
