/* -*- C++ -*- */

#ifndef _CUBECONTAINER_H_
#define _CUBECONTAINER_H_

#include "Draw3D.h"

/** Abstract definition of a cube container. */
class CubeContainer
{
public:
	CubeContainer (const int _sizeX,
		       const int _sizeY,
		       const char _sizeZ);
	CubeContainer();
	CubeContainer& operator= (const CubeContainer& _oldCube);

	virtual ~CubeContainer();

	/** Adds a cube with the color color to position pos.
	 * No check is done if there already exists a cube
	 * -  causes memory leak if so -.
	 *
	 * Cubes are oriented from pos to
	 *  pos.getX+1, pos.getY+1, pos.getZ+ 1.
	 *
	 * On clear the color is deleted.
	 */
	void addCube (const Position& pos, const Color* color) { addCube (pos, color, cubes); };

	/** Returns the count of cubes set in this figure.*/
	int getCountOfCubes() const;

	virtual const Color** getCubes() const { return cubes; };

	int getSize() const { return size; };

	/** Returns the extend in x direction. */
	int getSizeX() const { return sizeX; };
	/** Returns the extend in y direction. */
	int getSizeY() const { return sizeY; };
	/** Returns the extend in z direction. */
	int getSizeZ() const { return sizeZ; };

	/** Removes and deletes all cubes. */
	virtual void clear (bool remove=true);

	/** Increments the position of x,y,z with order x,y,z. */
	void incrementPosition (int *x, int *y, int *z) const;

protected:
	/** Return index of pos in linear addressing cubes or planeZBuffer. */
	int get3DIndex (const Position& pos) const { return sizeY*sizeX*(int) pos.getZ()+sizeX*(int) pos.getY()+(int) pos.getX(); };

	/** Return index of pos in linear addressing maxHeight. */
	int get2DIndex (const int x, const int y) const { return sizeX*y+x; };

	virtual void addCube (const Position& pos, const Color* color, const Color** buffer);

	/** Extend of figure. */
	int sizeX, sizeY, sizeZ;

	/** sizeX*sizeY*sizeZ */
	int size;

	/** sizeX*sizeY */
	int sizeXY;

	/** Stores the color of the cubes or null if no cube present at position.
	 * cubes=[z][y][x] with 0<=x<sizeX, 0<=y<sizeY, 0<=z<sizeZ.
	 */
	const Color **cubes;
};

#endif _CUBECONTAINER_H_
