/* -*- C++ -*-

  $Id: XmlWriter.cc,v 1.2 1998/10/12 13:13:46 sattler Exp $

  This file is part of KIllustrator.
  Copyright (C) 1998 Kai-Uwe Sattler (kus@iti.cs.uni-magdeburg.de)

  modified for kvoctrain by Ewald Arnold ewald@ewald-arnold.de in April 99

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by  
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "XmlWriter.h"

XmlWriter::XmlWriter (ostream& os) 
  : strm (os.rdbuf ())
{
  autoendl = true;
  strm << "<?xml version=\"1.0\"?>" << endl;
}

XmlWriter::~XmlWriter () {
  flush ();
}

void XmlWriter::startTag (const char* id, bool closeIt, bool empty, bool eol) {

  if (!id) return;

  strm << "<" << id;
  if (!empty) {
    lastTags.push (id);
  }

  if (closeIt) {
    if (empty) 
      strm << "/";
    strm << ">";
    if (eol || autoendl)
      strm << endl;
  }
}
 
void XmlWriter::endTag (const char* id, bool eol) {

  if (!id) return;

  strm << "</";
  if (id)
    strm << id;
  else {
    QString tag = lastTags.top ();
    lastTags.pop ();
    strm << tag;
  }
  strm << ">";
  if (eol || autoendl)
    strm << endl;
}

void XmlWriter::closeTag (bool empty, bool eol) {
  if (empty) {
    strm << "/";
    lastTags.pop ();
  }
  strm << ">";
  if (eol || autoendl)
    strm << endl;
}

void XmlWriter::setAutoEndl (const bool flag)
{
  autoendl = flag;
}

void XmlWriter::endline()
{
  strm << endl;
}

void XmlWriter::addAttribute (const char* name, const QString& value) {

  if (!name) return;

  QString val = value;
  // escape dangerous characters in sgml-style
  int pos = 0;
  while ((pos = val.find ('&', pos)) >= 0) {
    val.insert (pos+1, "amp;");
    pos += 5; // skip &amp;
  }
  pos = 0;
  while ((pos = val.find ('\n', pos)) >= 0) {
    val.remove (pos, 1);
    val.insert (pos, "&nl;");
    pos += 4; // skip &nl;
  }
  pos = 0;
  while ((pos = val.find ('\r', pos)) >= 0) {
    val.remove (pos, 1);
    val.insert (pos+1, "lf;");
    pos += 4; // skip &lf;
  }
  pos = 0;
  while ((pos = val.find ('\"', pos)) >= 0) {
    val.remove (pos, 1);
    val.insert (pos, "&quot;");
    pos += 6; // skip &qout;
  }

  strm << " ";
  strm << name << "=\"";
  strm << val;
  strm << "\"";
}

void XmlWriter::addAttribute (const char* name, int value) {

  if (!name) return;

  strm << " ";
  strm << name << "=\"";
  strm << value;
  strm << "\"";
}

void XmlWriter::addAttribute (const char* name, float value) {

  if (!name) return;

  strm << " ";
  strm << name << "=\"";
  strm << value;
  strm << "\"";
}

void XmlWriter::addAttribute (const char* name, double value) {

  if (!name) return;

  strm << " ";
  strm << name << "=\"";
  strm << value;
  strm << "\"";
}

void XmlWriter::addAttribute (const char* name, const QColor& color) {

  if (!name) return;

  strm << " ";
  strm << name << "=\"";
  strm << color.red () << ' ' << color.green () << ' ' << color.blue ();
  strm << "\"";
}

void XmlWriter::addAttribute (const char* name, const QWMatrix& m) {

  if (!name) return;

  strm << " ";
  strm << name << "=\"";
  strm << m.m11 () << ' ' << m.m12 () << ' ' << 0 << ' ';
  strm << m.m21 () << ' ' << m.m22 () << ' ' << 0 << ' ';
  strm << m.dx () << ' ' << m.dy () << ' ' << 1;
  strm << "\"";
}

void XmlWriter::writeText (const char* s) {
  const char* c = s;
  while (c && *c != '\0') {
    if (*c == '<') strm << "&lt;";
    else if (*c == '&') strm << "&amp;";
    else if (*c == '>') strm << "&gt;";
    else strm << *c;
    c++;
  }
}

void XmlWriter::writeTag (const char* s) {
  strm << "<" << (s ? s : "") << ">";
}

void XmlWriter::flush () {
  strm.flush ();
}
