/***************************************************************************
                       QueryManager.cpp  - manage queries
                             -------------------
    begin                : Sat Jul 11 20:50:53 MET 1999

    copyright            : (C) 1999,2000 by Ewald Arnold
    email                : ewald@ewald-arnold.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "QueryManager.h"
#include "resource.h"
#include "eadebug.h"
#include "compat_2x.h"

#include <kapp.h>
#include <kvoctraindoc.h>
#include <time.h>
#include <iostream.h>
#include <vector.h>

vector<QString> QueryManager::userTypes;

struct t_type_rel
{
   const char *short_ref;
   const char *long_ref;
};


#ifndef i18n_noop
# define i18n_noop(x) (x)
#endif

// types are hierarchical !
// - . divides main from sub type
// - sub types must follow main type

// user types are strings-references like this: #1

static t_type_rel InternalTypeRelations [] =
{
  { QM_ADJ,                           i18n_noop("adjective") },

  { QM_ADV,                           i18n_noop("adverb") },

  { QM_ART,                           i18n_noop("article") },
  { QM_ART QM_TYPE_DIV QM_ART_DEF,    i18n_noop("article definite") },
  { QM_ART QM_TYPE_DIV QM_ART_IND,    i18n_noop("article indefinite") },

  { QM_CON,                           i18n_noop("conjunction") },

//{ QM_FIG,                           i18n_noop("figure") },

//{ QM_INFORMAL,                      i18n_noop("informal") },

  { QM_NAME,                          i18n_noop("name") },   // old type "3"

  { QM_NOUN,                          i18n_noop("noun") },   // old type "2"
  { QM_NOUN QM_TYPE_DIV QM_NOUN_M,    i18n_noop("noun male") },
  { QM_NOUN QM_TYPE_DIV QM_NOUN_F,    i18n_noop("noun female") },
  { QM_NOUN QM_TYPE_DIV QM_NOUN_S,    i18n_noop("noun natural") },

  { QM_NUM,                           i18n_noop("numeral") },
  { QM_NUM QM_TYPE_DIV QM_NUM_ORD,    i18n_noop("numeral ordinal") },
  { QM_NUM QM_TYPE_DIV QM_NUM_CARD,   i18n_noop("numeral cardinal") },

  { QM_PHRASE,                        i18n_noop("phrase") },

  { QM_PREP,                          i18n_noop("preposition") },

  { QM_PRON,                          i18n_noop("pronoun") },
  { QM_PRON QM_TYPE_DIV QM_PRON_POS,  i18n_noop("pronoun possessive") },
  { QM_PRON QM_TYPE_DIV QM_PRON_PER,  i18n_noop("pronoun personal") },

  { QM_QUEST,                         i18n_noop("question") },

  { QM_VERB,                          i18n_noop("verb") },   // old type "1"
  { QM_VERB  QM_TYPE_DIV QM_VERB_IRR, i18n_noop("verb irregular") },

  { 0, 0 }  // the end
};


QString QueryManager::getSubType (QString type)
{
  int i;
  if ((i = type.find(QM_TYPE_DIV)) >= 0) {
    type.remove(0, i+1);
    return type;
  }
  else
    return "";
}


QString QueryManager::getMainType (QString type)
{
  int i;
  if ((i = type.find(QM_TYPE_DIV)) >= 0)
    return type.left(i);
  else
    return type;
}


QueryManager::QueryManager ()
{
  datecomp = DontCare;
  badcomp = DontCare;
  querycomp = DontCare;
  gradecomp = DontCare;
  typecomp = DontCare;

  dateitem = 0;
  queryitem = 0;
  baditem = 0;
  typeitem = "";
  gradeitem = 0;
  lessonitem = Current;
}


void QueryManager::loadConfig (KConfig *config)
{
  datecomp = (CompType) config->readNumEntry(CFG_QM_DATE_COMP, DontCare);
  badcomp = (CompType) config->readNumEntry(CFG_QM_BAD_COMP, DontCare);
  querycomp = (CompType) config->readNumEntry(CFG_QM_QUERY_COMP, DontCare);
  gradecomp = (CompType) config->readNumEntry(CFG_QM_GRADE_COMP, DontCare);
  typecomp = (CompType) config->readNumEntry(CFG_QM_TYPE_COMP, DontCare);
  lessoncomp = (CompType) config->readNumEntry(CFG_QM_LESSON_COMP, Current);

  dateitem = (CompType) config->readNumEntry(CFG_QM_DATE_ITEM, 0);
  baditem = (CompType) config->readNumEntry(CFG_QM_BAD_ITEM, 0);
  queryitem = (CompType) config->readNumEntry(CFG_QM_QUERY_ITEM, 0);
  gradeitem = (CompType) config->readNumEntry(CFG_QM_GRADE_ITEM, 0);
  typeitem = config->readEntry(CFG_QM_TYPE_ITEM, "");
  lessonitem = config->readNumEntry(CFG_QM_LESSON_ITEM, 0);

  blockItems.clear();
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"1", 1 *      60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"2", 2 *      60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"3", 4 *      60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"4", 1 *    7*60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"5", 2 *    7*60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"6", 1 * 30*7*60*60*24));
  blockItems.push_back(config->readNumEntry(CFG_QM_BLOCK_ITEM"7", 2 * 30*7*60*60*24));
 
  expireItems.clear();
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"1", 2 *      60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"2", 4 *      60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"3", 1 *    7*60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"4", 2 *    7*60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"5", 1 * 30*7*60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"6", 2 * 30*7*60*60*24));
  expireItems.push_back(config->readNumEntry(CFG_QM_EXPIRE_ITEM"7", 4 * 30*7*60*60*24));
}


void QueryManager::saveConfig (KConfig *config)
{
  config->writeEntry(CFG_QM_DATE_COMP, (int) datecomp);
  config->writeEntry(CFG_QM_BAD_COMP, (int) badcomp);
  config->writeEntry(CFG_QM_QUERY_COMP, (int) querycomp);
  config->writeEntry(CFG_QM_GRADE_COMP, (int) gradecomp);
  config->writeEntry(CFG_QM_TYPE_COMP, (int) typecomp);
  config->writeEntry(CFG_QM_LESSON_COMP, (int) lessoncomp);

  config->writeEntry(CFG_QM_DATE_ITEM, (int) dateitem);
  config->writeEntry(CFG_QM_BAD_ITEM, (int) baditem);
  config->writeEntry(CFG_QM_QUERY_ITEM, (int) queryitem);
  config->writeEntry(CFG_QM_GRADE_ITEM, (int) gradeitem);
  config->writeEntry(CFG_QM_TYPE_ITEM, typeitem);
  config->writeEntry(CFG_QM_LESSON_ITEM, (int) lessonitem);

  QString s;
  for (int i = KV_LEV1_GRADE;
       i <= KV_MAX_GRADE && i <= (int) blockItems.size(); i++) {
    s.setNum(i);
    s.insert(0, CFG_QM_BLOCK_ITEM);
    config->writeEntry(s, blockItems[i-1]);
  }

  for (int i = KV_LEV1_GRADE;
       i <= KV_MAX_GRADE && i <= (int) expireItems.size(); i++) {
    s.setNum(i);
    s.insert(0, CFG_QM_EXPIRE_ITEM);
    config->writeEntry(s, expireItems[i-1]);
  }
}


int QueryManager::blockItem (int grade) const
{
   if (grade <= (int) blockItems.size() && grade > 0)
    return  blockItems[grade-1];
   else
    return 0;
}


int QueryManager::expireItem (int grade) const
{
   if (grade <= (int) expireItems.size() && grade > 0)
    return expireItems[grade-1];
   else
    return 0;
}


void QueryManager::setBlockItem (int item, int grade)
{
   if (grade > KV_MAX_GRADE || grade > 0)
     return;

   for (int i = (int) blockItems.size(); i <= grade; i++)
     blockItems.push_back(0);
   blockItems[grade-1] = item;
}


void QueryManager::setExpireItem (int item, int grade)
{
   if (grade > KV_MAX_GRADE || grade > 0)
     return;

   for (int i = (int) expireItems.size(); i <= grade; i++)
     expireItems.push_back(0);
   expireItems[grade-1] = item;
}


QuerySelection QueryManager::select(kvoctrainDoc *doc, int act_lesson,
                                    int oindex, int tindex,
                                    bool swap, bool block, bool expire)
{
   QuerySelection random;
   random.resize(doc->numLessons()+1);
   for (int i = 0; i < doc->numEntries(); i++)
     doc->getEntry(i)->setSelected(false);

   // selecting might take rather long
   int ent_no = 0;
   int ent_percent = doc->numEntries() / 100;
   float f_ent_percent = doc->numEntries() / 100.0;
   emit doc->progressChanged(doc, 0);

   for (int i = 0; i < doc->numEntries(); i++) {
     ent_no++;
     if (ent_percent != 0 && (ent_no % ent_percent) == 0 )
       emit doc->progressChanged(doc, ent_no / f_ent_percent);

     kvoctrainExpr *expr = doc->getEntry(i);
     if (swap) {
       if (  validate (expr, act_lesson,
                     oindex, tindex,
                     block, expire)
           ||validate (expr, act_lesson,
                     tindex, oindex,
                     block, expire)) {
         random[expr->getLesson()].push_back (QueryEntryRef(expr, i));
         expr->setSelected(true);
       }
     }
     else {
       if (validate (expr, act_lesson,
                     oindex, tindex,
                     block, expire)) {
         random[expr->getLesson()].push_back (QueryEntryRef(expr, i));
         expr->setSelected(true);
       }
     }
   }

   // remove empty lesson elements
   for (int i = (int) random.size()-1; i >= 0; i--)
     if (random[i].size() == 0)
       random.erase(&random[i], &random[i+1]);
   return random;
}


bool QueryManager::validate(kvoctrainExpr *expr, int act_lesson,
                            int oindex, int tindex,
                            bool block, bool expire)
{
   int index = tindex ? tindex : oindex;
   if ((compareExpiring(expr->getGrade(index, oindex != 0),
                        expr->getQueryDate(index, oindex != 0), expire)
        ||

        (
            compareGrade (gradecomp, expr->getGrade(index, oindex != 0), gradeitem)
         && compareQuery (querycomp, expr->getQueryCount(index, oindex != 0), queryitem)
         && compareBad (badcomp, expr->getBadCount(index, oindex != 0), baditem)
         && compareDate (datecomp, expr->getQueryDate(index, oindex != 0), dateitem)

         && compareBlocking(expr->getGrade(index, oindex != 0),
                            expr->getQueryDate(index, oindex != 0), block)
        )
       )
//     lesson + word type must ALWAYS match
//     (and there must be a word on both sides)
       && compareLesson (lessoncomp, expr->getLesson(), lessonitem, act_lesson)
       && compareType (typecomp, expr->getType(index), typeitem)
       && !expr->getOriginal().isEmpty()
       && !expr->getTranslation(index).isEmpty()
      )
     return true;
   else
     return false;

}


QuerySelection QueryManager::select(kvoctrainDoc *doc, int act_lesson,
                                    int idx, QString type)
{
   QuerySelection random;
   random.resize(doc->numLessons()+1);

   // selecting might take rather long
   int ent_no = 0;
   int ent_percent = doc->numEntries() / 100;
   float f_ent_percent = doc->numEntries() / 100.0;
   emit doc->progressChanged(doc, 0);

   for (int i = 0; i < doc->numEntries(); i++) {
     ent_no++;
     if (ent_percent != 0 && (ent_no % ent_percent) == 0 )
       emit doc->progressChanged(doc, ent_no / f_ent_percent);

     kvoctrainExpr *expr = doc->getEntry(i);
     if (validate (expr, act_lesson, idx, type)) {
       random[expr->getLesson()].push_back (QueryEntryRef(expr, i));
     }
   }

   // remove empty lesson elements
   for (int i = (int) random.size()-1; i >= 0; i--)
     if (random[i].size() == 0)
       random.erase(&random[i], &random[i+1]);

   return random;
}


bool QueryManager::validate(kvoctrainExpr *expr, int act_lesson,
                            int idx, QString query_type)
{
   QString qtype;
   int pos = query_type.find (QM_TYPE_DIV);
   if (pos >= 0)
     qtype = query_type.left (pos);
   else
     qtype = query_type;

   QString expr_type = expr->getType(idx);
//   cout << expr->getOriginal() << " " << expr_type << " "<< qtype << " ";
   bool type_ok = false;
   if (qtype == QM_NOUN) {
     type_ok =    expr_type == QM_NOUN  QM_TYPE_DIV  QM_NOUN_S
               || expr_type == QM_NOUN  QM_TYPE_DIV  QM_NOUN_M
               || expr_type == QM_NOUN  QM_TYPE_DIV  QM_NOUN_F;

//     cout << "NOUN " << (int) type_ok << " ";
   }
   else if (qtype == QM_VERB) {
     type_ok = (   expr_type == QM_VERB
                || expr_type == QM_VERB  QM_TYPE_DIV  QM_VERB_IRR
               )
               && expr->getConjugation(idx).numEntries() > 0;

//     cout << "VERB " << (int) type_ok << " ";
   }
   else if (qtype == QM_ADJ) {
     type_ok =    expr_type == QM_ADJ
               && !expr->getComparison(idx).isEmpty();
//     cout << "ADJ " << (int) type_ok << " ";
   }
   else
     return false;

   if (compareLesson (lessoncomp, expr->getLesson(),
                      lessonitem, act_lesson)
       && type_ok) {
//     cout << "LESSON " << (int) true << endl;
     return true;
     }
   else {
//     cout << "LESSON " << (int) false << endl;
     return false;
   }
}


QString QueryManager::compStr (CompType type)
{
   QString str = "???";
   switch (type) {
    case DontCare: str = i18n("don`t care"); break;
    case WorseThan: str = i18n("worse than"); break;
    case WorseEqThan: str = i18n("equal/worse than"); break;
    case MoreThan: str = i18n(">"); break;
    case MoreEqThan: str = i18n(">="); break;
    case BetterEqThan: str = i18n("equal/better than"); break;
    case BetterThan: str = i18n("better than"); break;
    case LessEqThan: str = i18n("<="); break;
    case LessThan: str = i18n("<"); break;

    case EqualTo : str = i18n("equal to"); break;
    case NotEqual: str = i18n("not equal"); break;

    case Within    : str = i18n("within last"); break;
    case Before    : str = i18n("before"); break;
    case NotQueried: str = i18n("not queried"); break;

    case Current    : return i18n("current lesson"); break;
    case NotAssigned: return i18n("not assigned"); break;
    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::CompStr]: unknown type");
   }
   return str;
}


QString QueryManager::gradeStr (int i)
{
  switch (i) {
    case KV_NORM_GRADE:    return i18n(KV_NORM_TEXT); break;
    case KV_LEV1_GRADE:    return i18n(KV_LEV1_TEXT); break;
    case KV_LEV2_GRADE:    return i18n(KV_LEV2_TEXT); break;
    case KV_LEV3_GRADE:    return i18n(KV_LEV3_TEXT); break;
    case KV_LEV4_GRADE:    return i18n(KV_LEV4_TEXT); break;
    case KV_LEV5_GRADE:    return i18n(KV_LEV5_TEXT); break;
    case KV_LEV6_GRADE:    return i18n(KV_LEV6_TEXT); break;
    case KV_LEV7_GRADE:    return i18n(KV_LEV7_TEXT); break;
    default:               return i18n(KV_LEV1_TEXT); break;
  }
}


vector<TypeRelation> QueryManager::getRelation (bool only_maintypes)
{
  vector<TypeRelation> vec;
  for (int i = 0; i < (int) userTypes.size(); i++) {
    QString s;
    s.setNum(i+1);
    s.insert(0, QM_USER_TYPE);
    vec.push_back(TypeRelation(s, userTypes[i]));
  }


  t_type_rel *type = InternalTypeRelations;
  while (type->short_ref != 0) {
    if (!only_maintypes || strstr (type->short_ref, QM_TYPE_DIV) == 0)
      vec.push_back(TypeRelation(type->short_ref, i18n(type->long_ref)));
    type++;
  }

  return vec;
}


QString QueryManager::typeStr (const QString id)
{
  if (id.left(1) == QM_USER_TYPE) {
    QString num = id;
    num.remove (0, 1);
    int i = num.toInt()-1;
    if (i >= 0 && i < (int) userTypes.size() )
      return userTypes[i];
    else
      return "";
  }
  else {
    t_type_rel *type = InternalTypeRelations;
    while (type->short_ref != 0) {
      if (type->short_ref == id)
        return i18n(type->long_ref);
      type++;
    }
  }
  return "";
}


bool QueryManager::compareBlocking (int grade, int date, bool use_it)
{
   time_t cmp = blockItem(grade);
   if (grade == KV_NORM_GRADE || cmp == 0 || !use_it) // don't care || all off
     return true;
   else {
     time_t now = time(0);
     return date+cmp < now;
   }
}


bool QueryManager::compareExpiring (int grade, int date, bool use_it)
{
   time_t cmp = expireItem(grade);
   if (grade == KV_NORM_GRADE || cmp == 0 || !use_it) // don't care || all off
     return false;
   else {
     time_t now = time(0);
     return date+cmp < now;
   }
}


bool QueryManager::compareDate (CompType type, time_t qd, time_t limit)
{
// KDEBUG3 (KDEBUG_INFO, 0, "compareDate type:%d qd:%d limit:%d", type, qd, limit);
   time_t now = time(0);
   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case Before: erg = qd == 0 || qd < now-limit; // never queried or older date
    break;

    case Within: erg = qd >= now-limit;           // newer date
    break;

    case NotQueried: erg = qd == 0;    
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareDate]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareDate erg:%d", erg);
   return erg;
}


bool QueryManager::compareQuery (CompType type, int qgrade, int limit)
{
// KDEBUG3 (KDEBUG_INFO, 0, "compareQuery type:%d qgrade:%d limit:%d", type, qgrade, limit);
   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case MoreThan: erg = qgrade > limit;    // sel has higher query count
    break;

    case MoreEqThan: erg = qgrade >= limit;
    break;

    case EqualTo: erg = qgrade == limit;
    break;

    case NotEqual: erg = qgrade != limit;
    break;

    case LessEqThan: erg = qgrade <= limit;   // sel has less count
    break;

    case LessThan: erg = qgrade < limit;   
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareQuery]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareQuery erg:%d", erg);
   return erg;
}


bool QueryManager::compareBad (CompType type, int bcount, int limit)
{
// KDEBUG3 (KDEBUG_INFO, 0, "compareBad type:%d bcount:%d limit:%d", type, bcount, limit);
   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case MoreThan: erg = bcount > limit;    // sel has higher bad count
    break;

    case MoreEqThan: erg = bcount >= limit; 
    break;

    case EqualTo: erg = bcount == limit;
    break;

    case NotEqual: erg = bcount != limit;
    break;

    case LessEqThan: erg = bcount <= limit;   // sel has less count
    break;

    case LessThan: erg = bcount < limit; 
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareBad]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareBad erg:%d", erg);
   return erg;
}


bool QueryManager::compareGrade (CompType type, grade_t qgrade, grade_t limit)
{
// KDEBUG3 (KDEBUG_INFO, 0, "compareGrade type:%d qgrade:%d limit:%d", type, qgrade, limit);
   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case WorseThan: erg = qgrade < limit;    // sel has worse grade
    break;

    case WorseEqThan: erg = qgrade <= limit;
    break;

    case EqualTo: erg = qgrade == limit;
    break;

    case NotEqual: erg = qgrade != limit;
    break;

    case BetterEqThan: erg = qgrade >= limit;   // sel has better grade
    break;

    case BetterThan: erg = qgrade > limit;
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareGrade]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareGrade erg:%d", erg);
   return erg;
}


bool QueryManager::compareType (CompType type, QString exprtype, QString limit)
{
// KDEBUG3 (KDEBUG_INFO, 0, "compareType type:%d exprtype:%s limit:%s ", type, (const char*) exprtype, (const char*) limit);
   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case EqualTo: erg = getMainType(exprtype) == getMainType(limit);       // type is same
    break;

    case NotEqual: erg = getMainType(exprtype) != getMainType(limit);      // other type
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareType]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareType erg:%d", erg);
   return erg;
}


bool QueryManager::compareLesson (CompType type, int less, int limit, int current)
{
// KDEBUG4 (KDEBUG_INFO, 0, "compareLesson type:%d less:%d limit:%d current:%d", type, less, limit, current);

   bool erg = true;
   switch (type) {
    case DontCare: erg = true;
    break;

    case EqualTo: erg = less == limit;
    break;

    case NotEqual: erg = less != limit;
    break;

    case Current: erg = less == current;
    break;

    case NotAssigned: erg = less == 0;
    break;

    default:
      KDEBUG (KDEBUG_ERROR, 0, "[QueryManager::compareLesson]: unknown type");
   }
// KDEBUG1 (KDEBUG_INFO, 0, "compareLesson erg:%d", erg);
   return erg;
}


void QueryManager::setTypeNames (vector<QString> names)
{
  userTypes = names;
}
