/***************************************************************************
                          vvokfile.cpp  -  description
                             -------------------
    begin                : Wed Feb 9 2000
    copyright            : (C) 2000 by Friedrich W. H. Kossebau
    email                : Friedrich.W.H@Kossebau.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "vvokfile.h"

#include "../vvocabulary.h"

static QString printableToString(const QString& s)
{
  if (!s.contains('\\'))
    return s;
    
  QString Result="";
  unsigned int i = 0;
  if (s.length()>1){ // remember: s.length() is unsigned....
    for (i=0; i<s.length()-1; i++){
      if (s[i] == '\\')
      {
      	i++;
      	if (s[i] == '\\')
      	  Result.insert(Result.length(), s[i]);
      	else if (s[i] == 'n')
      	  Result.append("\n");
      	else if (s[i] == 't')
      	  Result.append("\t");
      	else if (s[i] == 'r')
      	  Result.append("\r");
      	else 
      	{
      	  Result.append( "\\" );
      	  Result.insert( Result.length(), s[i]);
      	}
      }
      else
	    Result.insert(Result.length(), s[i]);
    }
  }
  
  // because the loop only goes to the next to the last character as "\." takes two
  if (i<s.length())
    Result.insert( Result.length(), s[i]);
    
  return Result;
}

static QString stringToPrintable(const QString& s)
{
  QString Result="";
  unsigned int i = 0;
  if (s.length()>0) // remember: s.length() is unsigned....
  {
    for (i=0; i<s.length(); i++)
    {
      switch (s[i])
      {
        case '\n': Result.append( "\\n" ); break;
        case '\t': Result.append( "\\t" ); break;
        case '\r': Result.append( "\\r" ); break;
        default  : Result.insert( Result.length(), s[i]);
      }
    }
  }
      
  return Result;
}


VVokFile::VVokFile() : Group("<default>"), GroupDict( 37, false )
{
  GroupDict.setAutoDelete( true );
}



VVokFile::~VVokFile()
{

}


void VVokFile::clear()
{
  GroupDict.clear();
}


void VVokFile::close()
{
  File.close();
}


/**  */
bool VVokFile::isReadable( const QString &Filename )
{
	File.setName( Filename );
	
	return( File.open( IO_ReadOnly ) );
}

 
/**  */
void VVokFile::readExtraData( VVocabulary *Vocabulary )
{
  //reset the lists
  GroupDict.clear();  
    
  // prepare the operators
  QString CurrentLine;
  QString CurrentGroup = "";
  
  VEntryDict* CurrentDict = GroupDict[ "<default>" ];
  
  // reset the stream's device
  File.at(0);
  QTextStream *Stream = new QTextStream( &File );
  
  while( !Stream->eof() )
  {
    CurrentLine = Stream->readLine();

    // check for a group
    int LeftBracket = CurrentLine.find( '[' );
    int RightBracket = CurrentLine.find( ']', 1 );
    if( LeftBracket == 0 && RightBracket != -1 )
    {
	    // group found; get the group name by taking everything in  
  	  // between the brackets  
	    CurrentGroup = CurrentLine.mid( 1, RightBracket-1 );
	        
	    // if the vocabulary starts return
  	  if( CurrentGroup == "Vocabulary" )
  	  {
        delete Stream;
	      return;
	    }

  	  // check if there already is such a group in the group
	    // dictionary
  	  CurrentDict = GroupDict[ CurrentGroup ];
	    if( !CurrentDict )
	    {
	      // no such group and not the vocabulary -> create a new entry dictionary
	      VEntryDict* NewDict = new VEntryDict( 37, false );
	      NewDict->setAutoDelete( true );
	      GroupDict.insert( CurrentGroup, NewDict );

	      // this is now the current group
	      CurrentDict = NewDict;
	    }
	    continue;
  	};

    if( CurrentLine[0] == '#' )
     	// comment character in the first column, skip the line
      continue;
      	    
    int EqualsPos = CurrentLine.find( '=' );
    if( EqualsPos == -1 )
		// no equals sign: incorrect or empty line, skip it
		  continue;
      
    // insert the key/value line into the current dictionary
	  QString *Entry = new QString;
	  (*Entry) = printableToString(CurrentLine.right( CurrentLine.length()-EqualsPos-1 )
			      ).stripWhiteSpace(); 
			      
    CurrentDict->insert( CurrentLine.left( EqualsPos ).stripWhiteSpace(), Entry );
    
  }
    
  delete Stream;
}


/**  */
void VVokFile::readData( VVocabulary *Vocabulary )
{
  QTextStream *Stream = new QTextStream( &File );
  
  while( !Stream->eof() )
  {
    Vocabulary->newVokabel( Stream->readLine() );    
  }
    
  delete Stream;
}



/**  */
bool VVokFile::isWriteable( const QString &Filename )
{
	File.setName( Filename );
	
	return File.open( IO_Truncate | IO_WriteOnly );    
}

	  
/**  */
void VVokFile::write( VVocabulary *Vocabulary )
{
  QTextStream* Stream = new QTextStream( &File );

  *Stream << "# DokVok File\n";
  
  // write back -- start with the default group
  VEntryDict* DefWriteGroup = GroupDict[ "<default>" ];
  if( DefWriteGroup )
	{
	  QDictIterator<QString> WriteInnerIt( *DefWriteGroup );
	  while( WriteInnerIt.current() )
		{
			*Stream << WriteInnerIt.currentKey() << "=" 
					 << stringToPrintable( WriteInnerIt.current()->data() ) << '\n';
		  ++WriteInnerIt;
		}
	}
  
  VGroupIterator WriteIt( GroupDict );
  while( WriteIt.current() )
	{
	  // check if it's not the default group (which has already been written)
	  if( strcmp (WriteIt.currentKey(), "<default>" ) )
		{
		  *Stream << '[' << WriteIt.currentKey() << ']' << '\n';
		  VEntryIterator WriteInnerIt( *WriteIt.current() );
		  while( WriteInnerIt.current() )
			{
				*Stream << WriteInnerIt.currentKey() << "="
						 << stringToPrintable( WriteInnerIt.current()->data()) << '\n';
			  ++WriteInnerIt;
			}
		}
	  ++WriteIt;
	}
  
	// write the vocabulary
  *Stream << "#Term TAB Translation TAB Typ TAB Knowledge TAB Remark TAB Mistakes\n[Vocabulary]\n";
  
  QString Line;
	for ( Line=Vocabulary->firstLine(); !Line.isEmpty(); Line=Vocabulary->nextLine() )
	  *Stream << Line << "\n";
  
  // clean up
  delete Stream;
}

void VVokFile::setGroup( const char* NewGroup )
{
  if( !NewGroup )
    Group = "<default>";
  else
    Group = NewGroup;
}


const char* VVokFile::group() const
{
  static QString EmptyStr = "";
  if( Group == "<default>" )
    return EmptyStr;
  else
    return Group;
}

const QString VVokFile::readEntry( const char* Key, const char* Default ) const
{
  QString Value;
  // retrieve the current group dictionary
  VEntryDict* CurrentDict = GroupDict[ Group ];
  
  if( CurrentDict )
  {
		QString *Entry = (*CurrentDict)[ Key ];
    if( Entry )
		  Value = *Entry;
    else if( Default )
		{
		  Value = Default;
		}
  }
  else if( Default )
  	Value = Default;
 
  return Value;
}


int VVokFile::readListEntry( const char* Key, QStrList &List, char Separator ) const
{
  if( !hasKey( Key ) )
    return 0;
  QString ListString, Value;
  ListString = readEntry( Key );
  if( ListString.isEmpty() )
    return 0; 
  List.clear();
  int i;
  Value = "";
  int Length = ListString.length();
  for( i = 0; i < Length; i++ )
  {
    if( ListString[i] != Separator && ListString[i] != '\\' )
  	{
	    Value += ListString[i];
	    continue;
   	}
    if( ListString[i] == '\\' )
  	{
	    i++;
	    Value += ListString[i];
  	  continue;
	  }
    List.append( Value );
    Value.truncate(0);
  }
  if ( ListString[Length-1] != Separator )
    List.append( Value );
  return List.count();
}


int VVokFile::readNumEntry( const char* Key, int Default) const
{
  bool ok;
  int rc;

  QString Value = readEntry( Key );
  if( Value.isNull() )
  	return Default;
  else if( Value == "true" )
	  return 1;
  else if( Value == "on" )
  	return 1;
  else
	{
	  rc = Value.toInt( &ok );
	  return( ok ? rc : 0 );
	}
}


bool VVokFile::readBoolEntry( const char* Key, const bool Default ) const
{
  QString Value = readEntry( Key );
  if( Value.isNull() )
	return Default;
  else
	{
	  if( Value == "true" || Value == "on" )
  		return true;
	  else
		{
		  bool OK;
		  int Val = Value.toInt( &OK );
		  if( OK && Val != 0 )
  			return true;
		  else
	  		return false;
		}
	}
}


bool VVokFile::hasKey( const char* Key ) const
{
  QString Value = readEntry( Key );
  return !Value.isNull();
}


void VVokFile::writeEntry( const char* Key, const char* Value )
{
  // retrieve the current group dictionary
  VEntryDict* CurrentDict = GroupDict[ Group ];

  if( !CurrentDict )
	{
	  // no such group -> create a new entry dictionary
	  VEntryDict* NewDict = new VEntryDict( 37, false );
	  NewDict->setAutoDelete( true );
	  GroupDict.insert( Group, NewDict );
	  
	  // this is now the current group
	  CurrentDict = NewDict;
	}

  // try to retrieve the current entry for this key
  QString *Entry = (*CurrentDict)[ Key ];
  if( !Entry )
	{
	  // the key currently does not exist
	  Entry = new QString;
    
	  // insert the new entry into group dictionary
	  CurrentDict->insert( Key, Entry );
	}
    
 (*Entry) = Value; // set new value
}
