/***************************************************************************
                          bookmarkmanager.cpp  -  description                              
                             -------------------                                         
    begin                : Fri Jul 16 1999                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <qstring.h>
#include <qheader.h>

#include <kapp.h>
#include <kconfig.h>
#include <kiconloader.h>

#include "resource.h"
#include "bmgroupproperties.h"
#include "bookmarkmanager.h"

BookmarkManager::BookmarkManager(QWidget *parent, const char *name) :
								QListView (parent, name)
{
	groupPix 			= ICON(ICON_GROUP);
	textPix 			= ICON(ICON_TEXT);
	dictionaryPix = ICON(ICON_DICTIONARY);
	commentaryPix = ICON(ICON_COMMENTARY);

	bookmarkPopup = new QPopupMenu();
	bookmarkPopup->insertItem(Icon("mini/mini-exclam.xpm"),i18n("Add Bookmark"),this, SLOT(slotAddBookmark()),0,ID_BOOKMARK_ADD);
	bookmarkPopup->insertItem(Icon("mini/mini-cross.xpm"),i18n("Delete Bookmark"),this, SLOT(slotDeleteBookmark()),0,ID_BOOKMARK_DELETE);
	bookmarkPopup->insertSeparator();
	bookmarkPopup->insertItem(Icon("mini/mini-exp.xpm"),i18n("New Group"),this, SLOT(slotNewGroup()),0,ID_BOOKMARK_GROUP_NEW);
	bookmarkPopup->insertItem(Icon("mini/mini-cross.xpm"),i18n("Delete Group"),this, SLOT(slotDeleteGroup()),0,ID_BOOKMARK_GROUP_DELETE);
	bookmarkPopup->insertItem(Icon("mini/mini-hammer.xpm"),i18n("Group Properties"),this, SLOT(slotGroupProperties()),0,ID_BOOKMARK_GROUP_PROPERTIES);
	bookmarkPopup->insertSeparator();
	bookmarkPopup->insertItem(Icon("cut.xpm"),i18n("Mark selected entry"),this, SLOT(slotMarkItem()),0,ID_BM_MARK);
	bookmarkPopup->insertItem(Icon("paste.xpm"),i18n("Move selected entry"),this, SLOT(slotMoveItem()),0,ID_BM_MOVE);
	
	connect( this, SIGNAL(rightButtonPressed ( QListViewItem *, const QPoint &, int )),
		this, SLOT( slotPopupMenu( QListViewItem *, const QPoint &, int ) ) );
	connect( this, SIGNAL( doubleClicked ( QListViewItem * ) ),
		this, SLOT( slotBookmarkChosen( QListViewItem * )));
	connect( this, SIGNAL( returnPressed ( QListViewItem * ) ),
		this, SLOT( slotBookmarkChosen( QListViewItem * )));
	
	currentKey 		= 0;
	currentModule = 0;
	
	groupList.setAutoDelete(true);
	bookmarkList.setAutoDelete(true);
	
	addColumn(i18n("Key"),-1);
	addColumn(i18n("Module"),-1);
	header()->hide();
	setRootIsDecorated(false);
	setSorting(-1, false);
  setFrameStyle( Panel | Sunken );
  setLineWidth( 2 );

	readOptions();
	
	QListViewItem *item = 0;
	if (groupList.count() == 0)
	{
		BookmarkInfo *bmInfo = new BookmarkInfo;
		bmInfo->item = new QListViewItem(this,i18n("Personal Bookmarks"));
		bmInfo->item->setPixmap(0, groupPix);
		bmInfo->type = Group;
		groupList.append( bmInfo );
	}
		
	item = groupList.at(0)->item;
	setCurrentItem(item);
	setSelected(item,true);
	setOpen(item, true);
	pasteBuffer.item = 0;
}


BookmarkManager::~BookmarkManager()
{
	saveOptions();
}

void BookmarkManager::readOptions()
{
	QListViewItem *parent				= 0;
	BookmarkInfo	*bmInfo				= 0;
	
	KConfig 			*config 			= kapp->getConfig();
	QStrList 			*parents 			= new QStrList;
	QStrList 			*names 				= new QStrList;
	QStrList 			*modules 			= new QStrList;
	QStrList 			*moduleTypes	= new QStrList;

	parents->setAutoDelete(true);
	names->setAutoDelete(true);
	modules->setAutoDelete(true);
	
	groupList.clear();
	bookmarkList.clear();
	config->setGroup("BookmarkManager");
	config->readListEntry("Group Parents"	,*parents);
	config->readListEntry("Group Names"		,*names);
	parents->first();
	names->first();
	while ((parents->current()) && (names->current()))
	{
		bmInfo = new BookmarkInfo;
		if (!qstrcmp(parents->current(),"-1"))
			bmInfo->item = new QListViewItem(this, names->current());
		else
		{
			parent = findParent(((QString) (parents->current())).toInt());
			if (parent)
				bmInfo->item = new QListViewItem(parent, names->current());
			else
				bmInfo->item = new QListViewItem(this, names->current());
		}
		bmInfo->item->setPixmap(0, groupPix);
		bmInfo->type = Group;
		groupList.append( bmInfo );
		parents->next();
		names->next();
	}
	// read bookmarks
	parents->clear();
	names->clear();
	config->readListEntry("Bookmark Parents"	,*parents);
	config->readListEntry("Bookmark Names"		,*names);
	config->readListEntry("Bookmark Modules"	,*modules);
	config->readListEntry("Module Types"			,*moduleTypes);
	parents->first();
	names->first();
	modules->first();
	moduleTypes->first();
	while ((parents->current()) && (names->current()) && (modules->current()) && (moduleTypes->current()))
	{
		bmInfo = new BookmarkInfo;
		parent = findParent(((QString) (parents->current())).toInt());
		if (parent)
			bmInfo->item = new QListViewItem(parent, names->current(), modules->current());
		else
			bmInfo->item = new QListViewItem(this, names->current(), modules->current());
	
		if (!qstrcmp(moduleTypes->current(),"0")) {
			bmInfo->item->setPixmap(0,textPix);
			bmInfo->type = Bible;
		}	else {
			if (!qstrcmp(moduleTypes->current(),"1")) {
				bmInfo->item->setPixmap(0,dictionaryPix);
				bmInfo->type = Dictionary;
			} else {
				bmInfo->item->setPixmap(0,commentaryPix);
				bmInfo->type = Commentary;
			}
		}
		bookmarkList.append( bmInfo );
		
		parents->next();
		names->next();
		modules->next();
		moduleTypes->next();
	}
}

void BookmarkManager::saveOptions()
{
	if (groupList.count() == 0)
		return;
		
	KConfig 	*config 			= kapp->getConfig();
	QStrList 	*parents 			= new QStrList;
	QStrList 	*names 				= new QStrList;
	QStrList 	*modules 			= new QStrList;
	QStrList 	*moduleTypes	= new QStrList;
	QString		tmp						= 0;
	parents->setAutoDelete(true);
	names->setAutoDelete(true);
	modules->setAutoDelete(true);

	groupList.first();
	config->setGroup("BookmarkManager");
	while (groupList.current())
	{
		tmp.setNum(parentId(&groupList, groupList.current()->item));
		parents->append(tmp.data());
		names->append(groupList.current()->item->text(0));
		groupList.next();
	}
	config->writeEntry("Group Parents",*parents);
	config->writeEntry("Group Names"	,*names);
	// write bookmarks
	parents->clear();
	names->clear();
	bookmarkList.first();
	
	while (bookmarkList.current())
	{
		tmp.setNum(parentId(&groupList, bookmarkList.current()->item));
		parents->append(tmp.data());
		names->append(bookmarkList.current()->item->text(0));
		modules->append(bookmarkList.current()->item->text(1));
		if (bookmarkList.current()->type == Bible)
			moduleTypes->append("0");
		else
			if (bookmarkList.current()->type == Dictionary)
				moduleTypes->append("1");
			else
				if (bookmarkList.current()->type == Commentary)
					moduleTypes->append("2");
				else
					debug("wrong type in item!");
				
		bookmarkList.next();
	}
	config->writeEntry("Bookmark Parents"	,*parents);
	config->writeEntry("Bookmark Names"		,*names);
	config->writeEntry("Bookmark Modules"	,*modules);
	config->writeEntry("Module Types"			,*moduleTypes);
	
	delete parents;
	delete names;
	delete modules;
	delete moduleTypes;
}

QString BookmarkManager::parentName(QListViewItem *item)
{
	// search parent of item and return the Group name
	QString ret("#NULL#");
	if (item) {
		QListViewItem *parent = item->parent();
		if (parent)
			ret.setStr(parent->text(0));
	}
	return ret;
}

int BookmarkManager::parentId(BookmarkItemList *list, QListViewItem *item)
{
	// search parent of item and return the Group name
	int ret = -1;
	
	if (item) {
		QListViewItem *parent = item->parent();
		if (parent) {
			int cur = list->at();
			list->first();
			while ((list->current()) && (list->current()->item != parent)) list->next();
			if (list->current())
				ret = list->at();
			if (cur >= 0) list->at(cur);
		}
	}
	return ret;
}

QListViewItem *BookmarkManager::findParent(int id)
{
	int cur = groupList.at();
	QListViewItem *ret = groupList.at((uint) id)->item;
	groupList.at(cur);
	return ret;
}

void BookmarkManager::slotPopupMenu( QListViewItem *p, const QPoint &pnt, int )
{
	bool isGroupItem = false;
	if (p) {
		setSelected(p,true);
		groupList.first();
		while (groupList.current()) {
			if (groupList.current()->item == p) {
				isGroupItem = true;
				break;
			} else groupList.next();
		}
 		bookmarkPopup->setItemEnabled(ID_BOOKMARK_ADD,false);
 		bookmarkPopup->setItemEnabled(ID_BOOKMARK_DELETE,false);
 		bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_NEW,false);
		bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_DELETE,false);
 		bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_PROPERTIES,false);
 		bookmarkPopup->setItemEnabled(ID_BM_MOVE,false);
 		bookmarkPopup->setItemEnabled(ID_BM_MARK,false);
		
		if (isGroupItem) {
  		bookmarkPopup->setItemEnabled(ID_BOOKMARK_ADD,true);
  		bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_NEW,true);
  		bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_PROPERTIES,true);
  		if ((pasteBuffer.item != 0) && (p != pasteBuffer.item) && (!isChild(pasteBuffer.item,p)))
  			bookmarkPopup->setItemEnabled(ID_BM_MOVE,true);
  		if ((!p->childCount()) && (groupList.count() > 1))
  			bookmarkPopup->setItemEnabled(ID_BOOKMARK_GROUP_DELETE,true);
		} else {
  		bookmarkPopup->setItemEnabled(ID_BOOKMARK_DELETE,true);
  	}
  	
  	if (p != groupList.at(0)->item)
 			bookmarkPopup->setItemEnabled(ID_BM_MARK,true);
  	
		bookmarkPopup->exec(pnt);
	}
}

void BookmarkManager::slotAddBookmark()
{
	BookmarkInfo	*bmInfo = new BookmarkInfo;
	
	if (currentItem())
	{
		groupList.first();
		while (groupList.current())
		{
			if (groupList.current()->item == currentItem())
			{
				bmInfo->item = new QListViewItem(currentItem(), currentKey.data(), currentModule.data());
				bookmarkList.append( bmInfo );
				setOpen(currentItem(), true);
				break;
			} else
				groupList.next();
		}
	}
	else
	{
		bmInfo->item = new QListViewItem(this, currentKey.data(), currentModule.data());
		bookmarkList.append( bmInfo );
	}
	bmInfo->item->setPixmap(0, *currentPixmap);		
	if (currentPixmap == &textPix)
		bmInfo->type = Bible;
	else
		if (currentPixmap == &dictionaryPix)
			bmInfo->type = Dictionary;
		else
			bmInfo->type = Commentary;
			
}

void BookmarkManager::slotDeleteBookmark()
{
	// this entry delete is only enabled for bookmarks
	if (currentItem())
	{
		bookmarkList.first();
		while (bookmarkList.current())
		{
			if (bookmarkList.current()->item == currentItem())
			{
				bookmarkList.remove();
				break;
			} else bookmarkList.next();
		}
		delete currentItem();
	}
}

void BookmarkManager::slotNewGroup()
{
	QListViewItem *item = 0;
	BookmarkInfo	*bmInfo = 0;
	
	if (currentItem())
	{
		groupList.first();
		while (groupList.current())
		{
			if (groupList.current()->item == currentItem())
			{
				bmInfo = new BookmarkInfo;
				bmInfo->item = item = new QListViewItem(currentItem(), i18n("new Group"));
				bmInfo->type = Group;
				groupList.append( bmInfo );
				setOpen(currentItem(),true);
				break;
			} else
				groupList.next();
		}
	}	else {
		bmInfo = new BookmarkInfo;
		bmInfo->item = item = new QListViewItem(this, i18n("new Group"));
		groupList.append( bmInfo );
	}
	if (item)
	{
		item->setPixmap(0, groupPix);
		setCurrentItem(item);
		setSelected(item, true);
		slotGroupProperties();
	}
}

void BookmarkManager::slotDeleteGroup()
{
	// the menu entry delete is only enabled for empty groups
	if (groupList.count() < 2)
		return;
		
	
	groupList.first();
	while (groupList.current())
	{
		if (groupList.current()->item == currentItem())
		{
			groupList.remove();
			break;
		} else groupList.next();
	}
	delete currentItem();
}

void BookmarkManager::slotGroupProperties()
{
	if (!currentItem())
		return;
		
	BMGroupProperties *dlg = new BMGroupProperties();
	dlg->slotSetGroupName((QString)currentItem()->text(0));
	if (dlg->exec() == QDialog::Accepted)
	{
		currentItem()->setText(0,dlg->groupName().data());
		// Maybe resize column
  	currentItem()->widthChanged ( 0 );
  	triggerUpdate();
	}
	delete dlg;
}

void BookmarkManager::slotMarkItem()
{
	if (currentItem())
	{
		restoreMarkedItem();
		if (currentItem() == pasteBuffer.item)
		{
			pasteBuffer.item = 0;
			return;
		}
		// fill the new pasteBuffer
		pasteBuffer.type = Bookmark;
		groupList.first();
		while (groupList.current())
		{
			if (groupList.current()->item == currentItem())
			{
				pasteBuffer.type = Group;
				break;
			} else groupList.next();
		}
		QPixmap tmp = Icon(ICON_CUT);
		pasteBuffer.item = currentItem();
		currentItem()->setPixmap(0,tmp);
		// mark that item visibly
	}
}

void BookmarkManager::slotMoveItem()
{
	if ((!currentItem()) || (!pasteBuffer.item))
		return;
		
	// restore marked entry
	restoreMarkedItem();
	
	// traverse through subtree (could be one item or whole tree
	traverseTree(currentItem(), pasteBuffer.item);
	setOpen(currentItem(), true);
	pasteBuffer.item = 0;
}

void BookmarkManager::restoreMarkedItem()
{
	if (!pasteBuffer.item)
		return;
		
	if (pasteBuffer.type == Group)
	{
		pasteBuffer.item->setPixmap(0, groupPix);
	}	else {
		bookmarkList.first();
		while (bookmarkList.current())
		{					
			if (bookmarkList.current()->item == pasteBuffer.item)
			{
				if (bookmarkList.current()->type == Bible)
					bookmarkList.current()->item->setPixmap(0, textPix);
				else
					if (bookmarkList.current()->type == Dictionary)
						bookmarkList.current()->item->setPixmap(0, dictionaryPix);
					else
						bookmarkList.current()->item->setPixmap(0, commentaryPix);
				break;
			} else bookmarkList.next();
		}
	}
}

bool BookmarkManager::isChild( QListViewItem *par, QListViewItem *item )
{
	bool flagIsChild = false;
	QListViewItem *parentOfItem = item->parent();
	while (parentOfItem) {
		if (parentOfItem == par) {
			flagIsChild = true;
			break;
		} else
			parentOfItem = parentOfItem->parent();
	}
	return flagIsChild;
}

void BookmarkManager::traverseTree( QListViewItem *cur, QListViewItem *par )
{
	// get info for the current item we process
	BookmarkInfo 			*bmInfo = getBookmarkInfo(par);
	BookmarkItemList 	*list 	= 0;
	if (bmInfo->type == Group)
		list = &groupList;
	else
		list = &bookmarkList;
	
	// build a new par item with parent cur
	BookmarkInfo *newItem = new BookmarkInfo;
	newItem->item = new QListViewItem(cur, bmInfo->item->text(0), bmInfo->item->text(1));
	const QPixmap *pix = bmInfo->item->pixmap(0);
	if (pix)
		newItem->item->setPixmap(0,*pix);
	newItem->type = bmInfo->type;
	list->append( newItem );
			
	// get a list of all first level children
	QListViewItem *child = par->firstChild();
	while( child ) {
		// debug("child is: %s (%s)",child->text(0),child->text(1));
		// traverse down that possible subtree
		traverseTree(newItem->item, child);
		child = child->nextSibling();
	}
	// no children left, we can delete this item
	list->first();
	while (list->current())
	{
		if (list->current()->item == par)
		{
			list->remove();
			break;
		} else list->next();
	}
	delete par;
}

BookmarkInfo *BookmarkManager::getBookmarkInfo( QListViewItem *child )
{
	// search groupList first
	groupList.first();
	while ((groupList.current()) && (groupList.current()->item != child)) groupList.next();
	if (groupList.current())
		return groupList.current();
		
	bookmarkList.first();
	while ((bookmarkList.current()) && (bookmarkList.current()->item != child)) bookmarkList.next();
	if (bookmarkList.current())
		return bookmarkList.current();
	else
	{
		debug("getBookmarkInfo: internal error: unreferenced child!");
		return 0;
	}
}

void BookmarkManager::slotBookmarkChosen( QListViewItem *i )
{
	// debug("choose %s from %s",i->text(0),i->text(1));
	// check if item is a bookmark item
	bookmarkList.first();
	while (bookmarkList.current())
	{
		if (bookmarkList.current()->item == i)
		{
			QString module(i->text(1));
			QString key(i->text(0));
			emit sigBookmarkChosen(module, key);
			break;
		} else bookmarkList.next();
	}
}

void BookmarkManager::slotSetCurrentKeyModule(const QString &t, const QString &m, const QString &type)
{
	currentKey 		= t;
	currentModule = m;
	currentType 	= type;
	
	if (currentType.contains("biblical texts",false))
		currentPixmap = &textPix;
	else
		if (currentType.contains("lexicons / dictionaries",false))
			currentPixmap = &dictionaryPix;
		else
			currentPixmap = &commentaryPix;
		
	// debug("BookmarkManager: %s (%s)",currentKey.data(),currentModule.data());
}





