/*

    File: pref.cpp
    Version: 0.3.1
    Date: 31-DEC-1998
    Copyright: Bernhard Kuhn <kuhn@lpr.e-technik.tu-muenchen.de>

    This is part of "ksermon". The preference widget provides a
    GUI to select input device, maximum display rate
    and update interval.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <iostream.h>


#include <qlabel.h>
#include <qbttngrp.h> 
#include <qchkbox.h>
#include <qpushbt.h> 
#include <qmsgbox.h> 
#include <qlayout.h> 

#include "pref.h"

#include <kapp.h>
#include <klocale.h>



// define strings for combo-boxes
const char* devices[]={"/dev/ttyS0","/dev/ttyS1",
		       "/dev/cua0","/dev/cua1",0};
const char* rates[]={"5000","100","200","500","1000","2000","5000",
		     "10000","20000",0};
const char* intervals[]={"100","20","50","100","200",0};
const char* taus[]={"500","20","50","100","200","500","1000","2000",0};



// constructor

pref::pref(QWidget* parent=0,char* name="") :
  QFrame(parent,name) {

  // device selection 
  QLabel* deviceLabel = new QLabel(klocale->translate("Device"),this);
  deviceCombo = new QComboBox(TRUE,this);
  deviceCombo->insertStrList(devices);
  deviceCombo->setInsertionPolicy(QComboBox::AtTop);

  // separator
  QFrame* separator1 = new QFrame(this);
  separator1->setFrameStyle(Panel|Sunken);

  // maximum display rate selection
  QLabel* rateLabel1 = new QLabel(klocale->translate("Maximum"),this);
  rateCombo = new QComboBox(TRUE,this);
  // rateCombo->setAutoResize(TRUE);
  rateCombo->insertStrList(rates);
  rateCombo->setInsertionPolicy(QComboBox::NoInsertion);
  rateCombo->setSizeLimit(11);
  QLabel* rateLabel2 = new QLabel("[cps]",this);
  rateSlider = new QSlider(100,20000,100,5000,QSlider::Horizontal,this);
  rateSlider->setTickmarks((QSlider::TickSetting)1);
  rateSlider->setTickInterval(1000);

  // separator
  QFrame* separator2 = new QFrame(this);
  separator2->setFrameStyle(Panel|Sunken);

  // update interval
  QLabel* intervalLabel1 = new QLabel(klocale->translate("Interval"),this);
  intervalCombo = new QComboBox(TRUE,this);
  intervalCombo->insertStrList(intervals);
  intervalCombo->setInsertionPolicy(QComboBox::NoInsertion);
  intervalCombo->setSizeLimit(11);
  QLabel* intervalLabel2 = new QLabel("[ms]",this);
  intervalSlider = new QSlider(20,200,20,100,QSlider::Horizontal,this);
  intervalSlider->setTickmarks((QSlider::TickSetting)1);
  intervalSlider->setTickInterval(10);

  // separator
  QFrame* separator3 = new QFrame(this);
  separator3->setFrameStyle(Panel|Sunken);

  // RC-lowpass timing constant
  QLabel* tauLabel1 = new QLabel(klocale->translate("Tau"),this);
  tauCombo = new QComboBox(TRUE,this);
  tauCombo->insertStrList(taus);
  tauCombo->setInsertionPolicy(QComboBox::NoInsertion);
  tauCombo->setSizeLimit(11);
  QLabel* tauLabel2 = new QLabel("[ms]",this);
  tauSlider = new QSlider(20,2000,100,500,QSlider::Horizontal,this);
  tauSlider->setTickmarks((QSlider::TickSetting)1);
  tauSlider->setTickInterval(100);

  // separator
  QFrame* separator4 = new QFrame(this);
  separator4->setFrameStyle(Panel|Sunken);

  // "save" and "ok" buttons
  QPushButton* save = new QPushButton(klocale->translate("Save"),this);
  QPushButton* ok = new QPushButton(klocale->translate("Ok"),this);


  // place all elements

#ifdef AUTOLAYOUT

  QVBoxLayout* mainLayout=new QVBoxLayout(this,10);

  QHBoxLayout* deviceLayout=new QHBoxLayout(10);
  mainLayout->addLayout(deviceLayout);
  deviceLayout->addWidget(deviceLabel,1);
  deviceLayout->addWidget(deviceCombo,2);
  // deviceLayout->addStretch(1);

  mainLayout->addWidget(separator1);

  QVBoxLayout* maximumLayout=new QVBoxLayout(10);
  mainLayout->addLayout(maximumLayout);
  QHBoxLayout* maxcomboLayout=new QHBoxLayout();
  maximumLayout->addLayout(maxcomboLayout);
  maxcomboLayout->addWidget(rateLabel1,1);
  maxcomboLayout->addWidget(rateCombo,1);
  maxcomboLayout->addWidget(rateLabel2,1);
  maximumLayout->addWidget(rateSlider);

  mainLayout->addWidget(separator2);

  QVBoxLayout* intervalLayout=new QVBoxLayout(10);
  mainLayout->addLayout(intervalLayout);
  QHBoxLayout* intcomboLayout=new QHBoxLayout();
  intervalLayout->addLayout(intcomboLayout);
  intcomboLayout->addWidget(intervalLabel1,1);
  intcomboLayout->addWidget(intervalCombo,1);
  intcomboLayout->addWidget(intervalLabel2,1);
  intervalLayout->addWidget(intervalSlider);

  mainLayout->addWidget(separator3);
 
  QHBoxLayout* oksaveLayout=new QHBoxLayout(10);
  mainLayout->addLayout(oksaveLayout);
  oksaveLayout->addWidget(ok,1);
  oksaveLayout->addStretch(1);
  oksaveLayout->addWidget(save,1);

#else

  deviceLabel->setGeometry(20,20,60,20);
  deviceCombo->setGeometry(80,20,150,20);

  separator1->setGeometry(10,59,230,2);  

  rateLabel1->setGeometry(20,80,60,20);
  rateCombo->setGeometry(80,80,80,20);
  rateLabel2->setGeometry(180,80,60,20);
  rateSlider->setGeometry(20,110,210,20);

  separator2->setGeometry(10,149,230,2);  

  intervalLabel1->setGeometry(20,170,60,20);
  intervalCombo->setGeometry(80,170,80,20);
  intervalLabel2->setGeometry(180,170,60,20);
  intervalSlider->setGeometry(20,200,210,20);

  separator3->setGeometry(10,239,230,2);  

  tauLabel1->setGeometry(20,260,60,20);
  tauCombo->setGeometry(80,260,80,20);
  tauLabel2->setGeometry(180,260,60,20);
  tauSlider->setGeometry(20,290,210,20);

  separator4->setGeometry(10,329,230,2);  
  
  save->setGeometry(150,350,80,25);
  ok->setGeometry(20,350,80,25);

#endif  


  // connect elements to slots

  connect(deviceCombo,SIGNAL(activated(const char*)),
	  this,SLOT(deviceComboAktivated(const char*)));
  connect(rateCombo,SIGNAL(activated(const char*)),
	  this,SLOT(rateComboAktivated(const char*)));
  connect(rateSlider,SIGNAL(valueChanged(int)),
	  this,SLOT(rateSliderValueChanged(int)));
  connect(intervalCombo,SIGNAL(activated(const char*)),
	  this,SLOT(intervalComboAktivated(const char*)));
  connect(intervalSlider,SIGNAL(valueChanged(int)),
	  this,SLOT(intervalSliderValueChanged(int)));
  connect(tauCombo,SIGNAL(activated(const char*)),
	  this,SLOT(tauComboAktivated(const char*)));
  connect(tauSlider,SIGNAL(valueChanged(int)),
	  this,SLOT(tauSliderValueChanged(int)));
  connect(save,SIGNAL(released()),
	  this,SLOT(savePressed()));
  connect(ok,SIGNAL(released()),
	  this,SLOT(okPressed()));


  // appearence of preferences

  setCaption("KSermon Preferences");
  resize(250,405);
  setMinimumSize(250,390);
  setMaximumSize(250,390);

};



// device selected

void pref::deviceComboAktivated(const char* device) {
  emit deviceChanged(device);
};



// maximum selected by combo-box (invokes sliders-selection)

void pref::rateComboAktivated(const char* rate) {
  rateSlider->setValue(QString(rate).toInt());
};



// maximum selected by slider

void pref::rateSliderValueChanged(int rate) {

  rate=100*((rate+50)/100);

  if(rate<100) rate=100;
  if(rate>20000) rate=20000;
  
  rateSliderText.setNum(rate);
    
  rateCombo->changeItem(rateSliderText.data(),0);
  rateCombo->setCurrentItem(0);

  emit rateChanged((float)rate);
};



// interval selected by combo-box (invokes selection by slider)

void pref::intervalComboAktivated(const char* interval) {
  intervalSlider->setValue(QString(interval).toInt());
};



// interval selected by slider

void pref::intervalSliderValueChanged(int interval) {

  interval=10*((interval+5)/10);

  if(interval<20) interval=20;
  if(interval>200) interval=200;
  
  intervalSliderText.setNum(interval);
    
  intervalCombo->changeItem(intervalSliderText.data(),0);
  intervalCombo->setCurrentItem(0);

  emit intervalChanged(interval);
};



// tau selected by combo-box (invokes selection by slider)

void pref::tauComboAktivated(const char* tau) {
  tauSlider->setValue(QString(tau).toInt());
};



// tau selected by slider

void pref::tauSliderValueChanged(int tau) {

  tau=10*((tau+5)/10);

  if(tau<20) tau=20;
  if(tau>2000) tau=2000;
  
  tauSliderText.setNum(tau);
    
  tauCombo->changeItem(tauSliderText.data(),0);
  tauCombo->setCurrentItem(0);

  emit tauChanged(tau);
};



// "ok" button Pressed

void pref::savePressed() {
  hide();
  emit doSave();
};



// "save" button Pressed

void pref::okPressed() {
  hide();
  emit doOk();
};



// device was changed externaly

void pref::deviceChangedIn(const char* device) {
  int i=0;

  const char* dev_list_str;

  while((dev_list_str=devices[i])!=0) {

    if(strcmp(device,devices[i])==0) {
      deviceCombo->setCurrentItem(i);
      deviceComboAktivated(device);
      return;
    };  
    
    i++;
  };

  deviceCombo->insertItem(device);
  deviceCombo->setCurrentItem(deviceCombo->count()-1);
  deviceComboAktivated(device);

};



// maximum was changed externaly

void pref::rateChangedIn(float rate) {
  rateSlider->setValue((int)rate);
};



// interval was changed externaly

void pref::intervalChangedIn(int interval) {
  intervalSlider->setValue(interval);
};



// tau was changed externaly

void pref::tauChangedIn(int tau) {
  tauSlider->setValue(tau);
};







