/* -*- C -*-
 * This source file is part of the libinetd package.
 * 
 * Copyright (C) 1988 Jrgen Sigvardsson / jorgen@linux.nu
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA
 *
 * $Id: inetd.h,v 1.11 1998/05/28 18:39:46 jorgen Exp $
 * $Log: inetd.h,v $
 * Revision 1.11  1998/05/28 18:39:46  jorgen
 * inetd_AddService semantics changed. inetd_Find, inetd_RemoveService semantics
 * also changed. Added support for comment-meta information in inetd.conf.
 *
 * Revision 1.10  1998/05/24 01:29:43  jorgen
 * Added inetd_CreateEmptyConf()
 *
 * Revision 1.9  1998/05/23 18:20:34  jorgen
 * Moved the inetd_XXXName() macros to inetd.h
 *
 * Revision 1.8  1998/05/23 18:18:58  jorgen
 * Renamed structs xxx to xxx_ in order to keep g++ happy. (G++ did not like
 * the forward references)
 *
 * Revision 1.7  1998/05/23 13:50:24  jorgen
 * More bugfixes - inetd_CurrentService() fixed.
 *
 * Revision 1.6  1998/05/23 13:11:27  jorgen
 * Added inetd_RemoveService() and changed the semantics for inetd_AddService().
 *
 * Revision 1.5  1998/05/23 11:53:41  jorgen
 * Code documentation done.
 *
 * Revision 1.4  1998/05/23 10:55:51  jorgen
 * Beta-ready
 *
 * Revision 1.3  1998/05/22 19:06:52  jorgen
 * inetd_Find() and inetd_Free() now works.
 *
 * Revision 1.2  1998/05/22 18:53:55  jorgen
 * inetd_ReadConf() now reads in /etc/inetd.conf and inetd_Flush() now flushes
 * the inetd information correctly.
 *
 * Revision 1.1.1.1  1998/05/22 17:46:22  jorgen
 * Initial version
 *
 */

#ifndef __INETD_H__
#define __INETD_H__

/* Constants and macros */
#define SOCKTYPE_UNKNOWN 0
#define SOCKTYPE_STREAM  1
#define SOCKTYPE_DGRAM   2

#define PROTO_UNKNOWN    0
#define PROTO_UDP        1
#define PROTO_TCP        2
#define PROTO_RPCUDP     3
#define PROTO_RPCTCP     4

#define FLAG_UNKNOWN     0
#define FLAG_WAIT        1
#define FLAG_NOWAIT      2

#define inetd_FlagName(x) ((x) == FLAG_WAIT ? "wait" : ((x) == FLAG_NOWAIT ? "nowait" : "unknown"))
#define inetd_SockTypeName(x) ((x) == SOCKTYPE_STREAM ? "stream" : ((x) == SOCKTYPE_DGRAM ? "dgram" : "unknown" ))
#define inetd_ProtoName(x)                              \
	((x) == PROTO_UDP ? "udp" : 			\
	 ((x) == PROTO_TCP ? "tcp" : 			\
	  ((x) == PROTO_RPCUDP ? "rpc/udp" : 		\
	   ((x) == PROTO_RPCTCP ? "rpc/tcp" : "unknown"	\
	    )						\
	   )						\
	  ))

#define inetd_OnOff(x) ((x) == 0 ? "off" : "on")

#define OFFMARKER "#off:"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Forward type references */
struct inetd_h_;
typedef struct inetd_h_* inetd_h;

struct inetd_Service_;
typedef struct inetd_Service_* inetd_Service;

/* 
 * Opens the inetd config file.
 * Opens /etc/inetd.conf and reads it into a 
 * datastructure in memory. A handle to that
 * datastructure is returned.
 *
 * Returns: an inetd config handle
 *
 * Author: Jrgen Sigvardsson
 */
inetd_h inetd_ReadConf();

/*
 * Creates an empty configuration.
 * No services are associated with the newly created 
 * inetd handle.
 *
 * Returns: A new handle. NULL on error.
 */
inetd_h inetd_CreateEmptyConf();

/*
 * Flushes an inetd configuration to file.
 * The config-content referenced to by handle
 * is written to the file filename.
 *
 * Precondition: handle and filename must NOT
 * be NULL.
 *
 * Returns: 0 on success, -1 on failure. On 
 * failure test errno for further information.
 *
 * Warning: The file will be truncated if it
 * alredy exists.
 *
 * Author: Jrgen Sigvardsson
 */
int inetd_Flush(
		inetd_h handle /* inetd handle */, 
		const char* filename /* name of file */
		);

/* 
 * Free's the memory occupied by the handle.
 * Precondition: handle must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void inetd_Free(inetd_h handle /* inetd handle */ );

/*
 * Looks up a service.
 * Searches the inetd configuration for a service named
 * servname which uses the socket type socktype.
 * Will find the first occurence of servname|socktype.
 * 
 * Precondition: handle must NOT be NULL.
 * 
 * Returns: A non-NULL value on success pointing to a
 * service. On failure NULL is returned.
 *
 * Warning: The return value points to a datastructure
 * inside the inetd-configuration. Do NOT free it or
 * manipulate it directly. Use the serv_XXX functions
 * for manipulation.
 *
 * Author: Jrgen Sigvardsson
 */
inetd_Service inetd_Find(
			 /* inetd handle */
			 inetd_h handle, 
			 /* service name */
			 const char* servname, 
			 /* socket type */
			 int socktype
			 );

/* 
 * Adds a service to the inetd configuration.
 * 
 * Precondition: handle and service must NOT be NULL.
 *
 * Warning: The handle will assume ownership of the
 * service after it has been added to the inetd-config.
 * Do NOT free the service after it has been added.
 * This function also allows adding duplicates!
 *
 * Author: Jrgen Sigvardsson
 */
void inetd_AddService(
		      /* inetd handle */
		      inetd_h handle, 
		      /* The new service */
		      inetd_Service service
		      );

/*
 * Removes a service from the inetd configuration.
 *
 * Precondition: handle and service must NOT be NULL and
 * inetdconfig must contain the service.
 *
 * Warning: The service will be free()'d after the call to
 * inetd_RemoveService(). service will be a _dangling pointer_.
 * The function removes the first occurence of servicename|socket.
 *
 * Author: Jrgen Sigvardsson
 */
void inetd_RemoveService(
			 /* inetd handle */
			 inetd_h handle,
			 /* the service */
			 inetd_Service service
			 );

/*
 * Prepares a handle for iteration.
 * 
 * Precondition: handle must NOT be NULL.
 *
 * Warning: A call to inetd_BeginIter will reset
 * previous iteration prepatations on the same handle.
 *
 * Author: Jrgen Sigvardsson
 */
void inetd_BeginIter(
		     /* inetd handle */
		     inetd_h handle
		     );

/*
 * Returns the current service.
 * Called during iteration of an inetd configuration.
 *
 * Precondition: handle must NOT be NULL and that
 * inetd_BeginIter() has been called earlier.
 *
 * Returns: The current service. If the iteration has 
 * hit the end, NULL will be returned.
 *
 * Author: Jrgen Sigvardsson
 */
inetd_Service inetd_CurrentService(
				   /* inetd handle */
				   inetd_h handle
				   );

/*
 * Advances current service to next service.
 * Called during iteration of an inetd configuration.
 *
 * Precondition: handle must NOT be NULL &&
 * inetd_CurrentService(handle) != NULL
 *
 * Author: Jrgen Sigvardsson
 */
void inetd_Iterate(
		   /* inetd handle */
		   inetd_h handle
		   );

/*
 * Returns the number of arguments associated with the
 * service.
 *
 * Precondition: service and handle must NOT be NULL.
 * 
 * Returns: number of arguments
 *
 * Author: Jrgen Sigvardsson
 */
int serv_GetNumArgs(
		    /* The service */
		    inetd_Service service 
		    );
/* 
 * Returns the n'th argument associated with the service.
 * Note: The first argument is the 0'th argument.
 *
 * Precondition: service must NOT be NULL and
 * n must be less than serv_GetNumArgs(service).
 *
 * Returns: the argument - a string
 *
 * Warning: The string wich is returned is internal to
 * the service. Do NOT modify it in any way.
 *
 * Author: Jrgen Sigvardsson
 */
const char* serv_GetArg(
			/* the service */
			inetd_Service service, 
			/* which argument to get */
			int n);

/* 
 * Returns the service name associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the service name - a string
 *
 * Warning: The string wich is returned is internal to
 * the service. Do NOT modify it in any way.
 *
 * Author: Jrgen Sigvardsson
 */
const char* serv_GetName(
			 /* the service */
			 inetd_Service service
			 );
/* 
 * Returns the server path associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the server path - a string
 *
 * Warning: The string wich is returned is internal to
 * the service. Do NOT modify it in any way.
 *
 * Author: Jrgen Sigvardsson
 */
const char* serv_GetPath(
			 /* the service */
			 inetd_Service service
			 );
/* 
 * Returns the user associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the user - a string
 *
 * Warning: The string wich is returned is internal to
 * the service. Do NOT modify it in any way.
 *
 * Author: Jrgen Sigvardsson
 */
const char* serv_GetUser(
			 /* the service */
			 inetd_Service service
			 );

/* 
 * Returns the socket type associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the socket type. Possible return values are
 * SOCKTYPE_STREAM, SOCKTYPE_DGRAM or SOCKTYPE_UNKNOWN.
 *
 * Author: Jrgen Sigvardsson
 */
int serv_GetSockType(
		     /* the service */
		     inetd_Service service
		     );
/* 
 * Returns the protocol associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the protocol type. Possible return values are
 * PROTO_UDP, PROTO_TCP, PROTO_RPCUDP, PROTO_RPCTCP or
 * PROTO_UNKNOWN.
 *
 * Author: Jrgen Sigvardsson
 */
int serv_GetProto(
		  /* the service */
		  inetd_Service service
		  );

/* 
 * Returns the socket flag associated with the service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Returns: the socket type. Possible return values are
 * FLAG_WAIT, FLAG_NOWAIT or FLAG_UNKNOWN.
 *
 * Author: Jrgen Sigvardsson
 */
int serv_GetFlags(
		  /* the service */
		  inetd_Service service
		  );

/*
 * Removes all arguments associated with the service.
 * 
 * Precondition:  service must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_RemoveArgs(
		     /* the service */
		     inetd_Service service
		     );
/* 
 * Adds an argument to the service. The argument is
 * appended.
 *
 * Precondition: service and argument must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_AddArg(
		 /* the service */
		 inetd_Service service, 
		 /* the argument */
		 const char* argument
		 );
/*
 * Sets the service name of the service.
 *
 * Precondition: service and name must NOT be NULL.
 *
 * Warning: The service will not assume ownership of the
 * string after it has been set. The service will make
 * a copy to it instead. You may free the string after
 * setting it.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetName(
		  /* the service */
		  inetd_Service service, 
		  /* the new service name */
		  const char* name
		  );
/*
 * Sets the server path of the service.
 *
 * Precondition: service and path must NOT be NULL.
 *
 * Warning: The service will not assume ownership of the
 * string after it has been set. The service will make
 * a copy to it instead. You may free the string after
 * setting it.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetPath(
		  /* the service */
		  inetd_Service service, 
		  /* the server path */
		  const char* path
		  );
/*
 * Sets the user of the service.
 *
 * Precondition: service and user must NOT be NULL.
 *
 * Warning: The service will not assume ownership of the
 * string after it has been set. The service will make
 * a copy to it instead. You may free the string after
 * setting it.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetUser(
		  /* the service */
		  inetd_Service service, 
		  /* the user */
		  const char* user
		  );
/* Sets the socket type of a service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetSockType(
		      /* the service */
		      inetd_Service service, 
		      /* the socket type, possible values
		       * SOCKTYPE_STREAM or SOCKTYPE_DGRAM
		       */
		      int socktype);
/* Sets the protocol type of a service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetProto(
		   /* the service */
		   inetd_Service service, 
		   /* the protocol type, possible values
		    * PROTO_UDP, PROTO_TCP, PROTO_RPCUDP
		    * or PROTO_RPCTCP
		    */
		   int proto
		   );

/*
 * Sets the flags of a service.
 *
 * Precondition: service must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetFlags(
		   /* the service */
		   inetd_Service service, 
		   /* the flag, possible values
		    * FLAG_WAIT or FLAG_NOWAIT
		    */
		   int flag
		   );

/* 
 * Turns a service on or off.
 * An onoff value of zero turns the service off. Any nonzer value
 * turns the service on.
 *
 * Precondition: service must NOT be NULL.
 *
 * Author: Jrgen Sigvardsson
 */
void serv_SetOnOff(
		  /* the service */
		  inetd_Service service,
		  /* 0 for off, nonzero for on */
		  int onoff
		  );

/* 
 * Determines if the specified service is turned off or not.
 *
 * Precondition: Service must NOT be NULL.
 *
 * Returns: Nonzero if service is on. Zero if it is on.
 *
 * Author: Jrgen Sigvardsson
 */
int serv_IsOff(
	       /* the service */
	       inetd_Service service
	       );


/* 
 * Creates an empty service.
 * All service fields are NULL and must be initialized with
 * the serv_XXX functions.
 *
 * Returns: An empty service.
 *
 * Author: Jrgen Sigvardsson
 */
inetd_Service serv_Create(void);

#ifdef __cplusplus
} /* extern "C" { */
#endif /* __cplusplus */

#endif /* __INETD_H__ */
