// Copyright 2010 Jens-Michael Hoffmann <jmho@c-xx.com>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library. If not, see <http://www.gnu.org/licenses/>.

#ifndef MARBLE_TEXTURETILE_H
#define MARBLE_TEXTURETILE_H

#include <QtCore/QDateTime>
#include <QtGui/QImage>

#include "TileId.h"

class QImage;

namespace Marble
{
class Blending;

/*!
    \class TextureTile
    \brief A class that resembles a texture tile .

    A texture tile provides a bitmap image for a certain (geographic) area and 
    for a given zoom level. Each TextureTile can be identified via a unique 
    TileId.
    
    A stack of TextureTiles that cover the same area and the same 
    zoom level can be stored (and painted) layer by layer in a StackedTile object. 
    For this purpose each TextureTile specifies a blending type. 

    Usually the tiles are organized in so called quad tiles: This means that
    with increasing zoom level four other tiles cover the same area as a 
    single "parent" tile in the previous zoom level. These four tiles have 
    the same pixel size as the "parent" tile.
    
    The process of "filling the tile with data is done in stages: The 
    State describes the current progress of loading the data (Empty, Scaled, 
    Expired, StateUptodate).
    
    The life time cycle of a TextureTile can also be influenced by its
    expiration time which will trigger a reload of the tile data.
*/

class TextureTile
{
 public:
    TextureTile( TileId const & tileId, QImage const & image, const Blending * blending );
    ~TextureTile();

/*!
    \brief Returns a unique ID for the tile.
    \return A TileId object that encodes zoom level, position and map theme.
*/     
    TileId const & id() const;

/*!
    \brief Returns the QImage that describes the look of the TextureTile
    \return A non-zero pointer to a QImage associated with the tile.
*/
    QImage const * image() const;
    
/*!
    \brief Returns the kind of blending used for the tile.
    \return A pointer to the blending object used for painting/merging the TextureTile.
    
    If no blending is set the pointer returned will be zero.
*/     
    Blending const * blending() const;
    int byteCount() const;

 private:
    Q_DISABLE_COPY( TextureTile )

    TileId const m_id;
    Blending const * const m_blending;
    QImage const m_image;
};


// inline definitions

inline TileId const & TextureTile::id() const
{
    return m_id;
}

inline QImage const * TextureTile::image() const
{
    return &m_image;
}

inline Blending const * TextureTile::blending() const
{
    return m_blending;
}

inline int TextureTile::byteCount() const
{
    return m_image.byteCount();
}

}

#endif
